package com.tencent.wecast.sender.cloud.service

import android.app.job.JobParameters
import android.app.job.JobService
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.os.Handler
import android.os.HandlerThread
import android.text.TextUtils
import com.tencent.wecast.WeCastEnv
import com.tencent.wecast.utils.Logger
import com.tencent.wecast.utils.NetworkUtil
import com.tencent.xcast.Log
import java.net.DatagramPacket
import java.net.DatagramSocket
import java.net.InetAddress
import java.net.MulticastSocket
import java.util.*

class SSDPDiscoveryJobService : JobService() {
    companion object {
        private const val TAG = "SSDPDiscoveryJobService"

        const val DLNA_RESPONSE_KEY = "dlna_discovery_rsp"
        const val GOOGLECAST_RESPONSE_KEY = "googlecast_discovery_rsp"

        private const val KEY_ST = "ST"
        private const val GCAST_SEARCH_TARGET_VALUE = "urn:dial-multiscreen-org:service:dial:1"
    }

    private var mDLNARsp: String? = null
    private var mGCastRsp: String? = null

    private var mIsStop: Boolean = false

    private var mMultiCastSocket: MulticastSocket? = null
    private var mDiscoveryRspSocket: DatagramSocket? = null

    private var mSSDPHandler: Handler? = null
    private var mSSDPServiceThread: HandlerThread? = null


    override fun onStartJob(params: JobParameters): Boolean {
        mDLNARsp = params.extras.getString(DLNA_RESPONSE_KEY)
        mGCastRsp = params.extras.getString(GOOGLECAST_RESPONSE_KEY)
        Logger.kp().d("dlnaRsp is $mDLNARsp\n" +
                "googleCastRsp is $mGCastRsp")
        if (!TextUtils.isEmpty(mGCastRsp) && !TextUtils.isEmpty(mDLNARsp)) {
            if (WeCastEnv.getCurrentEnv() == WeCastEnv.ENV_DEVELOP && null != mDLNARsp) {
                val cm = getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
                val startIndex = mDLNARsp!!.indexOf("LOCATION:") + "LOCATION:".length
                val dlnaUrl = mDLNARsp?.substring(startIndex, mDLNARsp!!.indexOf("\r\n", startIndex))
                val mClipData = ClipData.newPlainText("Label", dlnaUrl)
                cm.primaryClip = mClipData
            }
            mSSDPServiceThread = HandlerThread("ssdp_service")
            mSSDPServiceThread?.start()
            mSSDPHandler = Handler(mSSDPServiceThread?.looper)
            mSSDPHandler?.post(mDiscoveryRunnable)
        }
        return true
    }

    override fun onStopJob(params: JobParameters): Boolean {
        mIsStop = true
        return false
    }


    override fun onDestroy() {
        super.onDestroy()
        Logger.kp().d("stop ssdp discovery service.")
        mIsStop = true
        mMultiCastSocket?.close()
        mDiscoveryRspSocket?.close()
        mMultiCastSocket = null
        mDiscoveryRspSocket = null
        mSSDPHandler?.removeCallbacks(mDiscoveryRunnable)
    }

    private var mDiscoveryRunnable: Runnable = Runnable {
        Logger.kp().d("start ssdp discovery service.")
        mIsStop = false
        val mLocalV4Ip = NetworkUtil.getLocalV4Address()
        val mLocalV6Ip = NetworkUtil.getLocalV6Address()
        mMultiCastSocket = MulticastSocket(1900)
        mMultiCastSocket?.loopbackMode = false
        mMultiCastSocket?.joinGroup(InetAddress.getByName("239.255.255.250"))

        val buf = ByteArray(1024)
        val packet = DatagramPacket(buf, buf.size)    // 创建接收报文，以接收通过广播传递过来的
        mDiscoveryRspSocket = DatagramSocket()
        while (!mIsStop) {
            try {
                Log.i(TAG, "sender proxy ssdp server waiting request.")
                mMultiCastSocket?.receive(packet) // 接收报文，程序停滞等待第三方app发送报文
                // 接受第三方播放器的广播
                val receivePacket = String(packet.data, 0, packet.length)
                // 收到 MAN: "ssdp:discover_cast"  数据表示的是接收端修改了SSDP协议字段的广播
                //Logger.t(TAG).d("receive packet is $receivePacket" + "\n localIp:$mLocalV4Ip/$mLocalV6Ip >  sourceIP:${packet.address.hostAddress}")
                if (!receivePacket.contains("discover_cast") &&
                        (mLocalV4Ip == packet.address.hostAddress || mLocalV6Ip == packet.address.hostAddress)) {
                    // Logger.t(TAG).d("receive packet is $receivePacket" + "\n localIp:$mLocalV4Ip >  sourceIP:${packet.address.hostName}")
                    // 根据回包对DLNA和googleCast做拆分处理 将接收端的信息传给第三方播放器
                    val discoveryRspData: ByteArray? = if (checkContainGcastST(receivePacket, KEY_ST)) {
                        //Logger.t(TAG).d("response GoogleCast discovery request:\n$mGCastRsp")
                        mGCastRsp?.toByteArray(charset("utf-8"))
                    } else {
                        val stValue = getSTValue(receivePacket)
                        var dlnaRspNew = mDLNARsp
                        if ("" != stValue && "upnp:rootdevice" != stValue) {
                            dlnaRspNew = dlnaRspNew?.replace("upnp:rootdevice", stValue)
                        }
                        //Logger.t(TAG).d("response DLNA discovery request:\n$dlnaRspNew")
                        dlnaRspNew?.toByteArray(charset("utf-8"))
                    }
                    if (null != discoveryRspData) {
                        val discoveryRspPkg = DatagramPacket(discoveryRspData, discoveryRspData.size, packet.address, packet.port)
                        mDiscoveryRspSocket?.send(discoveryRspPkg)
                    }
                }
            } catch (e: Exception) {
                mMultiCastSocket?.close()
                mDiscoveryRspSocket?.close()
                Logger.t(TAG).e("Exception:" + e.toString())
                e.printStackTrace()
            }
        }
    }

    // 不直接check contains 提高匹配效率
    private fun checkContainGcastST(content: String, headerName: String): Boolean {
        val s = Scanner(content)
        s.nextLine()
        while (s.hasNextLine()) {
            val line = s.nextLine()
            val index = line.indexOf(':')
            if (index == -1) {
                continue
            }
            val header = line.substring(0, index)
            if (headerName.equals(header.trim { it <= ' ' }, ignoreCase = true)) {
                val headerValue = line.substring(index + 1).trim { it <= ' ' }
                return headerValue.contains(GCAST_SEARCH_TARGET_VALUE)
            }
        }
        return false
    }

    private fun getSTValue(content: String): String {
        val s = Scanner(content)
        s.nextLine()
        while (s.hasNextLine()) {
            val line = s.nextLine()
            if (line.startsWith("ST:")) {
                val stIndex = line.indexOf("ST:")
                if (stIndex != -1) {
                    return line.substring(stIndex + 3, line.length).trim()
                }
            }
        }
        return ""
    }

}
