package com.tencent.wecast.utils;

import android.annotation.SuppressLint;
import android.bluetooth.BluetoothAdapter;
import android.content.Context;
import android.content.SharedPreferences;
import android.media.AudioManager;
import android.opengl.EGL14;
import android.opengl.GLES20;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.provider.Settings;
import android.text.TextUtils;
import android.util.ArrayMap;
import android.util.DisplayMetrics;
import android.view.WindowManager;

import com.tencent.wecast.jni.JniHelper;
import com.tencent.wecast.jni.ParamsPackage;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.net.NetworkInterface;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;

import javax.microedition.khronos.egl.EGL10;

public class DeviceInfoUtil {
    public static final String TAG = "DeviceInfoUtil";

    // private static int INSTANCE_ID = WmpDefine.WmpInstance.DEFAULT;
    private static final String OS = "android";

    private static String mCpuName;
    private static String mDevicesName;

    public static String getOsVer() {
        return "Android " +
                getVersion() +
                "(" +
                getApiLevel()
                + ")";
    }

    private static String getVersion() {
        String version = "";
        try {
            version = Build.VERSION.RELEASE;
        } catch (Throwable ignore) {
        }
        return version;
    }

    private static String getApiLevel() {
        String apiLevel = "";
        try {
            apiLevel = Build.VERSION.SDK;
        } catch (Throwable ignore) {
        }
        return apiLevel;
    }

    public static String getCpuName() {
        if (!TextUtils.isEmpty(mCpuName)) {
            return mCpuName;
        }
        FileReader fr = null;
        BufferedReader br = null;
        try {
            fr = new FileReader("/proc/cpuinfo");
            br = new BufferedReader(fr);
            String text = br.readLine();
            if (!TextUtils.isEmpty(text)) {
                String[] array = text.split(":\\s+", 2);
                mCpuName = array[1];
                return mCpuName;
            }
        } catch (Exception e) {
            Logger.t(TAG).e("DeviceInfoUtil e = " + e);
        } finally {
            try {
                if (fr != null) {
                    fr.close();
                }
                if (br != null) {
                    br.close();
                }
            } catch (IOException e) {
                Logger.t(TAG).e("IOException: " + e.toString());
            }
        }
        return null;
    }


    public interface GetGpuNameCallback {
        void onResult(String name);
    }


    private static String mGpuName;

    public static void getGpuName(final GetGpuNameCallback getGpuNameCallback) {
        if (!TextUtils.isEmpty(mGpuName)) {
            if (getGpuNameCallback != null) {
                getGpuNameCallback.onResult(mGpuName);
            }
            return;
        }
        new Thread(new GetGpuNameTask(getGpuNameCallback)).start();
    }


    /**
     * 获取 GPU 信息需要 OpenGL 上下文环境，
     */
    static class GetGpuNameTask implements Runnable {

        private EGL10 mEGL10 = null;
        private javax.microedition.khronos.egl.EGLDisplay mEGL10Display = null;
        private javax.microedition.khronos.egl.EGLSurface mEGL10Surface = null;
        private javax.microedition.khronos.egl.EGLContext mEGL10Context = null;

        private android.opengl.EGLDisplay mEGL14Display = null;
        private android.opengl.EGLSurface mEGL14Surface = null;
        private android.opengl.EGLContext mEGL14Context = null;

        private void createEGL10Context() {
            mEGL10 = (EGL10) javax.microedition.khronos.egl.EGLContext.getEGL();
            mEGL10Display = mEGL10.eglGetDisplay(EGL10.EGL_DEFAULT_DISPLAY);
            int[] vers = new int[2];
            mEGL10.eglInitialize(mEGL10Display, vers);

            int[] configAttr = {EGL10.EGL_COLOR_BUFFER_TYPE, EGL10.EGL_RGB_BUFFER, EGL10.EGL_LEVEL, 0, EGL10.EGL_SURFACE_TYPE, EGL10.EGL_PBUFFER_BIT, EGL10.EGL_NONE};
            javax.microedition.khronos.egl.EGLConfig[] configs = new javax.microedition.khronos.egl.EGLConfig[1];
            int[] numConfig = new int[1];
            mEGL10.eglChooseConfig(mEGL10Display, configAttr, configs, 1, numConfig);
            if (numConfig[0] == 0) {
                Logger.t(TAG).e("TROUBLE! No config found.");
            }
            javax.microedition.khronos.egl.EGLConfig config = configs[0];

            int[] surfAttr = {EGL10.EGL_WIDTH, 64, EGL10.EGL_HEIGHT, 64, EGL10.EGL_NONE};
            mEGL10Surface = mEGL10.eglCreatePbufferSurface(mEGL10Display, config, surfAttr);
            final int EGL_CONTEXT_CLIENT_VERSION = 0x3098; // missing in EGL10
            int[] ctxAttrib = {EGL_CONTEXT_CLIENT_VERSION, 2, EGL10.EGL_NONE};
            mEGL10Context = mEGL10.eglCreateContext(mEGL10Display, config, EGL10.EGL_NO_CONTEXT, ctxAttrib);
            mEGL10.eglMakeCurrent(mEGL10Display, mEGL10Surface, mEGL10Surface, mEGL10Context);
        }

        private void destroyEGL10Context() {
            if (mEGL10 != null && mEGL10Display != null && mEGL10Surface != null && mEGL10Context != null) {
                mEGL10.eglMakeCurrent(mEGL10Display, EGL10.EGL_NO_SURFACE, EGL10.EGL_NO_SURFACE, EGL10.EGL_NO_CONTEXT);
                mEGL10.eglDestroySurface(mEGL10Display, mEGL10Surface);
                mEGL10.eglDestroyContext(mEGL10Display, mEGL10Context);
                mEGL10.eglTerminate(mEGL10Display);
            }

        }

        @SuppressLint("NewApi")
        private void createEGL14Context() {
            mEGL14Display = EGL14.eglGetDisplay(EGL14.EGL_DEFAULT_DISPLAY);
            int[] vers = new int[2];
            EGL14.eglInitialize(mEGL14Display, vers, 0, vers, 1);

            int[] configAttr = {
                    EGL14.EGL_COLOR_BUFFER_TYPE, EGL14.EGL_RGB_BUFFER,
                    EGL14.EGL_LEVEL, 0,
                    EGL14.EGL_RENDERABLE_TYPE, EGL14.EGL_OPENGL_ES2_BIT,
                    EGL14.EGL_SURFACE_TYPE, EGL14.EGL_PBUFFER_BIT,
                    EGL14.EGL_NONE
            };
            android.opengl.EGLConfig[] configs = new android.opengl.EGLConfig[1];
            int[] numConfig = new int[1];
            EGL14.eglChooseConfig(mEGL14Display, configAttr, 0,
                    configs, 0, 1, numConfig, 0);
            if (numConfig[0] == 0) {
                Logger.t(TAG).e("TROUBLE! No config found.");
            }
            android.opengl.EGLConfig config = configs[0];

            int[] surfAttr = {
                    EGL14.EGL_WIDTH, 64,
                    EGL14.EGL_HEIGHT, 64,
                    EGL14.EGL_NONE
            };
            mEGL14Surface = EGL14.eglCreatePbufferSurface(mEGL14Display, config, surfAttr, 0);

            int[] ctxAttrib = {
                    EGL14.EGL_CONTEXT_CLIENT_VERSION, 2,
                    EGL14.EGL_NONE
            };
            mEGL14Context = EGL14.eglCreateContext(mEGL14Display, config, EGL14.EGL_NO_CONTEXT, ctxAttrib, 0);
            EGL14.eglMakeCurrent(mEGL14Display, mEGL14Surface, mEGL14Surface, mEGL14Context);
        }

        @SuppressLint("NewApi")
        private void destroyEGL14Context() {
            if (mEGL14Display != null && mEGL14Surface != null && mEGL14Context != null) {
                EGL14.eglMakeCurrent(mEGL14Display, EGL14.EGL_NO_SURFACE, EGL14.EGL_NO_SURFACE,
                        EGL14.EGL_NO_CONTEXT);
                EGL14.eglDestroySurface(mEGL14Display, mEGL14Surface);
                EGL14.eglDestroyContext(mEGL14Display, mEGL14Context);
                EGL14.eglTerminate(mEGL14Display);
            }

        }

        private GetGpuNameCallback mGetGpuNameCallback;

        public GetGpuNameTask(GetGpuNameCallback getGpuNameCallback) {
            mGetGpuNameCallback = getGpuNameCallback;
        }

        @Override
        public void run() {
            try {
                if (Build.VERSION.SDK_INT >= 17) {
                    createEGL14Context();
                } else {
                    createEGL10Context();
                }

                mGpuName = GLES20.glGetString(GLES20.GL_VENDOR) + " " + GLES20.glGetString(GLES20.GL_RENDERER);
                if (mGetGpuNameCallback != null) {
                    mGetGpuNameCallback.onResult(mGpuName);
                }

                if (Build.VERSION.SDK_INT >= 17) {
                    destroyEGL14Context();
                } else {
                    destroyEGL10Context();
                }
            } catch (Throwable e) {
                if (mGetGpuNameCallback != null) {
                    mGetGpuNameCallback.onResult("");
                }
            }
        }
    }

    public static void setDeviceName(String deviceName) {
        mDevicesName = deviceName;
    }

    private static String mDeviceName = "";

    public static String getDeviceName() {
        if (!TextUtils.isEmpty(mDevicesName)) {
            return mDevicesName;
        }

        try {
            BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();
            mDeviceName = adapter.getName();
        } catch (Throwable t) {
            Logger.t(TAG).e("getDeviceName e = " + t);
        }
        if (TextUtils.isEmpty(mDeviceName)) {
            mDeviceName = Build.MANUFACTURER + " " + Build.MODEL;
        }

        return mDeviceName;
    }

    public static String getModel() {
        return Build.MANUFACTURER + " " + Build.MODEL;
    }

    public static String getOS() {
        return OS;
    }

    /**
     * 获取无线 MAC 地址
     * <p>需添加权限 {@code <uses-permission android:name="android.permission.INTERNET" />}</p>
     */
    public static String getWlanMacAddress() {
        return getMacAddressByNetworkInterface("wlan0");
    }


    /**
     * 获取有线 MAC 地址
     * <p>需添加权限 {@code <uses-permission android:name="android.permission.INTERNET" />}</p>
     */
    public static String getWiredMacAddress() {
        return getMacAddressByNetworkInterface("eth0");
    }


    /**
     * 获取设备 MAC 地址
     * <p>需添加权限 {@code <uses-permission android:name="android.permission.INTERNET" />}</p>
     *
     * @return MAC 地址
     */
    private static String getMacAddressByNetworkInterface(String name) {
        try {
            Enumeration<NetworkInterface> networkInterfaceEnumeration = NetworkInterface.getNetworkInterfaces();
            if (networkInterfaceEnumeration != null) {
                List<NetworkInterface> nis = Collections.list(networkInterfaceEnumeration);
                for (NetworkInterface ni : nis) {
                    if (!ni.getName().equalsIgnoreCase(name)) continue;
                    byte[] macBytes = ni.getHardwareAddress();
                    if (macBytes != null && macBytes.length > 0) {
                        StringBuilder res1 = new StringBuilder();
                        for (byte b : macBytes) {
                            res1.append(String.format("%02x:", b));
                        }
                        return res1.deleteCharAt(res1.length() - 1).toString();
                    }
                }
            } else {
                Logger.t(TAG).e("networkInterfaceEnumeration is null");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

    public static String getRomSize() {
        try {
            File path = Environment.getDataDirectory();
            StatFs stat = new StatFs(path.getPath());
            long blockSize = stat.getBlockSize();
            long totalBlocks = stat.getBlockCount();
            float size = totalBlocks * blockSize / 1073741824.0f;
            return String.format(Locale.getDefault(), "%.1fG", size);
        } catch (Throwable ignore) {
        }
        return null;
    }

    public static String getRamSize() {
        String str1 = "/proc/meminfo";
        FileReader fr = null;
        BufferedReader localBufferedReader = null;
        try {
            fr = new FileReader(str1);
            localBufferedReader = new BufferedReader(fr, 8192);
            String text = localBufferedReader.readLine();
            if (!TextUtils.isEmpty(text)) {
                String[] array = text.split(":\\s+", 2);
                String tempRamSize = array[1].toLowerCase();
                String newRamSize = tempRamSize.replace("kb", "").trim();
                float i = Float.parseFloat(newRamSize) / 1048576;
                return String.format(Locale.getDefault(), "%.1fG", i);
            }
        } catch (Throwable ignore) {
        } finally {
            try {
                if (localBufferedReader != null) {
                    localBufferedReader.close();
                }
                if (fr != null) {
                    fr.close();
                }
            } catch (Throwable ignore) {
            }
        }
        return null;
    }

    public static String getResolution(Context context) {
        DisplayMetrics dm = getDisplayMetrics(context);
        return dm == null ? "" : (dm.widthPixels + "*" + dm.heightPixels);
    }

    public static DisplayMetrics getDisplayMetrics(Context context) {
        try {
            DisplayMetrics metric = new DisplayMetrics();
            WindowManager manager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
            manager.getDefaultDisplay().getMetrics(metric);
            return metric;
        } catch (Throwable ignore) {
        }
        return null;
    }

    @SuppressLint("HardwareIds")
    public static String getAndroidId(Context context) {
        return Settings.Secure.getString(
                context.getContentResolver(),
                Settings.Secure.ANDROID_ID
        );
    }

    private static String uniqueID = null;
    private static final String PREF_UNIQUE_ID = "device_uuid";

    public static String getUUID(Context context) {
        if (uniqueID == null) {
            SharedPreferences sharedPrefs = context.getSharedPreferences(
                    PREF_UNIQUE_ID, Context.MODE_PRIVATE);
            uniqueID = sharedPrefs.getString(PREF_UNIQUE_ID, null);
            if (uniqueID == null) {
                uniqueID = UUID.randomUUID().toString();
                SharedPreferences.Editor editor = sharedPrefs.edit();
                editor.putString(PREF_UNIQUE_ID, uniqueID);
                editor.apply();
            }
        }
        return uniqueID;
    }

    public static String getSystemInfo() {
//        StringBuilder sb = new StringBuilder();
//        int pid = Process.myPid();
//        int procMem = getProcessMemory(pid);
//        MemoryInfo sysMemory = getMemoryInfo();
//        CpuRate cpuRate = getCpuRate(pid);
//        sb.append("[CPU Usage:").append(cpuRate.procRate)
//                .append("%, TotalMem:").append(sysMemory.getTotalMemory() / 1024)
//                .append("MB, FreeMem:").append(sysMemory.getFreeMemory() / 1024)
//                .append("MB, ProcUsed:").append(procMem / 1024).append("]");
        // java层无法获取，native层暂时可以获取信息
        String sysInfo = "";
        ParamsPackage.ParamsFromCpp ret = JniHelper.callJniFunc("getSysInfo");
        if (ret != null) {
            try {
                sysInfo = (String) (ret.getNext());
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return sysInfo;
    }

    // 获取进程当前占用物理内存的大小(kB)
    public static int getProcessMemory(int pid) {
        RandomAccessFile reader = null;
        try {
            reader = new RandomAccessFile("/proc/" + pid + "/status", "r");
            String line = reader.readLine();
            while (!TextUtils.isEmpty(line) && line.contains("VmRSS:")) {
                String[] VmRSS = line.split(" ");
                if (VmRSS.length == 3) {
                    return Integer.valueOf(VmRSS[1]);
                }
                line = reader.readLine();
            }
        } catch (Throwable throwable) {
            throwable.printStackTrace();
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
            } catch (IOException exception) {
                exception.printStackTrace();
            }
        }
        return 0;
    }

    // 获取系统内存信息
    private static MemoryInfo getMemoryInfo() {
        MemoryInfo memoryInfo = new MemoryInfo();
        RandomAccessFile reader = null;
        String infoKeys[] = {"MemTotal:", "MemFree:", "Buffers:", "Cached:"};
        try {
            reader = new RandomAccessFile("/proc/meminfo", "r");
            String line = reader.readLine();
            while (!TextUtils.isEmpty(line)) {
                for (String key : infoKeys) {
                    if (line.contains(key)) {
                        memoryInfo.setMemory(key, line.split(" ")[1]);
                        break;
                    }
                }
                if (memoryInfo.isMemInfoSet()) {
                    break;
                }
                line = reader.readLine();
            }
        } catch (Throwable throwable) {
            throwable.printStackTrace();
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
            } catch (IOException exception) {
                exception.printStackTrace();
            }
        }
        return memoryInfo;
    }

    public static class MemoryInfo {
        private int mMemTotal = 0;
        private int mMemFree = 0;
        private int mBuffers = 0;
        private int mCached = 0;
        private int mSetCnt = 0;

        void setMemory(String key, String value) {
            if ("MemTotal:".equals(key)) {
                mMemTotal = Integer.valueOf(value);
                mSetCnt++;
            } else if ("MemFree:".equals(key)) {
                mMemFree = Integer.valueOf(value);
                mSetCnt++;
            } else if ("Buffers:".equals(key)) {
                mBuffers = Integer.valueOf(value);
                mSetCnt++;
            } else if ("Cached:".equals(key)) {
                mCached = Integer.valueOf(value);
                mSetCnt++;
            }
        }

        boolean isMemInfoSet() {
            return mSetCnt == 4;
        }

        public int getTotalMemory() {
            return mMemTotal;
        }

        public int getFreeMemory() {
            return mMemFree + mBuffers + mCached;
        }
    }

    public static class CpuInfo {
        long mUtime;
        long mNtime;
        long mStime;
        long mItime;
        long mIowaitTime;
        long mIrqTime;
        long mSoftirqTime;
        long mStealStolenTime;
        long mGuestTime;
    }

    public static class ProcessInfo {
        long mUtime;
        long mStime;
        long mCutime;
        long mCstime;
    }

    private static class CpuProcessComp {
        CpuInfo mCpuInfo;
        ProcessInfo mProcessInfo;

        CpuProcessComp(CpuInfo cpu, ProcessInfo process) {
            mCpuInfo = cpu;
            mProcessInfo = process;
        }
    }

    private static Map<Integer, CpuProcessComp> mCpuProcessMap = new ArrayMap<>();

    public static ProcessInfo getProcessInfo(int pid) {
        RandomAccessFile reader = null;
        ProcessInfo info = new ProcessInfo();
        try {
            reader = new RandomAccessFile("/proc/" + pid + "/stat", "r");
            String line = reader.readLine();
            if (!TextUtils.isEmpty(line)) {
                String[] stat = line.split(" ");
                if (stat.length > 17) {
                    info.mUtime = Long.valueOf(stat[13]);
                    info.mStime = Long.valueOf(stat[14]);
                    info.mCutime = Long.valueOf(stat[15]);
                    info.mCstime = Long.valueOf(stat[16]);
                }
            }
        } catch (Throwable throwable) {
            throwable.printStackTrace();
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
            } catch (IOException exception) {
                exception.printStackTrace();
            }
        }
        return info;
    }

    public static CpuInfo getCpuInfo() {
        RandomAccessFile reader = null;
        CpuInfo info = new CpuInfo();
        try {
            reader = new RandomAccessFile("/proc/stat", "r");
            String line = reader.readLine();
            if (!TextUtils.isEmpty(line)) {
                String[] stat = line.split(" ");
                if (stat.length >= 10) {
                    info.mUtime = Long.valueOf(stat[1]);
                    info.mNtime = Long.valueOf(stat[2]);
                    info.mStime = Long.valueOf(stat[3]);
                    info.mItime = Long.valueOf(stat[4]);
                    info.mIowaitTime = Long.valueOf(stat[5]);
                    info.mIrqTime = Long.valueOf(stat[6]);
                    info.mSoftirqTime = Long.valueOf(stat[7]);
                    info.mStealStolenTime = Long.valueOf(stat[8]);
                    info.mGuestTime = Long.valueOf(stat[9]);
                }
            }
        } catch (Throwable throwable) {
            throwable.printStackTrace();
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
            } catch (IOException exception) {
                exception.printStackTrace();
            }
        }
        return info;
    }

    public static class CpuRate {
        int procRate;   // 进程的cpu使用率
        int totalRate;  // 系统cpu使用率
    }

    public static CpuRate getCpuRate(int pid) {
        CpuRate rate = new CpuRate();
        CpuInfo newCpuInfo = getCpuInfo();
        ProcessInfo newProcessInfo = getProcessInfo(pid);
        if (mCpuProcessMap.containsKey(pid)) {
            long newTotalTime = newCpuInfo.mUtime
                    + newCpuInfo.mStime
                    + newCpuInfo.mItime
                    + newCpuInfo.mNtime
                    + newCpuInfo.mIowaitTime
                    + newCpuInfo.mIrqTime
                    + newCpuInfo.mSoftirqTime
                    + newCpuInfo.mStealStolenTime
                    + newCpuInfo.mGuestTime;

            CpuProcessComp lastComp = mCpuProcessMap.get(pid);
            long lastTotalTime = lastComp.mCpuInfo.mUtime
                    + lastComp.mCpuInfo.mStime
                    + lastComp.mCpuInfo.mItime
                    + lastComp.mCpuInfo.mNtime
                    + lastComp.mCpuInfo.mIowaitTime
                    + lastComp.mCpuInfo.mIrqTime
                    + lastComp.mCpuInfo.mSoftirqTime
                    + lastComp.mCpuInfo.mStealStolenTime
                    + lastComp.mCpuInfo.mGuestTime;

            long totalTime = newTotalTime - lastTotalTime;
            long idleTime = newCpuInfo.mItime - lastComp.mCpuInfo.mItime;

            long procTime = (newProcessInfo.mUtime + newProcessInfo.mStime
                    + newProcessInfo.mCutime + newProcessInfo.mCstime)
                    - (lastComp.mProcessInfo.mUtime
                    + lastComp.mProcessInfo.mStime
                    + lastComp.mProcessInfo.mCutime
                    + lastComp.mProcessInfo.mCstime);
            if (totalTime > 0) {
                rate.procRate = (int) (procTime * 10000 / totalTime);
                rate.totalRate = (int) ((totalTime - idleTime) * 10000 / totalTime);
            }
        }

        mCpuProcessMap.put(pid, new CpuProcessComp(newCpuInfo, newProcessInfo));
        return rate;
    }

    public static int setMaxVolume(Context appContext) {
        if (appContext == null) {
            Logger.t(TAG).e("appContext is null");
            return -1;
        }
        AudioManager audioManager = (AudioManager) appContext.getApplicationContext().getSystemService(Context.AUDIO_SERVICE);
        int maxVolume = 0;
        if (audioManager != null) {
            maxVolume = audioManager.getStreamMaxVolume(AudioManager.STREAM_MUSIC);
            audioManager.setStreamVolume(AudioManager.STREAM_MUSIC, maxVolume, 0);
            Logger.t(TAG).d("has set the volume to " + maxVolume);
            return 0;
        }
        return -1;
    }
}