/*
 * Decompiled with CFR 0.152.
 */
package com.taobao.arthas.mcp.server.protocol.server.handler;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.taobao.arthas.mcp.server.protocol.config.McpServerProperties;
import com.taobao.arthas.mcp.server.protocol.server.McpTransportContextExtractor;
import com.taobao.arthas.mcp.server.protocol.server.handler.McpStatelessHttpRequestHandler;
import com.taobao.arthas.mcp.server.protocol.server.handler.McpStreamableHttpRequestHandler;
import com.taobao.arthas.mcp.server.protocol.spec.McpError;
import com.taobao.arthas.mcp.server.util.Assert;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http.DefaultFullHttpResponse;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.util.CharsetUtil;
import io.netty.util.concurrent.GenericFutureListener;
import java.nio.charset.Charset;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class McpHttpRequestHandler {
    private static final Logger logger = LoggerFactory.getLogger(McpHttpRequestHandler.class);
    public static final String APPLICATION_JSON = "application/json";
    public static final String TEXT_EVENT_STREAM = "text/event-stream";
    private static final String ACCEPT_HEADER = "Accept";
    private final String mcpEndpoint;
    private final ObjectMapper objectMapper;
    private final McpTransportContextExtractor<FullHttpRequest> contextExtractor;
    private final AtomicBoolean isClosing = new AtomicBoolean(false);
    private McpStatelessHttpRequestHandler statelessHandler;
    private McpStreamableHttpRequestHandler streamableHandler;
    private McpServerProperties.ServerProtocol protocol;

    public McpHttpRequestHandler(String mcpEndpoint, ObjectMapper objectMapper, McpTransportContextExtractor<FullHttpRequest> contextExtractor) {
        Assert.notNull(mcpEndpoint, "mcpEndpoint must not be null");
        Assert.notNull(objectMapper, "objectMapper must not be null");
        Assert.notNull(contextExtractor, "contextExtractor must not be null");
        this.mcpEndpoint = mcpEndpoint;
        this.objectMapper = objectMapper;
        this.contextExtractor = contextExtractor;
    }

    public void setProtocol(McpServerProperties.ServerProtocol protocol) {
        this.protocol = protocol;
    }

    public void setStatelessHandler(McpStatelessHttpRequestHandler statelessHandler) {
        this.statelessHandler = statelessHandler;
    }

    public void setStreamableHandler(McpStreamableHttpRequestHandler streamableHandler) {
        this.streamableHandler = streamableHandler;
    }

    public void handle(ChannelHandlerContext ctx, FullHttpRequest request) throws Exception {
        String uri = request.uri();
        if (!uri.endsWith(this.mcpEndpoint)) {
            this.sendError(ctx, HttpResponseStatus.NOT_FOUND, new McpError((Object)"Endpoint not found"));
            return;
        }
        if (this.isClosing.get()) {
            this.sendError(ctx, HttpResponseStatus.SERVICE_UNAVAILABLE, new McpError((Object)"Server is shutting down"));
            return;
        }
        logger.debug("Request {} {} -> using {} transport", new Object[]{request.method(), request.uri(), this.protocol});
        try {
            if (this.protocol == McpServerProperties.ServerProtocol.STREAMABLE) {
                if (this.streamableHandler == null) {
                    this.sendError(ctx, HttpResponseStatus.SERVICE_UNAVAILABLE, new McpError((Object)"Streamable transport handler not available"));
                    return;
                }
                this.streamableHandler.handle(ctx, request);
            } else {
                if (this.statelessHandler == null) {
                    this.sendError(ctx, HttpResponseStatus.SERVICE_UNAVAILABLE, new McpError((Object)"Stateless transport handler not available"));
                    return;
                }
                this.statelessHandler.handle(ctx, request);
            }
        }
        catch (Exception e) {
            logger.error("Error handling request: {}", (Object)e.getMessage(), (Object)e);
            this.sendError(ctx, HttpResponseStatus.INTERNAL_SERVER_ERROR, new McpError((Object)("Error processing request: " + e.getMessage())));
        }
    }

    public CompletableFuture<Void> closeGracefully() {
        return CompletableFuture.runAsync(() -> {
            this.isClosing.set(true);
            logger.debug("Initiating graceful shutdown of MCP handler");
            CompletableFuture<Object> statelessClose = CompletableFuture.completedFuture(null);
            CompletableFuture<Object> streamableClose = CompletableFuture.completedFuture(null);
            if (this.statelessHandler != null) {
                statelessClose = this.statelessHandler.closeGracefully();
            }
            if (this.streamableHandler != null) {
                streamableClose = this.streamableHandler.closeGracefully();
            }
            CompletableFuture.allOf(statelessClose, streamableClose).join();
            logger.debug("Graceful shutdown completed");
        });
    }

    private void sendError(ChannelHandlerContext ctx, HttpResponseStatus status, McpError mcpError) {
        try {
            String jsonError = this.objectMapper.writeValueAsString((Object)mcpError);
            ByteBuf content = Unpooled.copiedBuffer((CharSequence)jsonError, (Charset)CharsetUtil.UTF_8);
            DefaultFullHttpResponse response = new DefaultFullHttpResponse(HttpVersion.HTTP_1_1, status, content);
            response.headers().set((CharSequence)HttpHeaderNames.CONTENT_TYPE, (Object)APPLICATION_JSON);
            response.headers().set((CharSequence)HttpHeaderNames.CONTENT_LENGTH, (Object)response.content().readableBytes());
            response.headers().set((CharSequence)HttpHeaderNames.ACCESS_CONTROL_ALLOW_ORIGIN, (Object)"*");
            ctx.writeAndFlush((Object)response).addListener((GenericFutureListener)ChannelFutureListener.CLOSE);
        }
        catch (Exception e) {
            logger.error("Failed to send error response: {}", (Object)e.getMessage());
            DefaultFullHttpResponse response = new DefaultFullHttpResponse(HttpVersion.HTTP_1_1, HttpResponseStatus.INTERNAL_SERVER_ERROR);
            response.headers().set((CharSequence)HttpHeaderNames.CONTENT_LENGTH, (Object)0);
            ctx.writeAndFlush((Object)response).addListener((GenericFutureListener)ChannelFutureListener.CLOSE);
        }
    }

    public String getMcpEndpoint() {
        return this.mcpEndpoint;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String mcpEndpoint = "/mcp";
        private ObjectMapper objectMapper;
        private McpTransportContextExtractor<FullHttpRequest> contextExtractor = (request, context) -> context;
        private McpServerProperties.ServerProtocol protocol;

        public Builder mcpEndpoint(String mcpEndpoint) {
            Assert.notNull(mcpEndpoint, "MCP endpoint must not be null");
            this.mcpEndpoint = mcpEndpoint;
            return this;
        }

        public Builder objectMapper(ObjectMapper objectMapper) {
            Assert.notNull(objectMapper, "ObjectMapper must not be null");
            this.objectMapper = objectMapper;
            return this;
        }

        public Builder contextExtractor(McpTransportContextExtractor<FullHttpRequest> contextExtractor) {
            Assert.notNull(contextExtractor, "Context extractor must not be null");
            this.contextExtractor = contextExtractor;
            return this;
        }

        public Builder protocol(McpServerProperties.ServerProtocol protocol) {
            this.protocol = protocol;
            return this;
        }

        public McpHttpRequestHandler build() {
            Assert.notNull(this.objectMapper, "ObjectMapper must be set");
            Assert.notNull(this.mcpEndpoint, "MCP endpoint must be set");
            if (this.protocol == null) {
                this.protocol = McpServerProperties.ServerProtocol.STREAMABLE;
            }
            McpHttpRequestHandler handler = new McpHttpRequestHandler(this.mcpEndpoint, this.objectMapper, this.contextExtractor);
            handler.setProtocol(this.protocol);
            return handler;
        }
    }
}

