/*
 * Decompiled with CFR 0.152.
 */
package com.taobao.arthas.mcp.server.protocol.server;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.taobao.arthas.mcp.server.CommandExecutor;
import com.taobao.arthas.mcp.server.protocol.server.McpNettyServerExchange;
import com.taobao.arthas.mcp.server.protocol.server.McpNotificationHandler;
import com.taobao.arthas.mcp.server.protocol.server.McpRequestHandler;
import com.taobao.arthas.mcp.server.protocol.server.McpServerFeatures;
import com.taobao.arthas.mcp.server.protocol.spec.DefaultMcpStreamableServerSessionFactory;
import com.taobao.arthas.mcp.server.protocol.spec.McpError;
import com.taobao.arthas.mcp.server.protocol.spec.McpSchema;
import com.taobao.arthas.mcp.server.protocol.spec.McpServerTransportProvider;
import com.taobao.arthas.mcp.server.protocol.spec.McpStreamableServerTransportProvider;
import com.taobao.arthas.mcp.server.util.Assert;
import com.taobao.arthas.mcp.server.util.Utils;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.BiFunction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class McpNettyServer {
    private static final Logger logger = LoggerFactory.getLogger(McpNettyServer.class);
    private final McpServerTransportProvider mcpTransportProvider;
    private final ObjectMapper objectMapper;
    private final McpSchema.ServerCapabilities serverCapabilities;
    private final McpSchema.Implementation serverInfo;
    private final String instructions;
    private final CopyOnWriteArrayList<McpServerFeatures.ToolSpecification> tools = new CopyOnWriteArrayList();
    private final CopyOnWriteArrayList<McpSchema.ResourceTemplate> resourceTemplates = new CopyOnWriteArrayList();
    private final ConcurrentHashMap<String, McpServerFeatures.ResourceSpecification> resources = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, McpServerFeatures.PromptSpecification> prompts = new ConcurrentHashMap();
    private McpSchema.LoggingLevel minLoggingLevel = McpSchema.LoggingLevel.DEBUG;
    private List<String> protocolVersions;

    McpNettyServer(McpStreamableServerTransportProvider mcpTransportProvider, ObjectMapper objectMapper, Duration requestTimeout, McpServerFeatures.McpServerConfig features, CommandExecutor commandExecutor) {
        this.mcpTransportProvider = mcpTransportProvider;
        this.objectMapper = objectMapper;
        this.serverInfo = features.getServerInfo();
        this.serverCapabilities = features.getServerCapabilities();
        this.instructions = features.getInstructions();
        this.tools.addAll(features.getTools());
        this.resources.putAll(features.getResources());
        this.resourceTemplates.addAll(features.getResourceTemplates());
        this.prompts.putAll(features.getPrompts());
        Map<String, McpRequestHandler<?>> requestHandlers = this.prepareRequestHandlers();
        Map<String, McpNotificationHandler> notificationHandlers = this.prepareNotificationHandlers(features);
        this.protocolVersions = Collections.singletonList(mcpTransportProvider.protocolVersion());
        mcpTransportProvider.setSessionFactory(new DefaultMcpStreamableServerSessionFactory(requestTimeout, this::initializeRequestHandler, requestHandlers, notificationHandlers, commandExecutor));
    }

    private Map<String, McpNotificationHandler> prepareNotificationHandlers(McpServerFeatures.McpServerConfig features) {
        HashMap<String, McpNotificationHandler> notificationHandlers = new HashMap<String, McpNotificationHandler>();
        notificationHandlers.put("notifications/initialized", (exchange, commandContext, params) -> CompletableFuture.completedFuture(null));
        List<BiFunction<McpNettyServerExchange, List<McpSchema.Root>, CompletableFuture<Void>>> rootsChangeConsumers = features.getRootsChangeConsumers();
        if (Utils.isEmpty(rootsChangeConsumers)) {
            rootsChangeConsumers = Collections.singletonList((exchange, roots) -> CompletableFuture.runAsync(() -> logger.warn("Roots list changed notification, but no consumers provided. Roots list changed: {}", roots)));
        }
        notificationHandlers.put("notifications/roots/list_changed", this.rootsListChangedNotificationHandler(rootsChangeConsumers));
        return notificationHandlers;
    }

    private Map<String, McpRequestHandler<?>> prepareRequestHandlers() {
        HashMap requestHandlers = new HashMap();
        requestHandlers.put("ping", (exchange, commandContext, params) -> CompletableFuture.completedFuture(Collections.emptyMap()));
        if (this.serverCapabilities.getTools() != null) {
            requestHandlers.put("tools/list", this.toolsListRequestHandler());
            requestHandlers.put("tools/call", this.toolsCallRequestHandler());
        }
        if (this.serverCapabilities.getResources() != null) {
            requestHandlers.put("resources/list", this.resourcesListRequestHandler());
            requestHandlers.put("resources/read", this.resourcesReadRequestHandler());
            requestHandlers.put("resources/templates/list", this.resourceTemplateListRequestHandler());
        }
        if (this.serverCapabilities.getPrompts() != null) {
            requestHandlers.put("prompts/list", this.promptsListRequestHandler());
            requestHandlers.put("prompts/get", this.promptsGetRequestHandler());
        }
        if (this.serverCapabilities.getLogging() != null) {
            requestHandlers.put("logging/setLevel", this.setLoggerRequestHandler());
        }
        return requestHandlers;
    }

    private CompletableFuture<McpSchema.InitializeResult> initializeRequestHandler(McpSchema.InitializeRequest initializeRequest) {
        return CompletableFuture.supplyAsync(() -> {
            logger.info("Client initialize request - Protocol: {}, Capabilities: {}, Info: {}", new Object[]{initializeRequest.getProtocolVersion(), initializeRequest.getCapabilities(), initializeRequest.getClientInfo()});
            String serverProtocolVersion = this.protocolVersions.get(this.protocolVersions.size() - 1);
            if (this.protocolVersions.contains(initializeRequest.getProtocolVersion())) {
                serverProtocolVersion = initializeRequest.getProtocolVersion();
            } else {
                logger.warn("Client requested unsupported protocol version: {}, so the server will suggest {} instead", (Object)initializeRequest.getProtocolVersion(), (Object)serverProtocolVersion);
            }
            return new McpSchema.InitializeResult(serverProtocolVersion, this.serverCapabilities, this.serverInfo, this.instructions);
        });
    }

    public McpSchema.ServerCapabilities getServerCapabilities() {
        return this.serverCapabilities;
    }

    public McpSchema.Implementation getServerInfo() {
        return this.serverInfo;
    }

    public CompletableFuture<Void> closeGracefully() {
        return this.mcpTransportProvider.closeGracefully();
    }

    public void close() {
        this.mcpTransportProvider.close();
    }

    private McpNotificationHandler rootsListChangedNotificationHandler(List<BiFunction<McpNettyServerExchange, List<McpSchema.Root>, CompletableFuture<Void>>> rootsChangeConsumers) {
        return (exchange, commandContext, params) -> {
            CompletableFuture<McpSchema.ListRootsResult> futureRoots = exchange.listRoots();
            return futureRoots.thenCompose(listRootsResult -> {
                List<McpSchema.Root> roots = listRootsResult.getRoots();
                ArrayList<CompletionStage> futures = new ArrayList<CompletionStage>();
                for (BiFunction consumer : rootsChangeConsumers) {
                    CompletionStage future = ((CompletableFuture)consumer.apply(exchange, roots)).exceptionally(error -> {
                        logger.error("Error handling roots list change notification", error);
                        return null;
                    });
                    futures.add(future);
                }
                return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
            });
        };
    }

    public CompletableFuture<Void> addTool(McpServerFeatures.ToolSpecification toolSpecification) {
        if (toolSpecification == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Tool specification must not be null"));
            return future;
        }
        if (toolSpecification.getTool() == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Tool must not be null"));
            return future;
        }
        if (toolSpecification.getCall() == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Tool call handler must not be null"));
            return future;
        }
        if (this.serverCapabilities.getTools() == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Server must be configured with tool capabilities"));
            return future;
        }
        return ((CompletableFuture)CompletableFuture.supplyAsync(() -> {
            if (this.tools.stream().anyMatch(th -> th.getTool().getName().equals(toolSpecification.getTool().getName()))) {
                throw new CompletionException(new McpError((Object)("Tool with name '" + toolSpecification.getTool().getName() + "' already exists")));
            }
            this.tools.add(toolSpecification);
            logger.debug("Added tool handler: {}", (Object)toolSpecification.getTool().getName());
            return null;
        }).thenCompose(ignored -> {
            if (this.serverCapabilities.getTools().getListChanged().booleanValue()) {
                return this.notifyToolsListChanged();
            }
            return CompletableFuture.completedFuture(null);
        })).exceptionally(ex -> {
            Throwable cause = ex instanceof CompletionException ? ex.getCause() : ex;
            logger.error("Error while adding tool", cause);
            throw new CompletionException(cause);
        });
    }

    public CompletableFuture<Void> removeTool(String toolName) {
        if (toolName == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Tool name must not be null"));
            return future;
        }
        if (this.serverCapabilities.getTools() == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Server must be configured with tool capabilities"));
            return future;
        }
        return ((CompletableFuture)CompletableFuture.supplyAsync(() -> {
            boolean removed = this.tools.removeIf(spec -> spec.getTool().getName().equals(toolName));
            if (!removed) {
                throw new CompletionException(new McpError((Object)("Tool with name '" + toolName + "' not found")));
            }
            logger.debug("Removed tool handler: {}", (Object)toolName);
            return null;
        }).thenCompose(ignored -> {
            if (this.serverCapabilities.getTools().getListChanged().booleanValue()) {
                return this.notifyToolsListChanged();
            }
            return CompletableFuture.completedFuture(null);
        })).exceptionally(ex -> {
            Throwable cause = ex instanceof CompletionException ? ex.getCause() : ex;
            logger.error("Error while removing tool '{}'", (Object)toolName, (Object)cause);
            throw new CompletionException(cause);
        });
    }

    public CompletableFuture<Void> notifyToolsListChanged() {
        logger.debug("Notifying clients about tool list changes");
        return this.mcpTransportProvider.notifyClients("notifications/tools/list_changed", null);
    }

    private McpRequestHandler<McpSchema.ListToolsResult> toolsListRequestHandler() {
        return (exchange, commandContext, params) -> {
            ArrayList<McpSchema.Tool> tools = new ArrayList<McpSchema.Tool>();
            for (McpServerFeatures.ToolSpecification toolSpec : this.tools) {
                tools.add(toolSpec.getTool());
            }
            return CompletableFuture.completedFuture(new McpSchema.ListToolsResult(tools, null));
        };
    }

    private McpRequestHandler<McpSchema.CallToolResult> toolsCallRequestHandler() {
        return (exchange, commandContext, params) -> {
            McpSchema.CallToolRequest callToolRequest = (McpSchema.CallToolRequest)this.objectMapper.convertValue(params, (TypeReference)new TypeReference<McpSchema.CallToolRequest>(){});
            Optional<McpServerFeatures.ToolSpecification> toolSpecification = this.tools.stream().filter(tr -> callToolRequest.getName().equals(tr.getTool().getName())).findAny();
            if (!toolSpecification.isPresent()) {
                CompletableFuture future = new CompletableFuture();
                future.completeExceptionally(new McpError((Object)("no tool found: " + callToolRequest.getName())));
                return future;
            }
            return toolSpecification.get().getCall().apply(exchange, commandContext, callToolRequest);
        };
    }

    public CompletableFuture<Void> addResource(McpServerFeatures.ResourceSpecification resourceSpecification) {
        if (resourceSpecification == null || resourceSpecification.getResource() == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Resource must not be null"));
            return future;
        }
        if (this.serverCapabilities.getResources() == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Server must be configured with resource capabilities"));
            return future;
        }
        return ((CompletableFuture)CompletableFuture.supplyAsync(() -> {
            if (this.resources.putIfAbsent(resourceSpecification.getResource().getUri(), resourceSpecification) != null) {
                throw new CompletionException(new McpError((Object)("Resource with URI '" + resourceSpecification.getResource().getUri() + "' already exists")));
            }
            logger.debug("Added resource handler: {}", (Object)resourceSpecification.getResource().getUri());
            return null;
        }).thenCompose(ignored -> {
            if (this.serverCapabilities.getResources().getListChanged().booleanValue()) {
                return this.notifyResourcesListChanged();
            }
            return CompletableFuture.completedFuture(null);
        })).exceptionally(ex -> {
            Throwable cause = ex instanceof CompletionException ? ex.getCause() : ex;
            logger.error("Error while adding resource '{}'", (Object)resourceSpecification.getResource().getUri(), (Object)cause);
            throw new CompletionException(cause);
        });
    }

    public CompletableFuture<Void> removeResource(String resourceUri) {
        if (resourceUri == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Resource URI must not be null"));
            return future;
        }
        if (this.serverCapabilities.getResources() == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Server must be configured with resource capabilities"));
            return future;
        }
        return ((CompletableFuture)CompletableFuture.supplyAsync(() -> {
            McpServerFeatures.ResourceSpecification removed = this.resources.remove(resourceUri);
            if (removed == null) {
                throw new CompletionException(new McpError((Object)("Resource with URI '" + resourceUri + "' not found")));
            }
            logger.debug("Removed resource handler: {}", (Object)resourceUri);
            return null;
        }).thenCompose(ignored -> {
            if (this.serverCapabilities.getResources().getListChanged().booleanValue()) {
                return this.notifyResourcesListChanged();
            }
            return CompletableFuture.completedFuture(null);
        })).exceptionally(ex -> {
            Throwable cause = ex instanceof CompletionException ? ex.getCause() : ex;
            logger.error("Error while removing resource '{}'", (Object)resourceUri, (Object)cause);
            throw new CompletionException(cause);
        });
    }

    public CompletableFuture<Void> notifyResourcesListChanged() {
        return this.mcpTransportProvider.notifyClients("notifications/resources/list_changed", null);
    }

    private McpRequestHandler<McpSchema.ListResourcesResult> resourcesListRequestHandler() {
        return (exchange, commandContext, params) -> {
            ArrayList<McpSchema.Resource> resourceList = new ArrayList<McpSchema.Resource>();
            for (McpServerFeatures.ResourceSpecification spec : this.resources.values()) {
                resourceList.add(spec.getResource());
            }
            return CompletableFuture.completedFuture(new McpSchema.ListResourcesResult(resourceList, null));
        };
    }

    private McpRequestHandler<McpSchema.ListResourceTemplatesResult> resourceTemplateListRequestHandler() {
        return (exchange, commandContext, params) -> CompletableFuture.completedFuture(new McpSchema.ListResourceTemplatesResult(this.resourceTemplates, null));
    }

    private McpRequestHandler<McpSchema.ReadResourceResult> resourcesReadRequestHandler() {
        return (exchange, commandContext, params) -> {
            McpSchema.ReadResourceRequest resourceRequest = (McpSchema.ReadResourceRequest)this.objectMapper.convertValue(params, (TypeReference)new TypeReference<McpSchema.ReadResourceRequest>(){});
            String resourceUri = resourceRequest.getUri();
            McpServerFeatures.ResourceSpecification specification = this.resources.get(resourceUri);
            if (specification != null) {
                return specification.getReadHandler().apply(exchange, resourceRequest);
            }
            CompletableFuture future = new CompletableFuture();
            future.completeExceptionally(new McpError((Object)("Resource not found: " + resourceUri)));
            return future;
        };
    }

    public CompletableFuture<Void> addPrompt(McpServerFeatures.PromptSpecification promptSpecification) {
        if (promptSpecification == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Prompt specification must not be null"));
            return future;
        }
        if (this.serverCapabilities.getPrompts() == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Server must be configured with prompt capabilities"));
            return future;
        }
        return ((CompletableFuture)CompletableFuture.supplyAsync(() -> {
            McpServerFeatures.PromptSpecification existing = this.prompts.putIfAbsent(promptSpecification.getPrompt().getName(), promptSpecification);
            if (existing != null) {
                throw new CompletionException(new McpError((Object)("Prompt with name '" + promptSpecification.getPrompt().getName() + "' already exists")));
            }
            logger.debug("Added prompt handler: {}", (Object)promptSpecification.getPrompt().getName());
            return null;
        }).thenCompose(ignored -> {
            if (this.serverCapabilities.getPrompts().getListChanged().booleanValue()) {
                return this.notifyPromptsListChanged();
            }
            return CompletableFuture.completedFuture(null);
        })).exceptionally(ex -> {
            Throwable cause = ex instanceof CompletionException ? ex.getCause() : ex;
            logger.error("Error while adding prompt '{}'", (Object)promptSpecification.getPrompt().getName(), (Object)cause);
            throw new CompletionException(cause);
        });
    }

    public CompletableFuture<Void> removePrompt(String promptName) {
        if (promptName == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Prompt name must not be null"));
            return future;
        }
        if (this.serverCapabilities.getPrompts() == null) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new McpError((Object)"Server must be configured with prompt capabilities"));
            return future;
        }
        return ((CompletableFuture)CompletableFuture.supplyAsync(() -> {
            McpServerFeatures.PromptSpecification removed = this.prompts.remove(promptName);
            if (removed == null) {
                throw new CompletionException(new McpError((Object)("Prompt with name '" + promptName + "' not found")));
            }
            logger.debug("Removed prompt handler: {}", (Object)promptName);
            return null;
        }).thenCompose(ignored -> {
            if (this.serverCapabilities.getPrompts().getListChanged().booleanValue()) {
                return this.notifyPromptsListChanged();
            }
            return CompletableFuture.completedFuture(null);
        })).exceptionally(ex -> {
            Throwable cause = ex instanceof CompletionException ? ex.getCause() : ex;
            logger.error("Error while removing prompt '{}'", (Object)promptName, (Object)cause);
            throw new CompletionException(cause);
        });
    }

    public CompletableFuture<Void> notifyPromptsListChanged() {
        return this.mcpTransportProvider.notifyClients("notifications/prompts/list_changed", null);
    }

    private McpRequestHandler<McpSchema.ListPromptsResult> promptsListRequestHandler() {
        return (exchange, commandContext, params) -> {
            ArrayList<McpSchema.Prompt> promptList = new ArrayList<McpSchema.Prompt>();
            for (McpServerFeatures.PromptSpecification promptSpec : this.prompts.values()) {
                promptList.add(promptSpec.getPrompt());
            }
            return CompletableFuture.completedFuture(new McpSchema.ListPromptsResult(promptList, null));
        };
    }

    private McpRequestHandler<McpSchema.GetPromptResult> promptsGetRequestHandler() {
        return (exchange, commandContext, params) -> {
            McpSchema.GetPromptRequest promptRequest = (McpSchema.GetPromptRequest)this.objectMapper.convertValue(params, (TypeReference)new TypeReference<McpSchema.GetPromptRequest>(){});
            McpServerFeatures.PromptSpecification specification = this.prompts.get(promptRequest.getName());
            if (specification != null) {
                return specification.getPromptHandler().apply(exchange, promptRequest);
            }
            CompletableFuture future = new CompletableFuture();
            future.completeExceptionally(new McpError((Object)("Prompt not found: " + promptRequest.getName())));
            return future;
        };
    }

    public CompletableFuture<Void> loggingNotification(McpSchema.LoggingMessageNotification loggingMessageNotification) {
        Assert.notNull(loggingMessageNotification, "Logging message must not be null");
        if (loggingMessageNotification.getLevel().level() < this.minLoggingLevel.level()) {
            return CompletableFuture.completedFuture(null);
        }
        return this.mcpTransportProvider.notifyClients("notifications/message", loggingMessageNotification);
    }

    private McpRequestHandler<Map<String, Object>> setLoggerRequestHandler() {
        return (exchange, commandContext, params) -> {
            try {
                McpSchema.SetLevelRequest request = (McpSchema.SetLevelRequest)this.objectMapper.convertValue(params, McpSchema.SetLevelRequest.class);
                this.minLoggingLevel = request.getLevel();
                return CompletableFuture.completedFuture(Collections.emptyMap());
            }
            catch (Exception e) {
                CompletableFuture future = new CompletableFuture();
                future.completeExceptionally(new McpError((Object)("An error occurred while processing a request to set the log level: " + e.getMessage())));
                return future;
            }
        };
    }

    public void setProtocolVersions(List<String> protocolVersions) {
        this.protocolVersions = protocolVersions;
    }
}

