/**
 *
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "[]"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2016 Alibaba Group
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package com.taobao.weex.ui.view.border;

import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.ColorFilter;
import android.graphics.Outline;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Shader;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.SparseArray;

import com.taobao.weex.dom.flex.FloatUtil;
import com.taobao.weex.dom.flex.Spacing;
import com.taobao.weex.utils.WXLogUtils;
import com.taobao.weex.utils.WXViewUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

/**
 * A subclass of
 * {@link Drawable} used for background of {@link com.taobao.weex.ui.component.WXComponent}.
 * It supports drawing background color and borders (including rounded borders) by providing a react
 * friendly API (setter for each of those properties). The implementation tries to allocate as few
 * objects as possible depending on which properties are set. E.g. for views with rounded
 * background/borders we allocate {@code mPathForBorderDrawn} and {@code mTempRectForBorderRadius}.
 * In case when view have a rectangular borders we allocate {@code mBorderWidthResult} and similar.
 * When only background color is set we won't allocate any extra/unnecessary objects.
 */
public class BorderDrawable extends Drawable {

  public static final int BORDER_TOP_LEFT_RADIUS = 0;
  public static final int BORDER_TOP_RIGHT_RADIUS = 1;
  public static final int BORDER_BOTTOM_RIGHT_RADIUS = 2;
  public static final int BORDER_BOTTOM_LEFT_RADIUS = 3;
  public static final int BORDER_RADIUS_ALL = 8;
  static final int DEFAULT_BORDER_COLOR = Color.BLACK;
  static final float DEFAULT_BORDER_WIDTH = 0;
  static final float DEFAULT_BORDER_RADIUS = 0;
  static final BorderStyle DEFAULT_BORDER_STYLE = BorderStyle.SOLID;
  private static final String TAG = "Border";
  private final Paint mPaint = new Paint(Paint.ANTI_ALIAS_FLAG);

  private
  @Nullable
  SparseArray<Float> mBorderWidth;
  private
  @Nullable
  SparseArray<Float> mBorderRadius;
  private
  @Nullable
  SparseArray<Float> mOverlappingBorderRadius;
  private
  @Nullable
  SparseArray<Integer> mBorderColor;
  private
  @Nullable
  SparseArray<Integer> mBorderStyle;

  private
  @Nullable
  Path mPathForBorderOutline;
  private boolean mNeedUpdatePath = false;
  private int mColor = Color.TRANSPARENT;
  private int mAlpha = 255;

  public BorderDrawable() {
  }

  @Override
  public void draw(Canvas canvas) {
    canvas.save();
    updateBorderOutline();
    if (mPathForBorderOutline != null) {
      int useColor = WXViewUtils.multiplyColorAlpha(mColor, mAlpha);
      if ((useColor >>> 24) != 0) {
        mPaint.setStyle(Paint.Style.FILL);
        mPaint.setColor(useColor);
        mPaint.setShader(null);
        canvas.drawPath(mPathForBorderOutline, mPaint);
      }
    }
    mPaint.setStyle(Paint.Style.STROKE);
    mPaint.setStrokeJoin(Paint.Join.ROUND);
    drawBorders(canvas);
    mPaint.setShader(null);
    canvas.restore();
  }

  @Override
  protected void onBoundsChange(Rect bounds) {
    super.onBoundsChange(bounds);
    mNeedUpdatePath = true;
  }

  public void setBorderWidth(int position, float width) {
    if (mBorderWidth == null) {
      mBorderWidth = new SparseArray<>(5);
      mBorderWidth.put(Spacing.ALL, DEFAULT_BORDER_WIDTH);
    }
    if (!FloatUtil.floatsEqual(getBorderWidth(position), width)) {
      updateSparseArray(mBorderWidth, position, width);
      mBorderWidth.put(position, width);
      mNeedUpdatePath = true;
      invalidateSelf();
    }
  }

  float getBorderWidth(int position) {
    return fetchFromSparseArray(mBorderWidth, position, DEFAULT_BORDER_WIDTH);
  }

  @Override
  public void setAlpha(int alpha) {
    if (alpha != mAlpha) {
      mAlpha = alpha;
      invalidateSelf();
    }
  }

  private <T> void updateSparseArray(@NonNull SparseArray<T> array, int position, T value) {
    updateSparseArray(array, position, value, true);
  }

  @Override
  public int getAlpha() {
    return mAlpha;
  }

  private <T> T fetchFromSparseArray(@Nullable SparseArray<T> array, int position, T fallback) {
    return array == null ? fallback :
           array.get(position, array.get(Spacing.ALL));
  }

  /**
   * Do not support Color Filter
   */
  @Override
  public void setColorFilter(ColorFilter cf) {

  }

  private <T> void updateSparseArray(@NonNull SparseArray<T> array, int position, T value,
                                     boolean useSpacing) {
    if (useSpacing) {
      if (position == Spacing.ALL) {
        array.put(Spacing.ALL, value);
        array.put(Spacing.TOP, value);
        array.put(Spacing.LEFT, value);
        array.put(Spacing.RIGHT, value);
        array.put(Spacing.BOTTOM, value);
      } else {
        array.put(position, value);
      }
    } else {
      if (position == BORDER_RADIUS_ALL) {
        array.put(BORDER_RADIUS_ALL, value);
        array.put(BORDER_TOP_LEFT_RADIUS, value);
        array.put(BORDER_TOP_RIGHT_RADIUS, value);
        array.put(BORDER_BOTTOM_LEFT_RADIUS, value);
        array.put(BORDER_BOTTOM_RIGHT_RADIUS, value);
      } else {
        array.put(position, value);
      }
    }
  }

  @Override
  public int getOpacity() {
    return WXViewUtils.getOpacityFromColor(WXViewUtils.multiplyColorAlpha(mColor, mAlpha));
  }

  public void setBorderRadius(int position, float radius) {
    if (mBorderRadius == null) {
      mBorderRadius = new SparseArray<>(5);
      mBorderRadius.put(Spacing.ALL, DEFAULT_BORDER_RADIUS);
    }
    if (!FloatUtil.floatsEqual(getBorderRadius(mBorderRadius, position), radius)) {
      updateSparseArray(mBorderRadius, position, radius, false);
      mNeedUpdatePath = true;
      invalidateSelf();
    }
  }  /* Android's elevation implementation requires this to be implemented to know where to draw the
   shadow. */

  @Override
  public void getOutline(@NonNull Outline outline) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      if (mPathForBorderOutline == null) {
        mNeedUpdatePath = true;
      }
      updateBorderOutline();
      outline.setConvexPath(mPathForBorderOutline);
    }
  }

  float getBorderRadius(@Nullable SparseArray<Float> borderRadius, int position) {
    return fetchFromSparseArray(borderRadius, position, DEFAULT_BORDER_RADIUS);
  }

  public void setBorderColor(int position, int color) {
    if (mBorderColor == null) {
      mBorderColor = new SparseArray<>(5);
      mBorderColor.put(Spacing.ALL, DEFAULT_BORDER_COLOR);
    }
    if (getBorderColor(position) != color) {
      updateSparseArray(mBorderColor, position, color);
      invalidateSelf();
    }
  }

  int getBorderColor(int position) {
    return fetchFromSparseArray(mBorderColor, position, DEFAULT_BORDER_COLOR);
  }

  public void setBorderStyle(int position, @NonNull String style) {
    if (mBorderStyle == null) {
      mBorderStyle = new SparseArray<>(5);
      mBorderStyle.put(Spacing.ALL, DEFAULT_BORDER_STYLE.ordinal());
    }
    try {
      int borderStyle = BorderStyle.valueOf(style.toUpperCase(Locale.US)).ordinal();
      if (getBorderStyle(position) != borderStyle) {
        updateSparseArray(mBorderStyle, position, borderStyle);
        invalidateSelf();
      }
    } catch (IllegalArgumentException e) {
      WXLogUtils.e(TAG, WXLogUtils.getStackTrace(e));
    }
  }

  int getBorderStyle(int position) {
    return fetchFromSparseArray(mBorderStyle, position, BorderStyle.SOLID.ordinal());
  }

  public int getColor() {
    return mColor;
  }

  public void setColor(int color) {
    mColor = color;
    invalidateSelf();
  }

  public
  @NonNull
  Path getContentPath(int viewTopPadding,
                      int viewRightPadding,
                      int viewBottomPadding,
                      int viewLeftPadding,
                      @NonNull RectF contentBox) {
    RectF rectForBorderOutline = new RectF();
    Path contentClip = new Path();
    rectForBorderOutline.set(contentBox);
    if (mBorderRadius != null) {
      prepareBorderRadius();
      float topLeftRadius = getBorderRadius(mOverlappingBorderRadius, BORDER_TOP_LEFT_RADIUS);
      float topRightRadius = getBorderRadius(mOverlappingBorderRadius, BORDER_TOP_RIGHT_RADIUS);
      float bottomRightRadius = getBorderRadius(mOverlappingBorderRadius,
                                                BORDER_BOTTOM_RIGHT_RADIUS);
      float bottomLeftRadius = getBorderRadius(mOverlappingBorderRadius,
                                               BORDER_BOTTOM_LEFT_RADIUS);
      contentClip.addRoundRect(rectForBorderOutline,
                               new float[]{
                                   topLeftRadius - viewLeftPadding,
                                   topLeftRadius - viewTopPadding,
                                   topRightRadius - viewRightPadding,
                                   topRightRadius - viewTopPadding,
                                   bottomRightRadius - viewRightPadding,
                                   bottomRightRadius - viewBottomPadding,
                                   bottomLeftRadius - viewLeftPadding,
                                   bottomLeftRadius - viewBottomPadding
                               },
                               Path.Direction.CW);
    } else {
      contentClip.addRect(rectForBorderOutline, Path.Direction.CW);
    }
    return contentClip;
  }

  /**
   * This method is only used for Unit test, do not call this method, use
   * {@link #getBorderRadius(SparseArray, int)} instead.
   * @param position the index of the edge
   * @return the radius considering border-overlapping of the corner.
   */
  @Deprecated
  float getBorderRadius(int position) {
    return getBorderRadius(mOverlappingBorderRadius, position);
  }

  private void updateBorderOutline() {
    if (mNeedUpdatePath) {
      mNeedUpdatePath = false;
      if (mPathForBorderOutline == null) {
        mPathForBorderOutline = new Path();
      }
      mPathForBorderOutline.reset();
      RectF mRectForBorderOutline = new RectF();
      mRectForBorderOutline.set(getBounds());

      if (mBorderRadius != null) {
        prepareBorderRadius();
        float topLeftRadius = getBorderRadius(mOverlappingBorderRadius, BORDER_TOP_LEFT_RADIUS);
        float topRightRadius = getBorderRadius(mOverlappingBorderRadius, BORDER_TOP_RIGHT_RADIUS);
        float bottomRightRadius = getBorderRadius(mOverlappingBorderRadius,
                                                  BORDER_BOTTOM_RIGHT_RADIUS);
        float bottomLeftRadius = getBorderRadius(mOverlappingBorderRadius,
                                                 BORDER_BOTTOM_LEFT_RADIUS);
        mPathForBorderOutline.addRoundRect(
            mRectForBorderOutline,
            new float[]{
                topLeftRadius, topLeftRadius,
                topRightRadius, topRightRadius,
                bottomRightRadius, bottomRightRadius,
                bottomLeftRadius, bottomLeftRadius
            },
            Path.Direction.CW);
      } else {
        mPathForBorderOutline.addRect(mRectForBorderOutline, Path.Direction.CW);
      }
    }
  }

  private void drawBorders(Canvas canvas) {
    if (mBorderWidth != null) {
      RectF rectBounds = new RectF(getBounds());
      BorderCorner topLeft = new TopLeftCorner(
          getBorderRadius(mOverlappingBorderRadius, BORDER_TOP_LEFT_RADIUS),
          getBorderWidth(Spacing.LEFT),
          getBorderWidth(Spacing.TOP),
          rectBounds);
      BorderCorner topRight = new TopRightCorner(
          getBorderRadius(mOverlappingBorderRadius, BORDER_TOP_RIGHT_RADIUS),
          getBorderWidth(Spacing.TOP),
          getBorderWidth(Spacing.RIGHT),
          rectBounds);
      BorderCorner bottomRight = new BottomRightCorner(
          getBorderRadius(mOverlappingBorderRadius, BORDER_BOTTOM_RIGHT_RADIUS),
          getBorderWidth(Spacing.RIGHT),
          getBorderWidth(Spacing.BOTTOM),
          rectBounds);
      BorderCorner bottomLeft = new BottomLeftCorner(
          getBorderRadius(mOverlappingBorderRadius, BORDER_BOTTOM_LEFT_RADIUS),
          getBorderWidth(Spacing.BOTTOM),
          getBorderWidth(Spacing.LEFT),
          rectBounds);
      drawOneSide(canvas, new BorderEdge(topLeft, topRight, Spacing.TOP,
                                         getBorderWidth(Spacing.TOP)));
      drawOneSide(canvas, new BorderEdge(topRight, bottomRight, Spacing.RIGHT,
                                         getBorderWidth(Spacing.RIGHT)));
      drawOneSide(canvas, new BorderEdge(bottomRight, bottomLeft, Spacing.BOTTOM,
                                         getBorderWidth(Spacing.BOTTOM)));
      drawOneSide(canvas, new BorderEdge(bottomLeft, topLeft, Spacing.LEFT,
                                         getBorderWidth(Spacing.LEFT)));
    }
  }

  /**
   * Process overlapping curve according to https://www.w3.org/TR/css3-background/#corner-overlap .
   */
  private void prepareBorderRadius() {
    if (mBorderRadius != null) {
      float factor = getScaleFactor();
      if (mOverlappingBorderRadius == null) {
        mOverlappingBorderRadius = new SparseArray<>(5);
        mOverlappingBorderRadius.put(Spacing.ALL, 0f);
      }
      if (!Float.isNaN(factor) && factor < 1) {
        mOverlappingBorderRadius.put(BORDER_TOP_LEFT_RADIUS,
                                     getBorderRadius(mBorderRadius, BORDER_TOP_LEFT_RADIUS) *
                                     factor);
        mOverlappingBorderRadius.put(BORDER_TOP_RIGHT_RADIUS,
                                     getBorderRadius(mBorderRadius, BORDER_TOP_RIGHT_RADIUS) *
                                     factor);
        mOverlappingBorderRadius.put(BORDER_BOTTOM_RIGHT_RADIUS,
                                     getBorderRadius(mBorderRadius, BORDER_BOTTOM_RIGHT_RADIUS) *
                                     factor);
        mOverlappingBorderRadius.put(BORDER_BOTTOM_LEFT_RADIUS,
                                     getBorderRadius(mBorderRadius, BORDER_BOTTOM_LEFT_RADIUS) *
                                     factor);
      } else {
        mOverlappingBorderRadius.put(BORDER_TOP_LEFT_RADIUS,
                                     getBorderRadius(mBorderRadius, BORDER_TOP_LEFT_RADIUS));
        mOverlappingBorderRadius.put(BORDER_TOP_RIGHT_RADIUS,
                                     getBorderRadius(mBorderRadius, BORDER_TOP_RIGHT_RADIUS));
        mOverlappingBorderRadius.put(BORDER_BOTTOM_RIGHT_RADIUS,
                                     getBorderRadius(mBorderRadius, BORDER_BOTTOM_RIGHT_RADIUS));
        mOverlappingBorderRadius.put(BORDER_BOTTOM_LEFT_RADIUS,
                                     getBorderRadius(mBorderRadius, BORDER_BOTTOM_LEFT_RADIUS));
      }
    }
  }

  private float getScaleFactor() {
    Rect borderBox = getBounds();
    final float topRadius = getBorderRadius(mBorderRadius, BORDER_TOP_LEFT_RADIUS)
                            + getBorderRadius(mBorderRadius, BORDER_TOP_RIGHT_RADIUS);
    final float rightRadius = getBorderRadius(mBorderRadius, BORDER_TOP_RIGHT_RADIUS)
                              + getBorderRadius(mBorderRadius, BORDER_BOTTOM_RIGHT_RADIUS);
    final float bottomRadius = getBorderRadius(mBorderRadius, BORDER_BOTTOM_RIGHT_RADIUS)
                               + getBorderRadius(mBorderRadius, BORDER_BOTTOM_LEFT_RADIUS);
    final float leftRadius = getBorderRadius(mBorderRadius, BORDER_BOTTOM_LEFT_RADIUS)
                             + getBorderRadius(mBorderRadius, BORDER_TOP_LEFT_RADIUS);
    List<Float> factors = new ArrayList<>(4);
    updateFactor(factors, borderBox.width(), topRadius);
    updateFactor(factors, borderBox.height(), rightRadius);
    updateFactor(factors, borderBox.width(), bottomRadius);
    updateFactor(factors, borderBox.height(), leftRadius);
    float factor;
    if (factors.isEmpty()) {
      factor = Float.NaN;
    } else {
      factor = Collections.min(factors);
    }
    return factor;
  }

  private void updateFactor(@NonNull List<Float> list, float numerator, float denominator) {
    if (!FloatUtil.floatsEqual(denominator, 0)) {
      list.add(numerator / denominator);
    }
  }

  private void drawOneSide(Canvas canvas, @NonNull BorderEdge borderEdge) {
    if (!FloatUtil.floatsEqual(0, getBorderWidth(borderEdge.getEdge()))) {
      preparePaint(borderEdge.getEdge());
      borderEdge.drawEdge(canvas, mPaint);
    }
  }

  private void preparePaint(int side) {
    float borderWidth = getBorderWidth(side);
    int color = WXViewUtils.multiplyColorAlpha(getBorderColor(side), mAlpha);
    BorderStyle borderStyle = BorderStyle.values()[getBorderStyle(side)];
    Shader shader = borderStyle.getLineShader(borderWidth, color, side);
    mPaint.setShader(shader);
    mPaint.setColor(color);
    mPaint.setStrokeWidth(borderWidth);
  }
}
