/**
 *
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "[]"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2016 Alibaba Group
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package com.taobao.weex.dom;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Pair;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.taobao.weex.WXEnvironment;
import com.taobao.weex.WXSDKInstance;
import com.taobao.weex.WXSDKManager;
import com.taobao.weex.adapter.IWXUserTrackAdapter;
import com.taobao.weex.common.Constants;
import com.taobao.weex.common.WXErrorCode;
import com.taobao.weex.dom.flex.CSSLayoutContext;
import com.taobao.weex.dom.flex.CSSNode;
import com.taobao.weex.dom.flex.Spacing;
import com.taobao.weex.ui.IWXRenderTask;
import com.taobao.weex.ui.WXRenderManager;
import com.taobao.weex.ui.animation.WXAnimationBean;
import com.taobao.weex.ui.component.WXComponent;
import com.taobao.weex.ui.component.WXVContainer;
import com.taobao.weex.utils.WXDataStructureUtil;
import com.taobao.weex.utils.WXLogUtils;
import com.taobao.weex.utils.WXViewUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * <p>
 * This class is responsible for creating command object of DOM operation and
 * invoking command of corresponding object.
 * </p>
 * <p>
 * In the command design pattern,
 * this class acts as the <strong>invoker </strong>in the command pattern
 * despite that it is also responsible for creating <strong>Command</strong> object.
 * And,{@link IWXRenderTask} works as the <strong>Command</strong>
 * {@link WXDomManager} works as the <strong>Client</strong>
 * {@link WXRenderManager} works as the <strong>Receiver</strong>.
 * </p>
 * <p>
 * There exists one to one correspondence between WXDomStatement and WXInstance,
 * and {@link WXDomManager} is responsible for manage the relation of correspondence.
 * </p>
 */
class WXDomStatement {

  public static final String CHILDREN = "children";
  public static final String TYPE = "type";
  /** package **/ final ConcurrentHashMap<String, WXDomObject> mRegistry;
  private String mInstanceId;
  private WXRenderManager mWXRenderManager;
  private ArrayList<IWXRenderTask> mNormalTasks;
  private Set <Pair<String, Map<String, Object>>> animations;
  private CSSLayoutContext mLayoutContext;
  private volatile boolean mDirty;
  private boolean mDestroy;
  private Map<String, AddDomInfo> mAddDom = new HashMap<>();

  /**
   * Create an instance of {@link WXDomStatement},
   * One {@link WXSDKInstance} corresponding to one and only one {@link WXDomStatement}.
   * And all the instance of {@link WXDomManager} share the same {@link WXRenderManager}.
   * @param instanceId the id of the {@link WXSDKInstance}.
   *                   One {@link WXSDKInstance} corresponding to one {@link WXDomStatement},
   *                   and vice versa.
   * @param renderManager This acts as the Receiver of the command pattern
   */
  public WXDomStatement(String instanceId, WXRenderManager renderManager) {
    mDestroy = false;
    mInstanceId = instanceId;
    mLayoutContext = new CSSLayoutContext();
    mRegistry = new ConcurrentHashMap<>();
    mNormalTasks = new ArrayList<>();
    animations = new HashSet<>();
    mWXRenderManager = renderManager;
  }

  /**
   * Destroy current instance, which occurred when {@link WXSDKInstance#destroy()} is called.
   */
  public void destroy() {
    mDestroy = true;
    mRegistry.clear();
  }

  /**
   * Rebuild the component tree.
   * The purpose of this method is moving fixed components to the root component.
   * This method will be called when {@link #batch()} is executed.
   * @param root root dom
   */
  void rebuildingDomTree(WXDomObject root) {
    if (root != null && root.getFixedStyleRefs() != null) {
      int size = root.getFixedStyleRefs().size();
      for (int i = 0; i < size; i++) {
        String fixedRef = root.getFixedStyleRefs().get(i);
        WXDomObject wxDomObject = mRegistry.get(fixedRef);
        if (wxDomObject!=null && wxDomObject.parent != null) {
          wxDomObject.parent.remove(wxDomObject);
          root.add(wxDomObject, -1);
        }
      }
    }

  }

  /**
   * Batch the execution of command objects and execute all the command objects created other
   * places, e.g. call {@link IWXRenderTask#execute()}.
   * First, it will rebuild the dom tree and do pre layout staff.
   * Then call {@link com.taobao.weex.dom.flex.CSSNode#calculateLayout(CSSLayoutContext)} to
   * start calculate layout.
   * Next, call {@link #applyUpdate(WXDomObject)} to get changed dom and creating
   * corresponding command object.
   * Finally, walk through the queue, e.g. call {@link IWXRenderTask#execute()} for every task
   * in the queue.
   */
  void batch() {
    long start0 = System.currentTimeMillis();

    if (!mDirty || mDestroy) {
      return;
    }

    WXDomObject rootDom = mRegistry.get(WXDomObject.ROOT);
    if (rootDom == null) {
      return;
    }

    rebuildingDomTree(rootDom);

    layoutBefore(rootDom);
    long start = System.currentTimeMillis();


    rootDom.calculateLayout(mLayoutContext);

    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    if(instance != null) {
      instance.cssLayoutTime(System.currentTimeMillis() - start);
    }

    layoutAfter(rootDom);

    start = System.currentTimeMillis();
    applyUpdate(rootDom);

    if(instance != null) {
      instance.applyUpdateTime(System.currentTimeMillis() - start);
    }

    start = System.currentTimeMillis();
    updateDomObj();
    if(instance != null) {
      instance.updateDomObjTime(System.currentTimeMillis() - start);
    }
    parseAnimation();

    int count = mNormalTasks.size();
    for (int i = 0; i < count && !mDestroy; ++i) {
      mWXRenderManager.runOnThread(mInstanceId, mNormalTasks.get(i));
    }
    mNormalTasks.clear();
    mAddDom.clear();
    animations.clear();
    mDirty = false;
    if(instance != null) {
      instance.batchTime(System.currentTimeMillis() - start0);
    }

  }

  /**
   * Calling given dom and its children's {@link WXDomObject#layoutBefore()}
   * @param dom the root domObject for recursive call
   */
  private void layoutBefore(WXDomObject dom) {
    if (dom == null || !dom.hasUpdate() || mDestroy) {
      return;
    }
    dom.layoutBefore();
    int count = dom.childCount();
    for (int i = 0; i < count; ++i) {
      layoutBefore(dom.getChild(i));
    }
  }

  private void layoutAfter(WXDomObject dom){
    if (dom == null || !dom.hasUpdate() || mDestroy) {
      return;
    }
    dom.layoutAfter();
    int count = dom.childCount();
    for (int i = 0; i < count; ++i) {
      layoutAfter(dom.getChild(i));
    }
  }

  /**
   * Walk through the dom tree and create command object of re-calculating
   * {@link android.view.ViewGroup.LayoutParams} for dom that is old.
   * @param dom the root dom of the walk through.
   */
  private void applyUpdate(WXDomObject dom) {
    if (dom == null) {
      return;
    }
    if (dom.hasUpdate()) {
      dom.markUpdateSeen();
      if (!dom.isYoung()) {
        final WXDomObject copy = dom.clone();
        if (copy == null) {
          return;
        }
        mNormalTasks.add(new IWXRenderTask() {

          @Override
          public void execute() {
            mWXRenderManager.setLayout(mInstanceId, copy.getRef(), copy);
          }

          @Override
          public String toString() {
            return "setLayout";
          }
        });
        if (dom.getExtra() != null) {
          mNormalTasks.add(new IWXRenderTask() {

            @Override
            public void execute() {
              mWXRenderManager.setExtra(mInstanceId, copy.getRef(), copy.getExtra());
            }

            @Override
            public String toString() {
              return "setExtra";
            }
          });
        }
      }
    }
    int count = dom.childCount();
    for (int i = 0; i < count; ++i) {
      applyUpdate(dom.getChild(i));
    }
  }

  private void parseAnimation() {
    for(final Pair<String, Map<String, Object>> pair:animations) {
      if (!TextUtils.isEmpty(pair.first)) {
        final WXAnimationBean animationBean = createAnimationBean(pair.first, pair.second);
        if (animationBean != null) {
          mNormalTasks.add(new IWXRenderTask() {
            @Override
            public void execute() {
              mWXRenderManager.startAnimation(mInstanceId, pair.first, animationBean, null);
            }

            @Override
            public String toString() {
              return "startAnimationByStyle";
            }
          });
        }
      }
    }
  }

  /**
   * Create command object for creating body according to the JSONObject. And put the command
   * object in the queue.
   * @param element the jsonObject according to which to create command object.
   */
  void createBody(JSONObject element) {
    if (mDestroy) {
      return;
    }
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    if (element == null) {
      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_ERR_DOM_CREATEBODY);
      }
      return;
    }

    WXDomObject domObject = parseInner(element);
    if(domObject==null){
      return;
    }
    Map<String, Object> style = new HashMap<>(5);
    if (!domObject.getStyles().containsKey(Constants.Name.FLEX_DIRECTION)) {
      style.put(Constants.Name.FLEX_DIRECTION, "column");
    }
    if (!domObject.getStyles().containsKey(Constants.Name.BACKGROUND_COLOR)) {
      style.put(Constants.Name.BACKGROUND_COLOR, "#ffffff");
    }
    //If there is height or width in JS, then that value will override value here.
    if ( !domObject.getStyles().containsKey(Constants.Name.WIDTH)) {
      style.put(Constants.Name.WIDTH, WXViewUtils.getWebPxByWidth(WXViewUtils.getWeexWidth(mInstanceId)));
      domObject.setModifyWidth(true);
    }
    if ( !domObject.getStyles().containsKey(Constants.Name.HEIGHT)) {
      style.put(Constants.Name.HEIGHT, WXViewUtils.getWebPxByWidth(WXViewUtils.getWeexHeight(mInstanceId)));
      domObject.setModifyHeight(true);
    }
    WXDomObject.prepareRoot(domObject);
    domObject.updateStyle(style);
    transformStyle(domObject, true);

    try {
      final WXComponent component = mWXRenderManager.createBodyOnDomThread(mInstanceId, domObject);
      AddDomInfo addDomInfo = new AddDomInfo();
      addDomInfo.component = component;
      mAddDom.put(domObject.getRef(), addDomInfo);

      mNormalTasks.add(new IWXRenderTask() {

        @Override
        public void execute() {
          WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
          if (instance == null || instance.getContext() == null) {
            WXLogUtils.e("instance is null or instance is destroy!");
            return;
          }
          try {
            mWXRenderManager.createBody(mInstanceId, component);
          } catch (Exception e) {
            WXLogUtils.e("create body failed.", e);
          }
        }

        @Override
        public String toString() {
          return "createBody";
        }
      });
      animations.add(new Pair<String, Map<String, Object>>(domObject.getRef(),domObject.getStyles()));
      mDirty = true;

      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
      }
    }catch (Exception e){

      WXLogUtils.e("create body in dom thread failed." + e.getMessage());
    }
  }

  /**
   * Update all components' dom info stored in {@link #mAddDom}
   */
  private void updateDomObj() {
    long start = System.currentTimeMillis();
    Iterator<Map.Entry<String, AddDomInfo>> iterator = mAddDom.entrySet().iterator();
    Map.Entry<String, AddDomInfo> entry;
    AddDomInfo value;
    while (iterator.hasNext()) {
      entry = iterator.next();
      value = entry.getValue();
      updateDomObj(value.component);
    }
    if (WXEnvironment.isApkDebugable()) {
      WXLogUtils.d("updateDomObj", "time:" + (System.currentTimeMillis() - start));
    }
  }

  /**
   * Update the specified component's dom and mark it as old.
   * @param component the component to be updated
   */
  private void updateDomObj(WXComponent component) {
    if (component == null) {
      return;
    }
    WXDomObject domObject = mRegistry.get(component.getRef());
    if (domObject == null) {
      return;
    }
    domObject.old();
    component.updateDom(domObject.clone());
    if (component instanceof WXVContainer) {
      WXVContainer container = (WXVContainer) component;
      int count = container.childCount();
      for (int i = 0; i < count; ++i) {
        updateDomObj(container.getChild(i));
      }
    }
  }

  /**
   * Create a command object for adding a dom node to its parent in a specific location.
   * If dom's parent doesn't exist or the dom has been added in current {@link WXSDKInstance},
   * this method will return.
   * If the above request is met, then put the command object in the queue.
   * @param dom the dom object in the form of JSONObject
   * @param parentRef parent to which the dom is added.
   * @param index the location of which the dom is added.
   */
  void addDom(JSONObject dom, final String parentRef, final int index) {
    if (mDestroy) {
      return;
    }
    WXDomObject parent = mRegistry.get(parentRef);
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);

    if (parent == null) {
      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_ERR_DOM_ADDELEMENT);
      }
      return;
    }
    WXDomObject domObject = parseInner(dom);

    if (domObject == null || mRegistry.containsKey(domObject.getRef())) {
      if (WXEnvironment.isApkDebugable()) {
        WXLogUtils.e("[WXDomStatement] addDom error!!");
      }
      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_ERR_DOM_ADDELEMENT);
      }
      return;
    }

    findFixed(domObject);

    parent.add(domObject, index);

    transformStyle(domObject, true);

    //Create component in dom thread
    final WXComponent component = mWXRenderManager.createComponentOnDomThread(mInstanceId, domObject, parentRef, index);
    if(component == null){
      //stop redner, some fatal happened.
      return;
    }
    AddDomInfo addDomInfo = new AddDomInfo();
    addDomInfo.component = component;
    mAddDom.put(domObject.getRef(), addDomInfo);

    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
        if(instance == null || instance.getContext()== null) {
          WXLogUtils.e("instance is null or instance is destroy!");
          return;
        }
        try {
          mWXRenderManager.addComponent(mInstanceId, component, parentRef, index);
        }catch (Exception e){
          WXLogUtils.e("add component failed.", e);
        }
      }

      @Override
      public String toString() {
        return "AddDom";
      }
    });
    animations.add(new Pair<String, Map<String, Object>>(domObject.getRef(),domObject.getStyles()));
    mDirty = true;

    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  /**
   * Find fixed node and tell root dom
   * @param obj
   */
  void findFixed(WXDomObject obj){
    WXDomObject rootDom = mRegistry.get(WXDomObject.ROOT);
    if (rootDom == null) {
      return;
    }
    if (obj.isFixed()) {
      rootDom.add2FixedDomList(obj.getRef());
    }

    int childrenCount = obj.childCount();
    if(childrenCount > 0){
      for (int i = 0;i < childrenCount;i++){
        findFixed(obj.getChild(i));
      }
    }
  }


  /**
   * Create a command object for moving the specific {@link WXDomObject} to a new parent.
   * If any of the following situation is met,
   * <ul>
   * <li> dom to be moved is null </li>
   * <li> dom's parent is null </li>
   * <li> new parent is null </li>
   * <li> parent is under {@link CSSNode#hasNewLayout()} </li>
   * </ul>
   * this method will return. Otherwise, put the command object in the queue.
   * @param ref Reference of the dom to be moved.
   * @param parentRef Reference of the new parent DOM node
   * @param index the index of the dom to be inserted in the new parent.
   */
  void moveDom(final String ref, final String parentRef, final int index) {
    if (mDestroy) {
      return;
    }
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    WXDomObject domObject = mRegistry.get(ref);
    WXDomObject parentObject = mRegistry.get(parentRef);
    if (domObject == null || domObject.parent == null
        || parentObject == null || parentObject.hasNewLayout()) {
      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_ERR_DOM_MOVEELEMENT);
      }
      return;
    }
    if(domObject.parent.equals(parentObject)){
      return ;
    }
    domObject.parent.remove(domObject);
    parentObject.add(domObject, index);

    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        mWXRenderManager.moveComponent(mInstanceId, ref, parentRef, index);
      }

      @Override
      public String toString() {
        return "moveDom";
      }
    });

    mDirty = true;
    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  /**
   * Create a command object for removing the specified {@link WXDomObject}.
   * If the domObject is null or its parent is null, this method returns directly.
   * Otherwise, put the command object in the queue.
   * @param ref Reference of the dom.
   */
  void removeDom(final String ref) {
    if (mDestroy) {
      return;
    }
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    WXDomObject domObject = mRegistry.get(ref);
    if (domObject == null) {
      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_ERR_DOM_REMOVEELEMENT);
      }
      return;
    }
    WXDomObject parent = domObject.parent;
    if (parent == null) {
      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_ERR_DOM_REMOVEELEMENT);
      }
      return;
    }
    clearRegistryForDom(domObject);
    parent.remove(domObject);
    mRegistry.remove(ref);

    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        mWXRenderManager.removeComponent(mInstanceId, ref);
      }

      @Override
      public String toString() {
        return "removeDom";
      }
    });

    mDirty = true;
    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  /**
   * Clear the mapping relationship between Reference and {@link WXDomObject}.
   * The mapping info is stored in {@link #mRegistry}.
   * @param domObject
   */
  private void clearRegistryForDom(WXDomObject domObject) {
    int count = domObject.childCount();
    mRegistry.remove(domObject.getRef());
    for (int i = count - 1; i >= 0; --i) {
      clearRegistryForDom(domObject.getChild(i));
    }
  }

  /**
   * Update the attributes according to the given attribute. Then creating a
   * command object for updating corresponding view and put the command object in the queue.
   * @param ref Reference of the dom.
   * @param attrs the new style. This style is only a part of the full attribute set, and will be
   *              merged into attributes
   * @see #updateStyle(String, JSONObject)
   */
  void updateAttrs(String ref, final JSONObject attrs) {
    if (mDestroy) {
      return;
    }
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    final WXDomObject domObject = mRegistry.get(ref);
    if (domObject == null) {
      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_ERR_DOM_UPDATEATTRS);
      }
      return;
    }

    domObject.updateAttr(attrs);

    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        mWXRenderManager.updateAttrs(mInstanceId, domObject.getRef(), attrs);
      }

      @Override
      public String toString() {
        return "updateAttr";
      }
    });
    mDirty = true;

    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  /**
   * Update styles according to the given style. Then creating a
   * command object for updating corresponding view and put the command object in the queue.
   * @param ref Reference of the dom.
   * @param style the new style. This style is only a part of the full style, and will be merged
   *              into styles
   * @see #updateAttrs(String, JSONObject)
   */
  void updateStyle(String ref, JSONObject style) {
    if (mDestroy || style == null) {
      return;
    }
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    WXDomObject domObject = mRegistry.get(ref);
    if (domObject == null) {
      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_ERR_DOM_UPDATESTYLE);
      }
      return;
    }

    Map<String, Object> animationMap= WXDataStructureUtil.newHashMapWithExpectedSize(2);
    animationMap.put(WXDomObject.TRANSFORM, style.remove(WXDomObject.TRANSFORM));
    animationMap.put(WXDomObject.TRANSFORM_ORIGIN, style.remove(WXDomObject.TRANSFORM_ORIGIN));
    animations.add(new Pair<>(ref, animationMap));

    if(!style.isEmpty()){
      domObject.updateStyle(style);
      transformStyle(domObject, false);
      updateStyle(domObject, style);
    }
    mDirty = true;

    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  /**
   * Create the command object for updating style and put it the queue. If the given style
   * contains border-width and padding that will affect layout, then a command object for reset
   * padding will also be created.
   * @param domObject the given dom object
   * @param update the given style.
   */
  private void updateStyle(final WXDomObject domObject, final Map<String, Object> update) {
    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        mWXRenderManager.updateStyle(mInstanceId, domObject.getRef(), update);
      }

      @Override
      public String toString() {
        return "updateStyle";
      }
    });
    if (update.containsKey("padding") || update.containsKey("paddingTop") ||
        update.containsKey("paddingLeft") ||
        update.containsKey("paddingRight") ||
        update.containsKey("paddingBottom") || update.containsKey("borderWidth")) {
      mNormalTasks.add(new IWXRenderTask() {

        @Override
        public void execute() {
          Spacing padding = domObject.getPadding();
          Spacing border = domObject.getBorder();
          mWXRenderManager.setPadding(mInstanceId, domObject.getRef(), padding, border);
        }

        @Override
        public String toString() {
          return "setPadding";
        }
      });
    }
  }

  /**
   * Create a command object for adding a default event listener to the corresponding {@link
   * WXDomObject} and put the command object in the queue.
   * When the event is triggered, the eventListener will call {@link WXSDKManager#fireEvent(String, String, String)}
   * , and the JS will handle all the operations from there.
   *
   * @param ref Reference of the dom.
   * @param type the type of the event, this may be a plain event defined in
   * {@link com.taobao.weex.common.Constants.Event} or a gesture defined in {@link com.taobao
   * .weex.ui.view.gesture.WXGestureType}
   */
  void addEvent(final String ref, final String type) {
    if (mDestroy) {
      return;
    }
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    WXDomObject domObject = mRegistry.get(ref);
    if (domObject == null) {
      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_ERR_DOM_ADDEVENT);
      }
      return;
    }
    domObject.addEvent(type);
    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        mWXRenderManager.addEvent(mInstanceId, ref, type);
      }

      @Override
      public String toString() {
        return "Add event";
      }
    });

    mDirty = true;
    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  /**
   * Create a command object for removing the event listener of the corresponding {@link
   * WXDomObject} and put the command event in the queue.
   * @param ref Reference of the dom.
   * @param type the type of the event, this may be a plain event defined in
   * {@link com.taobao.weex.common.Constants.Event} or a gesture defined in {@link com.taobao
   * .weex.ui.view.gesture.WXGestureType}
   */
  void removeEvent(final String ref, final String type) {
    if (mDestroy) {
      return;
    }
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    WXDomObject domObject = mRegistry.get(ref);
    if (domObject == null) {
      if (instance != null) {
        instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_ERR_DOM_REMOVEEVENT);
      }
      return;
    }
    domObject.removeEvent(type);
    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        mWXRenderManager.removeEvent(mInstanceId, ref, type);
      }

      @Override
      public String toString() {
        return "removeEvent";
      }
    });

    mDirty = true;
    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  /**
   * Create a command object for scroll the given view to the specified position.
   * @param ref Reference of the dom.
   * @param options the specified position
   */
  void scrollToDom(final String ref, final JSONObject options) {
    if (mDestroy) {
      return;
    }
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);

    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        mWXRenderManager.scrollToComponent(mInstanceId, ref, options);
      }

      @Override
      public String toString() {
        return "scrollToPosition";
      }
    });

    mDirty = true;
    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  /**
   * Create a command object for notifying {@link WXRenderManager} that the process of creating
   * given view is finished, and put the command object in the queue.
   */
  void createFinish() {
    if (mDestroy) {
      return;
    }

    final WXDomObject root = mRegistry.get(WXDomObject.ROOT);
    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        mWXRenderManager.createFinish(mInstanceId,
                                      (int) root.getLayoutWidth(),
                                      (int) root.getLayoutHeight());
      }

      @Override
      public String toString() {
        return "createFinish";
      }
    });

    mDirty = true;
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  /**
   * Create a command object for notifying {@link WXRenderManager} that the process of refreshing
   * given view is finished, and put the command object in the queue.
   */
  void refreshFinish() {
    if (mDestroy) {
      return;
    }
    final WXDomObject root = mRegistry.get(WXDomObject.ROOT);
    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        int realWidth = (int) root.getLayoutWidth();
        int realHeight = (int) root.getLayoutHeight();
        mWXRenderManager.refreshFinish(mInstanceId, realWidth, realHeight);
      }

      @Override
      public String toString() {
        return "refreshFinish";
      }
    });

    mDirty = true;
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  /**
   * Parse the jsonObject to {@link WXDomObject} recursively
   * @param map the original JSONObject
   * @return Dom Object corresponding to the JSONObject.
   */
  private @Nullable WXDomObject parseInner(JSONObject map) {
    if (map == null || map.size() <= 0) {
      return null;
    }

    String type = (String) map.get(TYPE);
    WXDomObject domObject = WXDomObjectFactory.newInstance(type);
    if(domObject == null){
      return null;
    }
    domObject.parseFromJson(map);

    Object children = map.get(CHILDREN);
    if (children != null && children instanceof JSONArray) {
      JSONArray childrenArray = (JSONArray) children;
      int count = childrenArray.size();
      for (int i = 0; i < count; ++i) {
        domObject.add(parseInner(childrenArray.getJSONObject(i)),-1);
      }
    }

    return domObject;
  }

  /**
   * Create a command object for notifying {@link WXRenderManager} that the process of update
   * given view is finished, and put the command object in the queue.
   */
  void updateFinish() {
    if (mDestroy) {
      return;
    }
    mNormalTasks.add(new IWXRenderTask() {

      @Override
      public void execute() {
        mWXRenderManager.updateFinish(mInstanceId);
      }

      @Override
      public String toString() {
        return "updateFinish";
      }
    });

    mDirty = true;
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(mInstanceId);
    if (instance != null) {
      instance.commitUTStab(IWXUserTrackAdapter.DOM_MODULE, WXErrorCode.WX_SUCCESS);
    }
  }

  void startAnimation(@NonNull final String ref, @NonNull String animation,
                      @Nullable final String callBack){
    if (mDestroy) {
      return;
    }
    WXDomObject domObject = mRegistry.get(ref);
    if (domObject == null) {
      return;
    }
    final WXAnimationBean animationBean=createAnimationBean(ref, animation);
    if(animationBean!=null) {
      mNormalTasks.add(new IWXRenderTask() {
        @Override
        public void execute() {
          mWXRenderManager.startAnimation(mInstanceId, ref, animationBean, callBack);
        }

        @Override
        public String toString() {
          return "startAnimationByCall";
        }
      });
    }
  }

  private WXAnimationBean createAnimationBean(String ref, String animation){
    try {
      WXAnimationBean animationBean =
          JSONObject.parseObject(animation, WXAnimationBean.class);
      if (animationBean != null && animationBean.styles != null) {
        WXDomObject domObject=mRegistry.get(ref);
        int width=(int)domObject.getLayoutWidth();
        int height=(int)domObject.getLayoutHeight();
        animationBean.styles.init(animationBean.styles.transformOrigin,
                                  animationBean.styles.transform,width,height);
      }
      return animationBean;
    } catch (RuntimeException e) {
      WXLogUtils.e("", e);
      return null;
    }
  }

  private WXAnimationBean createAnimationBean(String ref,Map<String, Object> style){
    if (style != null) {
      try {
        Object transform = style.get(WXDomObject.TRANSFORM);
        if (transform instanceof String && !TextUtils.isEmpty((String) transform)) {
          String transformOrigin = (String) style.get(WXDomObject.TRANSFORM_ORIGIN);
          WXAnimationBean animationBean = new WXAnimationBean();
          WXDomObject domObject = mRegistry.get(ref);
          int width = (int) domObject.getLayoutWidth();
          int height = (int) domObject.getLayoutHeight();
          animationBean.styles = new WXAnimationBean.Style();
          animationBean.styles.init(transformOrigin, (String) transform, width, height);
          return animationBean;
        }
      }catch (RuntimeException e){
        WXLogUtils.e("", e);
        return null;
      }
    }
    return null;
  }

  /**
   * Creating the mapping between Reference to {@link WXDomObject}
   * and store the mapping in {@link #mRegistry}.
   * Then, parse and copy style
   * from DOM to {@link com.taobao.weex.dom.flex.CSSNode}.
   * Finally, DOM's children are also added to
   * {@link com.taobao.weex.dom.flex.CSSNode#mChildren} if added is true.
   * The above procedure will be done recursively.
   * @param dom the original DOM Object
   * @param isAdd true for adding children of
   * {@link WXDomObject} {@link com.taobao.weex.dom.flex.CSSNode#mChildren} and parsing style,
   *              false for only parsing style.
   */
  private void transformStyle(WXDomObject dom, boolean isAdd) {
    if (dom == null) {
      return;
    }

    if (isAdd) {
      dom.young();
      mRegistry.put(dom.getRef(), dom);
    }

    WXStyle style = dom.getStyles();

    /** merge default styles **/
    Map<String,String> defaults = dom.getDefaultStyle();
    if(defaults != null){
      Iterator<Map.Entry<String,String>> it = defaults.entrySet().iterator();
      while(it.hasNext()){
        Map.Entry<String,String> entry = it.next();
        if(!style.containsKey(entry.getKey())){
          style.put(entry.getKey(),entry.getValue());
        }
      }
    }

    if (dom.getStyles().size() > 0) {
      dom.applyStyleToNode();
    }

    int count = dom.childCount();
    WXDomObject child;
    for (int i = 0; i < count; ++i) {
      child = dom.getChild(i);
      transformStyle(child, isAdd);
    }
  }

  static class AddDomInfo {

    public WXComponent component;
  }
}
