/**
 *
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "[]"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2016 Alibaba Group
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package com.taobao.weex.bridge;

import android.os.Handler;
import android.os.Handler.Callback;
import android.os.Looper;
import android.os.Message;
import android.text.TextUtils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.taobao.weex.WXEnvironment;
import com.taobao.weex.WXRenderErrorCode;
import com.taobao.weex.WXSDKInstance;
import com.taobao.weex.WXSDKManager;
import com.taobao.weex.common.IWXBridge;
import com.taobao.weex.common.WXErrorCode;
import com.taobao.weex.common.WXJSBridgeMsgType;
import com.taobao.weex.common.WXRefreshData;
import com.taobao.weex.common.WXRuntimeException;
import com.taobao.weex.common.WXThread;
import com.taobao.weex.utils.WXConst;
import com.taobao.weex.utils.WXFileUtils;
import com.taobao.weex.utils.WXHack;
import com.taobao.weex.utils.WXHack.HackDeclaration.HackAssertionException;
import com.taobao.weex.utils.WXHack.HackedClass;
import com.taobao.weex.utils.WXJsonUtils;
import com.taobao.weex.utils.WXLogUtils;
import com.taobao.weex.utils.WXViewUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Stack;
import java.util.concurrent.LinkedBlockingQueue;

/**
 * Manager class for communication between JavaScript and Android.
 * <ol>
 *   <li>
 *     Handle Android to JavaScript call, can be one of the following
 *     <ul>
 *       <li>{@link #createInstance(String, String, Map, String)}</li>
 *       <li>{@link #destroyInstance(String)}</li>
 *       <li>{@link #refreshInstance(String, WXRefreshData)}</li>
 *       <li>{@link #registerModules(Map)}</li>
 *       <li>{@link #registerComponents(ArrayList)}</li>
 *       <li>{@link #invokeCallJSBatch(Message)}</li>
 *     </ul>
 *   </li>
 *   <li>
 *     Handle JavaScript to Android call
 *   </li>
 *   <li>
 *     Handle next tick of message.
 *   </li>
 * </ol>
 */
public class WXBridgeManager implements Callback {

  public static final String METHOD_CREATE_INSTANCE = "createInstance";
  public static final String METHOD_DESTROY_INSTANCE = "destroyInstance";
  public static final String METHOD_CALL_JS = "callJS";
  public static final String METHOD_REGISTER_MODULES = "registerModules";
  public static final String METHOD_REGISTER_COMPONENTS = "registerComponents";
  public static final String METHOD_FIRE_EVENT = "fireEvent";
  public static final String METHOD_CALLBACK = "callback";
  public static final String METHOD_REFRESH_INSTANCE = "refreshInstance";
  private static final String UNDEFINED = "-1";
  private static WXBridgeManager mBridgeManager;
  /**
   * next tick tasks, can set priority
   */
  private WXHashMap<String, ArrayList<WXHashMap<String, Object>>> mNextTickTasks = new WXHashMap<>();

  /**
   * UI task (callback&&fire event)
   */
  private Queue<WXHashMap<String, Object>> mUITasks = new LinkedBlockingQueue<>();
  /**
   * JSThread
   */
  private WXThread mJSThread;
  private Handler mJSHandler;
  private IWXBridge mWXBridge;

  private boolean mMock = false;
  /**
   * Whether JS Framework(main.js) has been initialized.
   */
  private boolean mInit;

  private List<ArrayList<Map<String, String>>> mRegisterComponentFailList = new ArrayList<>(8);
  private List<Map<String, Object>> mRegisterModuleFailList = new ArrayList<>(8);

  private String mDestroyedInstanceId = "-1";

  private StringBuilder mLodBuilder = new StringBuilder(500);

  private WXBridgeManager() {
    if (WXEnvironment.sDebugMode) {
      HackedClass<Object> waBridge;
      try {
        waBridge = WXHack.into("com.taobao.weex.bridge.WXWebsocketBridge");
        mWXBridge = (IWXBridge) waBridge.constructor(
            WXBridgeManager.class)
            .getInstance(WXBridgeManager.this);
      } catch (HackAssertionException e) {
        e.printStackTrace();
      }
    } else {
      mWXBridge = new WXBridge();
    }
    mJSThread = new WXThread("JSBridgeThread", this);
    mJSHandler = mJSThread.getHandler();
  }

  public static WXBridgeManager getInstance() {
    if (mBridgeManager == null) {
      synchronized (WXBridgeManager.class) {
        if (mBridgeManager == null) {
          mBridgeManager = new WXBridgeManager();
        }
      }
    }
    return mBridgeManager;
  }

  public boolean callModuleMethod(String instanceId, String moduleStr, String methodStr, JSONArray args) {
    return WXModuleManager.callModuleMethod(instanceId, moduleStr, methodStr, args);
  }

  /**
   * Get Handler for JS Thread.
   * @return Handler the handler in JS thread.
   */
  public Handler getJSHandler() {
    return mJSHandler;
  }

  /**
   * Model switch. For now, debug model and release model are supported
   */
  public void restart() {
    mInit = false;
    if (WXEnvironment.sDebugMode) {
      HackedClass<Object> waBridge;
      try {
        waBridge = WXHack.into("com.taobao.weex.bridge.WXWebsocketBridge");
        mWXBridge = (IWXBridge) waBridge.constructor(
            WXBridgeManager.class)
            .getInstance(WXBridgeManager.this);
      } catch (HackAssertionException e) {
        e.printStackTrace();
      }
    } else {
      mWXBridge = new WXBridge();
    }
  }

  /**
   * Set current Instance
   * @param instanceId {@link WXSDKInstance#mInstanceId}
   */
  public synchronized void setStackTopInstance(final String instanceId) {
    post(new Runnable() {

      @Override
      public void run() {
        mNextTickTasks.setStackTopInstance(instanceId);
      }
    }, instanceId);
  }

  private void post(Runnable r, Object token) {
    if (mJSHandler == null) {
      return;
    }

    Message m = Message.obtain(mJSHandler, r);
    m.obj = token;
    m.sendToTarget();
  }

  void setTimeout(String callbackId, String time) {
    Message message = Message.obtain();
    message.what = WXJSBridgeMsgType.SET_TIMEOUT;
    TimerInfo timerInfo = new TimerInfo();
    timerInfo.callbackId = callbackId;
    timerInfo.time = (long) Float.parseFloat(time);
    message.obj = timerInfo;

    mJSHandler.sendMessageDelayed(message, timerInfo.time);
  }

  /**
   * Dispatch the native task to be executed.
   * @param instanceId {@link WXSDKInstance#mInstanceId}
   * @param tasks tasks to be executed
   * @param callback next tick id
   */
  void callNative(String instanceId, String tasks, String callback) {
    if (TextUtils.isEmpty(tasks)) {
      if (WXEnvironment.isApkDebugable()) {
        WXLogUtils.e("[WXBridgeManager] callNative: call Native tasks is null");
      }
      return;
    }

    if (WXEnvironment.isApkDebugable()) {
      mLodBuilder.append("[WXBridgeManager] callNative >>>> instanceId:").append(instanceId)
          .append(", tasks:").append(tasks).append(", callback:").append(callback);
      WXLogUtils.d(mLodBuilder.substring(0));
      mLodBuilder.setLength(0);
    }

    JSONArray array = JSON.parseArray(tasks);
    int size = array.size();
    if (size > 0) {
      try {
        JSONObject task;
        for (int i = 0; i < size; ++i) {
          task = (JSONObject) array.get(i);
          if (task != null && WXSDKManager.getInstance().getSDKInstance(instanceId) != null) {
            WXModuleManager.callModuleMethod(instanceId, (String) task.get("module"), (String) task.get("method"), (JSONArray) task.get("args"));
          }
        }
      } catch (Exception e) {
        WXLogUtils.e("[WXBridgeManager] callNative exception: " + WXLogUtils.getStackTrace(e));
      }
    }

    if (UNDEFINED.equals(callback)
        || mDestroyedInstanceId.equals(instanceId)) {
      return;
    }
    // get next tick
    getNextTick(instanceId, callback);
  }

  private void getNextTick(final String instanceId, final String callback) {
    addNextTickTask(instanceId, callback, "{}");
    sendMessage(instanceId, WXJSBridgeMsgType.CALL_JS_BATCH);
  }

  private void addNextTickTask(String instanceId, Object... args) {
    if (args == null || args.length == 0) {
      return;
    }

    ArrayList<Object> argsList = new ArrayList<>();
    for (Object arg : args) {
      argsList.add(arg);
    }

    WXHashMap<String, Object> task = new WXHashMap<>();
    task.put(WXConst.KEY_METHOD, METHOD_CALLBACK);
    task.put(WXConst.KEY_ARGS, args);

    if (mNextTickTasks.get(instanceId) == null) {
      ArrayList<WXHashMap<String, Object>> list = new ArrayList<WXHashMap<String, Object>>();
      list.add(task);
      mNextTickTasks.put(instanceId, list);
    } else {
      mNextTickTasks.get(instanceId).add(task);
    }

    //        mNextTickTasks.put(instanceId, task);
  }

  private void sendMessage(String instanceId, int what) {
    Message msg = Message.obtain(mJSHandler);
    msg.obj = instanceId;
    msg.what = what;
    msg.sendToTarget();
  }

  /**
   * Initialize JavaScript framework
   * @param framework String representation of the framework to be init.
   */
  public synchronized void initScriptsFramework(String framework) {
    if (!WXEnvironment.sSupport) {
      return;
    }

    Message msg = mJSHandler.obtainMessage();
    msg.obj = framework;
    msg.what = WXJSBridgeMsgType.INIT_FRAMEWORK;
    msg.setTarget(mJSHandler);
    msg.sendToTarget();
  }

  /**
   * Notify the JavaScript about the event happened on Android
   */
  public void fireEvent(final String instanceId, final String ref,
                        final String type, final Map<String, Object> data) {
    if (TextUtils.isEmpty(instanceId) || TextUtils.isEmpty(ref)
        || TextUtils.isEmpty(type) || mJSHandler == null) {
      return;
    }
    if (!checkMainThread()) {
      throw new WXRuntimeException(
          "fireEvent must be called by main thread");
    }
    addUITask(METHOD_FIRE_EVENT, instanceId, ref, type, data);
    sendMessage(instanceId, WXJSBridgeMsgType.CALL_JS_BATCH);
  }

  private boolean checkMainThread() {
    return Looper.myLooper() == Looper.getMainLooper();
  }

  private void addUITask(String method, String instanceId, Object... args) {
    ArrayList<Object> argsList = new ArrayList<>();
    for (Object arg : args) {
      argsList.add(arg);
    }

    WXHashMap<String, Object> task = new WXHashMap<>();
    task.put(WXConst.KEY_METHOD, method);
    task.put(WXConst.KEY_ARGS, argsList);
    task.setTag(instanceId);
    mUITasks.offer(task);
  }

  /**
   * Invoke JavaScript callback
   */
  public void callback(String instanceId, String callback,
                       Map<String, Object> data) {
    if (TextUtils.isEmpty(instanceId) || TextUtils.isEmpty(callback)
        || mJSHandler == null) {
      return;
    }
    callback(instanceId, callback,
             data == null ? "{}" : WXJsonUtils.fromObjectToJSONString(data));
  }

  public void callback(final String instanceId, final String callback,
                       final String data) {
    if (TextUtils.isEmpty(instanceId) || TextUtils.isEmpty(callback)
        || mJSHandler == null) {
      return;
    }

    if (!checkMainThread()) {
      throw new WXRuntimeException(
          "callback must be called by main thread");
    }
    addUITask(METHOD_CALLBACK, instanceId, callback, data);
    sendMessage(instanceId, WXJSBridgeMsgType.CALL_JS_BATCH);
  }

  /**
   * Refresh instance
   */
  public void refreshInstance(final String instanceId, final WXRefreshData jsonData) {
    if (TextUtils.isEmpty(instanceId) || jsonData == null) {
      return;
    }
    mJSHandler.postDelayed((new Runnable() {
      @Override
      public void run() {
        invokeRefreshInstance(instanceId, jsonData);
      }
    }), 0);
  }

  private void invokeRefreshInstance(String instanceId, WXRefreshData refreshData) {
    try {
      if (!mInit) {
        WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(instanceId);
        if (instance != null) {
          instance.onRenderError(WXRenderErrorCode.WX_CREATE_INSTANCE_ERROR,
                                 "createInstance failed!");
        }
        String err = "[WXBridgeManager] invokeRefreshInstance: framework.js uninitialized.";
        WXErrorCode.WX_ERR_INVOKE_NATIVE.appendErrMsg(err);
        commitJSBridgeAlarmMonitor(instanceId, WXErrorCode.WX_ERR_INVOKE_NATIVE);
        WXLogUtils.e(err);
        return;
      }
      long start = System.currentTimeMillis();
      if (WXEnvironment.isApkDebugable()) {
        WXLogUtils.d("refreshInstance >>>> instanceId:" + instanceId
                     + ", data:" + refreshData.data + ", isDirty:" + refreshData.isDirty);
      }

      if (refreshData.isDirty) {
        return;
      }
      WXJSObject instanceIdObj = new WXJSObject(WXJSObject.String,
                                                instanceId);
      WXJSObject dataObj = new WXJSObject(WXJSObject.JSON,
                                          refreshData.data == null ? "{}" : refreshData.data);
      WXJSObject[] args = {instanceIdObj, dataObj};
      mWXBridge.execJS(instanceId, null, METHOD_REFRESH_INSTANCE, args);
      WXLogUtils.renderPerformanceLog("invokeRefreshInstance", System.currentTimeMillis() - start);
      commitJSBridgeAlarmMonitor(instanceId, WXErrorCode.WX_SUCCESS);
    } catch (Throwable e) {
      String err = "[WXBridgeManager] invokeRefreshInstance " + e.getCause();
      WXErrorCode.WX_ERR_INVOKE_NATIVE.appendErrMsg(err);
      commitJSBridgeAlarmMonitor(instanceId, WXErrorCode.WX_ERR_INVOKE_NATIVE);
      WXLogUtils.e(err);
    }
  }

  public void commitJSBridgeAlarmMonitor(String instanceId, WXErrorCode errCode) {
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(instanceId);
    if (instance == null) {
      return;
    }
    instance.commitUTStab(WXConst.JS_BRIDGE, errCode);
  }

  /**
   * Create instance.
   */
  public void createInstance(final String instanceId, final String template,
                             final Map<String, Object> options, final String data) {
    if (!WXEnvironment.sSupport || TextUtils.isEmpty(instanceId)
        || TextUtils.isEmpty(template) || mJSHandler == null) {
      WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(instanceId);
      if (instance != null) {
        instance.onRenderError(WXRenderErrorCode.WX_CREATE_INSTANCE_ERROR, "createInstance fail!");
      }
      return;
    }

    post(new Runnable() {
      @Override
      public void run() {
        long start = System.currentTimeMillis();
        invokeCreateInstance(instanceId, template, options, data);
        final long totalTime = System.currentTimeMillis() - start;
        WXSDKManager.getInstance().postOnUiThread(new Runnable() {

          @Override
          public void run() {
            WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(instanceId);
            if (instance != null) {
              instance.createInstanceFinished(totalTime);
            }
          }
        }, 0);
        WXLogUtils.renderPerformanceLog("invokeCreateInstance", totalTime);
      }
    }, instanceId);
  }

  private void invokeCreateInstance(String instanceId, String template,
                                    Map<String, Object> options, String data) {
    if (mMock) {
      mock(instanceId);
    } else {
      if (!mInit) {
        WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(instanceId);
        if (instance != null) {
          instance.onRenderError(WXRenderErrorCode.WX_CREATE_INSTANCE_ERROR, "createInstance "
                                                                             + "fail!");
        }
        String err = "[WXBridgeManager] invokeCreateInstance: framework.js uninitialized.";
        WXErrorCode.WX_ERR_INVOKE_NATIVE.appendErrMsg(err);
        commitJSBridgeAlarmMonitor(instanceId, WXErrorCode.WX_ERR_INVOKE_NATIVE);
        WXLogUtils.e(err);
        return;
      }
      try {
        if (WXEnvironment.isApkDebugable()) {
          WXLogUtils.d("createInstance >>>> instanceId:" + instanceId
                       + ", options:"
                       + WXJsonUtils.fromObjectToJSONString(options)
                       + ", data:" + data);
        }
        WXJSObject instanceIdObj = new WXJSObject(WXJSObject.String,
                                                  instanceId);
        WXJSObject instanceObj = new WXJSObject(WXJSObject.String,
                                                template);
        WXJSObject optionsObj = new WXJSObject(WXJSObject.JSON,
                                               options == null ? "{}"
                                                               : WXJsonUtils.fromObjectToJSONString(options));
        WXJSObject dataObj = new WXJSObject(WXJSObject.JSON,
                                            data == null ? "{}" : data);
        WXJSObject[] args = {instanceIdObj, instanceObj, optionsObj,
            dataObj};
        mWXBridge.execJS(instanceId, null, METHOD_CREATE_INSTANCE, args);
        commitJSBridgeAlarmMonitor(instanceId, WXErrorCode.WX_SUCCESS);
      } catch (Throwable e) {
        WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(instanceId);
        if (instance != null) {
          instance.onRenderError(WXRenderErrorCode.WX_CREATE_INSTANCE_ERROR,
                                 "createInstance failed!");
        }
        String err = "[WXBridgeManager] invokeCreateInstance " + e.getCause();
        WXErrorCode.WX_ERR_INVOKE_NATIVE.appendErrMsg(err);
        commitJSBridgeAlarmMonitor(instanceId, WXErrorCode.WX_ERR_INVOKE_NATIVE);
        WXLogUtils.e(err);
      }
    }
  }

  private void mock(String instanceId) {

  }

  public void destroyInstance(final String instanceId) {
    if (!WXEnvironment.sSupport || mJSHandler == null
        || TextUtils.isEmpty(instanceId)) {
      return;
    }
    mDestroyedInstanceId = instanceId;
    // clear message with instanceId
    mJSHandler.removeCallbacksAndMessages(instanceId);
    post(new Runnable() {
      @Override
      public void run() {
        mDestroyedInstanceId = "-1";
        removeTaskByInstance(instanceId);
        invokeDestroyInstance(instanceId);
      }
    }, instanceId);
  }

  private void removeTaskByInstance(String instanceId) {
    mNextTickTasks.removeFromMapAndStack(instanceId);
  }

  private void invokeDestroyInstance(String instanceId) {
    try {
      if (WXEnvironment.isApkDebugable()) {
        WXLogUtils.d("destroyInstance >>>> instanceId:" + instanceId);
      }
      WXJSObject instanceIdObj = new WXJSObject(WXJSObject.String,
                                                instanceId);
      WXJSObject[] args = {instanceIdObj};
      mWXBridge.execJS(instanceId, null, METHOD_DESTROY_INSTANCE, args);
      commitJSBridgeAlarmMonitor(instanceId, WXErrorCode.WX_SUCCESS);
    } catch (Throwable e) {
      String err = "[WXBridgeManager] invokeDestroyInstance " + e.getCause();
      WXErrorCode.WX_ERR_INVOKE_NATIVE.appendErrMsg(err);
      commitJSBridgeAlarmMonitor(instanceId, WXErrorCode.WX_ERR_INVOKE_NATIVE);
      WXLogUtils.e(err);
    }
  }

  @Override
  public boolean handleMessage(Message msg) {
    if (msg == null) {
      return false;
    }

    int what = msg.what;
    switch (what) {
      case WXJSBridgeMsgType.INIT_FRAMEWORK:
        invokeInitFramework(msg);
        break;
      case WXJSBridgeMsgType.CALL_JS_BATCH:
        invokeCallJSBatch(msg);
        break;
      case WXJSBridgeMsgType.SET_TIMEOUT:
        TimerInfo timerInfo = (TimerInfo) msg.obj;
        WXJSObject obj = new WXJSObject(WXJSObject.String, timerInfo.callbackId);
        WXJSObject[] args = {obj};
        mWXBridge.execJS("", null, "setTimeoutCallback", args);
        break;
      default:
        break;
    }
    return false;
  }

  private void invokeInitFramework(Message msg) {
    String framework = "";
    if (msg.obj != null) {
      framework = (String) msg.obj;
    }

    if (!mInit) {
      if (TextUtils.isEmpty(framework)) {
        if (WXEnvironment.isApkDebugable()) {
          WXLogUtils.d("framework from assets");
        }
        framework = WXFileUtils.loadFileContent("main.js", WXEnvironment.getApplication());
      }
      if (TextUtils.isEmpty(framework)) {
        mInit = false;
        return;
      }
      try {
        long start = System.currentTimeMillis();
        mWXBridge.initFramework(framework, assembleDefaultOptions());
        WXEnvironment.sJSLibInitTime = System.currentTimeMillis() - start;
        WXLogUtils.renderPerformanceLog("initFramework", WXEnvironment.sJSLibInitTime);
        mInit = true;
        execRegisterFailTask();
        WXEnvironment.JsFrameworkInit = true;
      } catch (Throwable e) {
        WXLogUtils.e("[WXBridgeManager] invokeInitFramework " + e.getCause());
      }
    }
  }

  @SuppressWarnings("unchecked")
  private void invokeCallJSBatch(Message message) {
    if ((mNextTickTasks.isEmpty() && mUITasks.isEmpty()) || !mInit) {
      if (!mInit) {
        WXLogUtils.e("[WXBridgeManager] invokeCallJSBatch: framework.js uninitialized.");
      }
      return;
    }
    try {
      Object task = null;

      Object instanceId = message.obj;
      // Execute UI task first
      task = mUITasks.poll();
      if (task != null) {
        instanceId = ((WXHashMap) task).getTag();
      }

      Object[] tasks = {task};

      //If UI task is empty, then next tick is executed
      if (task == null) {
        Stack<String> instanceStack = mNextTickTasks.getInstanceStack();
        int size = instanceStack.size();
        for (int i = size - 1; i >= 0; i--) {
          // Instance on the top of stack. The corresponding message of that instance will be
          // handled first
          instanceId = instanceStack.get(i);
          task = mNextTickTasks.remove(instanceId);
          if (task != null && !((ArrayList) task).isEmpty()) {
            break;
          }
        }
        tasks = ((ArrayList) task).toArray();
      }

      task = tasks;

      if (WXEnvironment.isApkDebugable()) {
        mLodBuilder.append("callJS >>>> instanceId:").append(instanceId).append(" tasks:").append(WXJsonUtils.fromObjectToJSONString(task));
        WXLogUtils.d(mLodBuilder.substring(0));
        mLodBuilder.setLength(0);
      }
      WXJSObject[] args = {
          new WXJSObject(WXJSObject.String, instanceId),
          new WXJSObject(WXJSObject.JSON,
                         WXJsonUtils.fromObjectToJSONString(task))};
      if (WXEnvironment.isApkDebugable()) {
        String ins = String.valueOf(instanceId);
        if (TextUtils.isEmpty(ins) || ins.equals("null") || args == null) {
          WXLogUtils.e("WXBridgeManager (TextUtils.isEmpty(ins) || ins.equals(\"null\") || args == null)");
        }
      }
      mWXBridge.execJS(String.valueOf(instanceId), null, METHOD_CALL_JS,
                       args);
      // If task is not empty, loop until it is empty
      if (!mNextTickTasks.isEmpty() && !mUITasks.isEmpty()) {
        mJSHandler.sendEmptyMessage(WXJSBridgeMsgType.CALL_JS_BATCH);
      }
    } catch (Throwable e) {
      WXLogUtils.e("WXBridgeManager" + e.getMessage());
    }
  }

  private WXParams assembleDefaultOptions() {
    Map<String, String> config = WXEnvironment.getConfig();
    WXParams wxParams = new WXParams();
    wxParams.setPlatform(config.get("os"));
    wxParams.setOsVersion(config.get("sysVersion"));
    wxParams.setAppVersion(config.get("appVersion"));
    wxParams.setWeexVersion(config.get("weexVersion"));
    wxParams.setDeviceModel(config.get("sysModel"));
    wxParams.setShouldInfoCollect(config.get("infoCollect"));
    String appName = config.get("appName");
    if (!TextUtils.isEmpty(appName)) {
      wxParams.setAppName(appName);
    }
    wxParams.setDeviceWidth(TextUtils.isEmpty(config.get("deviceWidth")) ? String.valueOf(WXViewUtils.getScreenWidth()) : config.get("deviceWidth"));
    wxParams.setDeviceHeight(TextUtils.isEmpty(config.get("deviceHeight")) ? String.valueOf(WXViewUtils.getScreenHeight()) : config.get("deviceHeight"));
    return wxParams;
  }

  private void execRegisterFailTask() {
    int moduleCount = mRegisterModuleFailList.size();
    if (moduleCount > 0) {
      for (int i = 0; i < moduleCount; ++i) {
        registerModules(mRegisterModuleFailList.get(i));
      }
      mRegisterModuleFailList.clear();
      WXLogUtils.e("[WXBridgeManager] execRegisterFailTask register module fail");
    }
    int componentCount = mRegisterComponentFailList.size();
    if (componentCount > 0) {
      for (int i = 0; i < componentCount; ++i) {
        registerComponents(mRegisterComponentFailList.get(i));
      }
      mRegisterComponentFailList.clear();
      WXLogUtils.e("[WXBridgeManager] execRegisterFailTask register component fail");
    }
  }

  /**
   * Register Android module
   * @param modules the format is like
   *                {'dom':['updateAttrs','updateStyle'],'event':['openUrl']}
   */

  public void registerModules(final Map<String, Object> modules) {
    if (!WXEnvironment.sSupport || mJSHandler == null || modules == null
        || modules.size() == 0) {
      return;
    }
    post(new Runnable() {
      @Override
      public void run() {
        invokeRegisterModules(modules);
      }
    }, null);
  }

  /**
   * Registered component
   */
  public void registerComponents(final ArrayList<Map<String, String>> components) {
    if (!WXEnvironment.sSupport || mJSHandler == null || components == null
        || components.size() == 0) {
      return;
    }
    post(new Runnable() {
      @Override
      public void run() {
        invokeRegisterComponents(components);
      }
    }, null);
  }

  private void invokeRegisterModules(Map<String, Object> modules) {
    if (modules == null || !mInit) {
      if (!mInit) {
        WXLogUtils.e("[WXBridgeManager] invokeCallJSBatch: framework.js uninitialized.");
      }
      mRegisterModuleFailList.add(modules);
      return;
    }

    WXJSObject[] args = {new WXJSObject(WXJSObject.JSON,
                                        WXJsonUtils.fromObjectToJSONString(modules))};
    try {
      mWXBridge.execJS("", null, METHOD_REGISTER_MODULES, args);
    } catch (Throwable e) {
      WXLogUtils.e("[WXBridgeManager] invokeRegisterModules:" + (e == null ? "" : e.getStackTrace()));
    }
  }

  private void invokeRegisterComponents(ArrayList<Map<String, String>> components) {
    if (components == null || !mInit) {
      if (!mInit) {
        WXLogUtils.e("[WXBridgeManager] invokeCallJSBatch: framework.js uninitialized.");
      }
      mRegisterComponentFailList.add(components);
      return;
    }

    WXJSObject[] args = {new WXJSObject(WXJSObject.JSON,
                                        WXJsonUtils.fromObjectToJSONString(components))};
    try {
      mWXBridge.execJS("", null, METHOD_REGISTER_COMPONENTS, args);
    } catch (Throwable e) {
      WXLogUtils.e("[WXBridgeManager] invokeRegisterComponents " + (e == null ? "" : e.getCause()));
    }
  }

  public void destroy() {
    if (mJSThread != null) {
      mJSThread.quit();
    }
    mBridgeManager = null;
  }

  /**
   * Report JavaScript Exception
   */
  public void reportJSException(String instanceId, String function,
                                String exception) {
    if (WXEnvironment.isApkDebugable()) {
      WXLogUtils.e("reportJSException >>>> instanceId:" + instanceId
                   + ", exception function:" + function + ", exception:"
                   + exception);
    }
    WXSDKInstance instance = WXSDKManager.getInstance().getSDKInstance(
        instanceId);
    if (instance != null) {
      // TODO add errCode
      instance.onJSException(null, function, exception);
    }
    WXErrorCode.WX_ERR_JS_EXECUTE.appendErrMsg(exception);
    commitJSBridgeAlarmMonitor(instanceId, WXErrorCode.WX_ERR_JS_EXECUTE);
  }

  public static class TimerInfo {

    public String callbackId;
    public long time;
    public String instanceId;
  }

}
