/**
 *
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "[]"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2016 Alibaba Group
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package com.taobao.weex.dom;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.taobao.weex.WXEnvironment;
import com.taobao.weex.WXSDKInstance;
import com.taobao.weex.WXSDKManager;
import com.taobao.weex.bridge.WXValidateProcessor;
import com.taobao.weex.common.Constants;
import com.taobao.weex.dom.flex.CSSLayoutContext;
import com.taobao.weex.dom.flex.CSSNode;
import com.taobao.weex.dom.flex.Spacing;
import com.taobao.weex.ui.component.WXBasicComponentType;
import com.taobao.weex.utils.WXLogUtils;
import com.taobao.weex.utils.WXViewUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * WXDomObject contains all the info about the given node, including style, attribute and event.
 * Unlike {@link com.taobao.weex.ui.component.WXComponent}, WXDomObject only contains info about
 * the dom, has nothing to do with rendering.
 * Actually, {@link com.taobao.weex.ui.component.WXComponent} hold references to
 * {@link android.view.View} and {@link WXDomObject}.
 */
public class WXDomObject extends CSSNode implements Cloneable,ImmutableDomObject {
  public static final String CHILDREN = "children";
  public static final String TYPE = "type";
  public static final String TAG = WXDomObject.class.getSimpleName();
  public static final String ROOT = "_root";
  public static final String TRANSFORM = "transform";
  public static final String TRANSFORM_ORIGIN = "transformOrigin";
  static final WXDomObject DESTROYED = new WXDomObject();
  static{
    DESTROYED.mRef = "_destroyed";
  }
  private AtomicBoolean sDestroy = new AtomicBoolean();

  private int mViewPortWidth =750;

  private DomContext mDomContext;

  /** package **/ String mRef = ROOT;

  /** package **/ String mType = WXBasicComponentType.DIV;

  /** package **/ WXStyle mStyles;

  /** package **/ WXAttr mAttributes;

  /** package **/ WXEvent mEvents;

  private List<WXDomObject> mDomChildren;

  /** Do not access this field directly. This field will be removed soon. **/
  @Deprecated
  public WXDomObject parent;

  private ArrayList<String> fixedStyleRefs;

  private boolean mYoung = false;

  /** package **/ void traverseTree(Consumer...consumers){
    if (consumers == null) {
      return;
    }

    for (Consumer consumer:consumers){
      consumer.accept(this);
    }

    int count = childCount();
    WXDomObject child;
    for (int i = 0; i < count; ++i) {
      child = getChild(i);
      child.traverseTree(consumers);
    }
  }


  public int getViewPortWidth() {
    return mViewPortWidth;
  }

  public void setViewPortWidth(int mViewPortWidth) {
    this.mViewPortWidth = mViewPortWidth;
  }

  public String getRef(){
    return mRef;
  }

  public String getType(){
    return mType;
  }

  public @NonNull WXStyle getStyles(){
    if(mStyles == null ){
      mStyles = new WXStyle();
    }
    return mStyles;
  }

  public @NonNull WXAttr getAttrs(){
    if(mAttributes == null){
      mAttributes = new WXAttr();
    }
    return mAttributes;
  }

  public @NonNull WXEvent getEvents(){
    if(mEvents == null){
      mEvents = new WXEvent();
    }

    return mEvents;
  }

  public @NonNull DomContext getDomContext() {
    return mDomContext;
  }

  public void clearEvents(){
    if(mEvents != null){
      mEvents.clear();
    }
  }

  public static void prepareRoot(WXDomObject domObj,float defaultHeight,float defaultWidth) {
    domObj.mRef = WXDomObject.ROOT;

    WXStyle domStyles = domObj.getStyles();
    Map<String, Object> style = new HashMap<>(5);
    if (!domStyles.containsKey(Constants.Name.FLEX_DIRECTION)) {
      style.put(Constants.Name.FLEX_DIRECTION, "column");
    }
    if (!domStyles.containsKey(Constants.Name.BACKGROUND_COLOR)) {
      style.put(Constants.Name.BACKGROUND_COLOR, "#ffffff");
    }

    style.put(Constants.Name.DEFAULT_WIDTH, defaultWidth);
    style.put(Constants.Name.DEFAULT_HEIGHT, defaultHeight);

    domObj.updateStyle(style);
  }

  protected final void copyFields(WXDomObject dest) {
    dest.cssstyle.copy(this.cssstyle);
    dest.mRef = mRef;
    dest.mType = mType;
    dest.mStyles = mStyles == null ? null : mStyles.clone();//mStyles == null ? null : mStyles.clone();
    dest.mAttributes = mAttributes == null ? null : mAttributes.clone();//mAttrs == null ? null : mAttrs.clone();
    dest.mEvents = mEvents == null ? null : mEvents.clone();
    dest.csslayout.copy(this.csslayout);
  }

  /**
   * Parse the jsonObject to {@link WXDomObject} recursively
   * @param map the original JSONObject
   */
  public void parseFromJson(JSONObject map){
    if (map == null || map.size() <= 0) {
      return;
    }

    String type = (String) map.get("type");
    this.mType = type;
    this.mRef = (String) map.get("ref");
    Object style = map.get("style");
    if (style != null && style instanceof JSONObject) {
      WXStyle styles = new WXStyle();
      styles.putAll((JSONObject) style,false);
      this.mStyles = styles;
    }
    Object attr = map.get("attr");
    if (attr != null && attr instanceof JSONObject) {
      WXAttr attrs = new WXAttr((JSONObject) attr);
      //WXJsonUtils.putAll(attrs, (JSONObject) attr);
      this.mAttributes = attrs;
    }
    Object event = map.get("event");
    if (event != null && event instanceof JSONArray) {
      WXEvent events = new WXEvent();
      JSONArray eventArray = (JSONArray) event;
      int count = eventArray.size();
      for (int i = 0; i < count; ++i) {
        events.add(eventArray.getString(i));
      }
      this.mEvents = events;
    }

  }


  /**
   * Do pre-staff before layout. Subclass may provide different implementation.
   */
  public void layoutBefore() {

  }

  /**
   * Do post-staff before layout. Subclass may provide different implementation.
   */
  public void layoutAfter(){

  }

  /**
   * Tell whether this object need to be updated. This is usually called when
   * {@link CSSNode#calculateLayout(CSSLayoutContext)} finishes and new layout has been
   * calculated. This method is a simple wrapper method for {@link #hasNewLayout()} and
   * {@link #isDirty()}.
   * @return true for need update since last update.
   */
  public final boolean hasUpdate() {
    return hasNewLayout() || isDirty();
  }

  /**
   * Mark the current node is young and unconsumed.
   */
  void young() {
    mYoung = true;
  }

  /**
   * Mark the current node is old and consumed.
   */
  void old() {
    mYoung = false;
  }

  /**
   * Tell whether this node is consumed since last layout.
   * @return true for consumed, false for not.
   */
  boolean isYoung() {
    return mYoung;
  }

  /**
   * Mark the update has been seen. After this method call, following call for {@link
   * #hasUpdate()} will return false. This method is also a wrapper for {@link #markUpdateSeen()}
   */
  public final void markUpdateSeen() {
    if (hasNewLayout()) {
      markLayoutSeen();
    }
  }

  public boolean isFixed() {
    return mStyles == null ? false : mStyles.isFixed();
  }

  public boolean canRecycled() {
    return mAttributes == null ? false : mAttributes.canRecycled();
  }

  public Object getExtra() {
    return null;
  }

  public void remove(WXDomObject child) {
    if (child == null || mDomChildren == null || sDestroy.get()) {
      return;
    }

    int index = mDomChildren.indexOf(child);
    removeFromDom(child);
    if (index != -1) {
      super.removeChildAt(index);
    }
  }

  public void removeFromDom(WXDomObject child) {
    if (child == null || mDomChildren == null || sDestroy.get()) {
      return;
    }

    int index = mDomChildren.indexOf(child);
    if (index == -1) {
      if (WXEnvironment.isApkDebugable()) {
        WXLogUtils.e("[WXDomObject] remove function error");
      }
      return;
    }
    mDomChildren.remove(index).parent = null;
  }

  public int index(WXDomObject child) {
    if (child == null || mDomChildren == null || sDestroy.get()) {
      return -1;
    }
    return mDomChildren.indexOf(child);
  }

  /**
   * Add the given WXDomObject as this object's child at specified index.
   * @param child the child to be added
   * @param index the index of child to be added. If the index is -1, the child will be added
   *              as the last child of current dom object.
   */
  public void add(WXDomObject child, int index) {
    if (child == null || index < -1 || sDestroy.get()) {
      return;
    }
    if (mDomChildren == null) {
      mDomChildren = new ArrayList<>();
    }

    int count = mDomChildren.size();
    index = index >= count ? -1 : index;
    if (index == -1) {
      mDomChildren.add(child);
      super.addChildAt(child, super.getChildCount());
    } else {
      mDomChildren.add(index, child);
      super.addChildAt(child, index);
    }
    child.parent = this;
  }

  @Deprecated
  public void add2Dom(WXDomObject child, int index) {
    if (child == null || index < -1 || sDestroy.get()) {
      return;
    }

    int count = super.getChildCount();
    index = index >= count ? -1 : index;
    if (index == -1) {
      super.addChildAt(child, super.getChildCount());
    } else {
      super.addChildAt(child, index);
    }
    child.parent = this;
  }

  public WXDomObject getChild(int index) {
    if (mDomChildren == null || sDestroy.get()) {
      return null;
    }
    return mDomChildren.get(index);
  }

  /**
   * Add the given event for current object.
   * @param e
   */
  public void addEvent(String e) {
    if (TextUtils.isEmpty(e)) {
      return;
    }
    if (mEvents == null) {
      mEvents = new WXEvent();
    }
    if (containsEvent(e)) {
      return;
    }
    mEvents.add(e);
  }

  public boolean containsEvent(String e) {
    if (mEvents == null) {
      return false;
    }
    return mEvents.contains(e);
  }

  public void removeEvent(String e) {
    if (TextUtils.isEmpty(e)) {
      return;
    }
    if (mEvents == null) {
      return;
    }
    mEvents.remove(e);
  }

  public void updateAttr(Map<String, Object> attrs) {
    if (attrs == null || attrs.isEmpty()) {
      return;
    }
    if (mAttributes == null) {
      mAttributes = new WXAttr();
    }
    mAttributes.putAll(attrs);
    super.dirty();
  }

  public void updateStyle(Map<String, Object> styles){
    updateStyle(styles,false);
  }

  public void updateStyle(Map<String, Object> styles, boolean byPesudo) {
    if (styles == null || styles.isEmpty()) {
      return;
    }
    if (mStyles == null) {
      mStyles = new WXStyle();
    }
    mStyles.putAll(styles,byPesudo);
    super.dirty();
  }

  /** package **/ void applyStyleToNode() {
    WXStyle stylesMap = getStyles();
    if (!stylesMap.isEmpty()) {
      for(Map.Entry<String,Object> item:stylesMap.entrySet()) {
        switch (item.getKey()) {
          case Constants.Name.ALIGN_ITEMS:
            setAlignItems(stylesMap.getAlignItems());
            break;
          case Constants.Name.ALIGN_SELF:
            setAlignSelf(stylesMap.getAlignSelf());
            break;
          case Constants.Name.FLEX:
            setFlex(stylesMap.getFlex());
            break;
          case Constants.Name.FLEX_DIRECTION:
            setFlexDirection(stylesMap.getFlexDirection());
            break;
          case Constants.Name.JUSTIFY_CONTENT:
            setJustifyContent(stylesMap.getJustifyContent());
            break;
          case Constants.Name.FLEX_WRAP:
            setWrap(stylesMap.getCSSWrap());
            break;
          case Constants.Name.MIN_WIDTH:
            setMinWidth(WXViewUtils.getRealPxByWidth(stylesMap.getMinWidth(),getViewPortWidth()));
            break;
          case Constants.Name.MIN_HEIGHT:
            setMinHeight(WXViewUtils.getRealPxByWidth(stylesMap.getMinHeight(),getViewPortWidth()));
            break;
          case Constants.Name.MAX_WIDTH:
            setMaxWidth(WXViewUtils.getRealPxByWidth(stylesMap.getMaxWidth(),getViewPortWidth()));
            break;
          case Constants.Name.MAX_HEIGHT:
            setMaxHeight(WXViewUtils.getRealPxByWidth(stylesMap.getMaxHeight(),getViewPortWidth()));
            break;
          case Constants.Name.DEFAULT_HEIGHT:
          case Constants.Name.HEIGHT:
            setStyleHeight(WXViewUtils.getRealPxByWidth(stylesMap.containsKey(Constants.Name.HEIGHT)?stylesMap.getHeight():stylesMap.getDefaultHeight(),getViewPortWidth()));
            break;
          case Constants.Name.WIDTH:
          case Constants.Name.DEFAULT_WIDTH:
            setStyleWidth(WXViewUtils.getRealPxByWidth(stylesMap.containsKey(Constants.Name.WIDTH)?stylesMap.getWidth():stylesMap.getDefaultWidth(),getViewPortWidth()));
            break;
          case Constants.Name.POSITION:
            setPositionType(stylesMap.getPosition());
            break;
          case Constants.Name.LEFT:
            setPositionLeft(WXViewUtils.getRealPxByWidth(stylesMap.getLeft(),getViewPortWidth()));
            break;
          case Constants.Name.TOP:
            setPositionTop(WXViewUtils.getRealPxByWidth(stylesMap.getTop(),getViewPortWidth()));
            break;
          case Constants.Name.RIGHT:
            setPositionRight(WXViewUtils.getRealPxByWidth(stylesMap.getRight(),getViewPortWidth()));
            break;
          case Constants.Name.BOTTOM:
            setPositionBottom(WXViewUtils.getRealPxByWidth(stylesMap.getBottom(),getViewPortWidth()));
            break;
          case Constants.Name.MARGIN:
            setMargin(Spacing.ALL, WXViewUtils.getRealPxByWidth(stylesMap.getMargin(),getViewPortWidth()));
            break;
          case Constants.Name.MARGIN_LEFT:
            setMargin(Spacing.LEFT, WXViewUtils.getRealPxByWidth(stylesMap.getMarginLeft(),getViewPortWidth()));
            break;
          case Constants.Name.MARGIN_TOP:
            setMargin(Spacing.TOP, WXViewUtils.getRealPxByWidth(stylesMap.getMarginTop(),getViewPortWidth()));
            break;
          case Constants.Name.MARGIN_RIGHT:
            setMargin(Spacing.RIGHT, WXViewUtils.getRealPxByWidth(stylesMap.getMarginRight(),getViewPortWidth()));
            break;
          case Constants.Name.MARGIN_BOTTOM:
            setMargin(Spacing.BOTTOM, WXViewUtils.getRealPxByWidth(stylesMap.getMarginBottom(),getViewPortWidth()));
            break;
          case Constants.Name.BORDER_WIDTH:
            setBorder(Spacing.ALL, WXViewUtils.getRealPxByWidth(stylesMap.getBorderWidth(),getViewPortWidth()));
            break;
          case Constants.Name.BORDER_TOP_WIDTH:
            setBorder(Spacing.TOP, WXViewUtils.getRealPxByWidth(stylesMap.getBorderTopWidth(),getViewPortWidth()));
            break;
          case Constants.Name.BORDER_RIGHT_WIDTH:
            setBorder(Spacing.RIGHT, WXViewUtils.getRealPxByWidth(stylesMap.getBorderRightWidth(),getViewPortWidth()));
            break;
          case Constants.Name.BORDER_BOTTOM_WIDTH:
            setBorder(Spacing.BOTTOM, WXViewUtils.getRealPxByWidth(stylesMap.getBorderBottomWidth(),getViewPortWidth()));
            break;
          case Constants.Name.BORDER_LEFT_WIDTH:
            setBorder(Spacing.LEFT, WXViewUtils.getRealPxByWidth(stylesMap.getBorderLeftWidth(),getViewPortWidth()));
            break;
          case Constants.Name.PADDING:
            setPadding(Spacing.ALL, WXViewUtils.getRealPxByWidth(stylesMap.getPadding(),getViewPortWidth()));
            break;
          case Constants.Name.PADDING_LEFT:
            setPadding(Spacing.LEFT, WXViewUtils.getRealPxByWidth(stylesMap.getPaddingLeft(),getViewPortWidth()));
            break;
          case Constants.Name.PADDING_TOP:
            setPadding(Spacing.TOP, WXViewUtils.getRealPxByWidth(stylesMap.getPaddingTop(),getViewPortWidth()));
            break;
          case Constants.Name.PADDING_RIGHT:
            setPadding(Spacing.RIGHT, WXViewUtils.getRealPxByWidth(stylesMap.getPaddingRight(),getViewPortWidth()));
            break;
          case Constants.Name.PADDING_BOTTOM:
            setPadding(Spacing.BOTTOM, WXViewUtils.getRealPxByWidth(stylesMap.getPaddingBottom(),getViewPortWidth()));
            break;
        }
      }
    }
  }

  public int childCount() {
    return mDomChildren == null ? 0 : mDomChildren.size();
  }

  public void hide() {
    setVisible(false);
  }

  public void show() {
    setVisible(true);
  }

  public boolean isVisible() {
    return super.isShow();
  }

  /**
   * Clone the current object. This is not a deep copy, only shadow copy of some reference.
   * @return The result object of clone.
   */
  @Override
  public WXDomObject clone() {
    if (sDestroy.get()) {
      return null;
    }
    WXDomObject dom = null;
    try {
      dom = WXDomObjectFactory.newInstance(mType);
      copyFields(dom);
    } catch (Exception e) {
      if (WXEnvironment.isApkDebugable()) {
        WXLogUtils.e("WXDomObject clone error: ", e);
      }
    }

    return dom;
  }

  public void destroy() {
    sDestroy.set(true);
    if (mStyles != null) {
      mStyles.clear();
    }
    if (mAttributes != null) {
      mAttributes.clear();
    }
    if (mEvents != null) {
      mEvents.clear();
    }
    if (mDomChildren != null) {
      int count = mDomChildren.size();
      for (int i = 0; i < count; ++i) {
        mDomChildren.get(i).destroy();
      }
      mDomChildren.clear();
    }
    mDomContext = null;
  }

  /** package **/
  /**
   * Get default style map for component.
   * @return
   */
  protected Map<String,String> getDefaultStyle(){
    return null;
  }

  public ArrayList<String> getFixedStyleRefs() {
    return fixedStyleRefs;
  }

  public void add2FixedDomList(String ref) {
    if (fixedStyleRefs == null) {
      fixedStyleRefs = new ArrayList<>();
    }
    fixedStyleRefs.add(ref);
  }

  public String dumpDomTree() {
    return mRef + ": " + toString();
  }

  /**
   * Parse the jsonObject to {@link WXDomObject} recursively
   * @param json the original JSONObject
   * @return Dom Object corresponding to the JSONObject.
   */
  public static  @Nullable WXDomObject parse(JSONObject json, WXSDKInstance wxsdkInstance){
      if (json == null || json.size() <= 0) {
        return null;
      }

      String type = (String) json.get(TYPE);

      if (wxsdkInstance.isNeedValidate()) {
        WXValidateProcessor processor = WXSDKManager.getInstance()
                .getValidateProcessor();
        if (processor != null) {
          WXValidateProcessor.WXComponentValidateResult result = processor
                  .onComponentValidate(wxsdkInstance, type);
          if (result != null && !result.isSuccess) {
            type = TextUtils.isEmpty(result.replacedComponent) ? WXBasicComponentType.DIV
                    : result.replacedComponent;
            json.put(TYPE, type);
            if(WXEnvironment.isApkDebugable()&&result.validateInfo!=null){
              String tag = "[WXDomObject]onComponentValidate failure. >>> "+result.validateInfo.toJSONString();
              WXLogUtils.e(tag);
            }
          }
        }
      }

      WXDomObject domObject = WXDomObjectFactory.newInstance(type);

      domObject.setViewPortWidth(wxsdkInstance.getInstanceViewPortWidth());

      if(domObject == null){
        return null;
      }
      domObject.parseFromJson(json);
      domObject.mDomContext = wxsdkInstance;

      Object children = json.get(CHILDREN);
      if (children != null && children instanceof JSONArray) {
        JSONArray childrenArray = (JSONArray) children;
        int count = childrenArray.size();
        for (int i = 0; i < count; ++i) {
          domObject.add(parse(childrenArray.getJSONObject(i),wxsdkInstance),-1);
        }
      }

      return domObject;
  }

  interface Consumer{
    void accept(WXDomObject dom);
  }
}
