/**
 *
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "[]"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2016 Alibaba Group
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package com.taobao.weex.dom;

import android.os.Message;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.taobao.weex.WXSDKInstance;
import com.taobao.weex.WXSDKManager;
import com.taobao.weex.bridge.SimpleJSCallback;
import com.taobao.weex.bridge.WXBridgeManager;
import com.taobao.weex.common.WXModule;
import com.taobao.weex.utils.WXLogUtils;
import com.taobao.weex.utils.WXViewUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * <p>
 * Module class for dom operation. Methods in this class will run in dom thread by default.
 * Actually, methods in this class are wrapper classes, they just wrap method call info, and hand
 * the wrapped info to the {@link WXDomHandler} for further process. This class is also singleton
 * in the {@link com.taobao.weex.WXSDKInstance}
 * </p>
 * <p>
 *   This module is work different with other regular module, method is invoked directly, without reflection.
 * </p>
 */
public final class WXDomModule extends WXModule {

  /** package **/
  // method
  static final String CREATE_BODY = "createBody";
  static final String UPDATE_ATTRS = "updateAttrs";
  static final String UPDATE_STYLE = "updateStyle";
  static final String REMOVE_ELEMENT = "removeElement";
  static final String ADD_ELEMENT = "addElement";
  static final String MOVE_ELEMENT = "moveElement";
  static final String ADD_EVENT = "addEvent";
  static final String REMOVE_EVENT = "removeEvent";
  static final String CREATE_FINISH = "createFinish";
  static final String REFRESH_FINISH = "refreshFinish";
  static final String UPDATE_FINISH = "updateFinish";
  static final String SCROLL_TO_ELEMENT = "scrollToElement";
  static final String ADD_RULE = "addRule";
  static final String GET_COMPONENT_RECT = "getComponentRect";

  public static final String WXDOM = "dom";


  public static final String INVOKE_METHOD = "invokeMethod";
  /**
   * Methods expose to js. Every method which will be called in js should add to this array.
   */
  public static final String[] METHODS = {CREATE_BODY, UPDATE_ATTRS, UPDATE_STYLE,
      REMOVE_ELEMENT, ADD_ELEMENT, MOVE_ELEMENT, ADD_EVENT, REMOVE_EVENT, CREATE_FINISH,
      REFRESH_FINISH, UPDATE_FINISH, SCROLL_TO_ELEMENT, ADD_RULE,GET_COMPONENT_RECT,
      INVOKE_METHOD};

  public WXDomModule(WXSDKInstance instance){
    mWXSDKInstance = instance;
  }

  public void callDomMethod(JSONObject task) {
    if (task == null) {
      return;
    }
    String method = (String) task.get(WXBridgeManager.METHOD);
    JSONArray args = (JSONArray) task.get(WXBridgeManager.ARGS);
    callDomMethod(method,args);
  }
  
  public Object callDomMethod(String method, JSONArray args) {

    if (method == null) {
      return null;
    }
    try {
      switch (method) {
        case CREATE_BODY:
          if (args == null) {
            return null;
          }
          createBody((JSONObject) args.get(0));
          break;
        case UPDATE_ATTRS:
          if (args == null) {
            return null;
          }
          updateAttrs((String) args.get(0), (JSONObject) args.get(1));
          break;
        case UPDATE_STYLE:
          if (args == null) {
            return null;
          }
          updateStyle((String) args.get(0), (JSONObject) args.get(1));
          break;
        case REMOVE_ELEMENT:
          if (args == null) {
            return null;
          }
          removeElement((String) args.get(0));
          break;
        case ADD_ELEMENT:
          if (args == null) {
            return null;
          }
          addElement((String) args.get(0), (JSONObject) args.get(1), (Integer) args.get(2));
          break;
        case MOVE_ELEMENT:
          if (args == null) {
            return null;
          }
          moveElement((String) args.get(0), (String) args.get(1), (Integer) args.get(2));
          break;
        case ADD_EVENT:
          if (args == null) {
            return null;
          }
          addEvent((String) args.get(0), (String) args.get(1));
          break;
        case REMOVE_EVENT:
          if (args == null) {
            return null;
          }
          removeEvent((String) args.get(0), (String) args.get(1));
          break;
        case CREATE_FINISH:
          createFinish();
          break;
        case REFRESH_FINISH:
          refreshFinish();
          break;
        case UPDATE_FINISH:
          updateFinish();
          break;
        case SCROLL_TO_ELEMENT:
          if (args == null) {
            return null;
          }
          JSONObject option =null;
          if(args.size()>1) {
            option = (JSONObject) args.get(1);
          }
          scrollToElement((String) args.get(0),option);
          break;
        case ADD_RULE:
          if (args == null) {
            return null;
          }
          addRule((String) args.get(0), (JSONObject) args.get(1));
          break;
        case GET_COMPONENT_RECT:
          if(args == null){
            return null;
          }
          getComponentRect((String) args.get(0),(String) args.get(1));
          break;
        case INVOKE_METHOD:
          if(args == null){
            return null;
          }
          invokeMethod((String) args.get(0),(String) args.get(1),(JSONArray) args.get(2));
          break;
      }

    } catch (IndexOutOfBoundsException e) {
      // no enougn args
      e.printStackTrace();
      WXLogUtils.e("Dom module call miss arguments.");
    } catch (ClassCastException cce) {
      WXLogUtils.e("Dom module call arguments format error!!");
    }
    return null;
  }

  /**
   * invoke dom method
   * @param ref
   * @param method
   * @param args
   */
  public void invokeMethod(String ref, String method, JSONArray args){
    if(ref == null || method == null){
      return;
    }

    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    List<Object> msgArgs = new ArrayList<>();
    msgArgs.add(ref);
    msgArgs.add(method);
    msgArgs.add(args);

    task.args = msgArgs;
    msg.what = WXDomHandler.MsgType.WX_DOM_INVOKE;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * Create a body for the current {@link com.taobao.weex.WXSDKInstance} according to given
   * parameter.
   * @param element info about how to create a body
   */
  public void createBody(JSONObject element) {
    if (element == null) {
      return;
    }
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    task.args = new ArrayList<>();
    task.args.add(element);
    msg.what = WXDomHandler.MsgType.WX_DOM_CREATE_BODY;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * Update attributes
   * @param ref
   * @param attr the expected attr
   */
  public void updateAttrs(String ref, JSONObject attr) {
    if (TextUtils.isEmpty(ref) || attr == null || attr.size() < 1) {
      return;
    }
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    task.args = new ArrayList<>();
    task.args.add(ref);
    task.args.add(attr);
    msg.what = WXDomHandler.MsgType.WX_DOM_UPDATE_ATTRS;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * Update DOM style.
   * @param ref DOM reference
   * @param style the expected style
   */
  public void updateStyle(String ref, JSONObject style) {
    if (TextUtils.isEmpty(ref) || style == null || style.size() < 1) {
      return;
    }
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    task.args = new ArrayList<>();
    task.args.add(ref);
    task.args.add(style);
    msg.what = WXDomHandler.MsgType.WX_DOM_UPDATE_STYLE;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * Remove a node for the node tree.
   * @param ref reference of the node to be removed.
   */
  public void removeElement(String ref) {
    if (TextUtils.isEmpty(ref)) {
      return;
    }
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    task.args = new ArrayList<>();
    task.args.add(ref);
    msg.what = WXDomHandler.MsgType.WX_DOM_REMOVE_DOM;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * Add a {@link WXDomObject} to the specified parent as its given n-th child.
   * @param parentRef reference of the parent.
   * @param element the node to be added
   * @param index the expected index that the new dom in its new parent
   */
  public void addElement(String parentRef, JSONObject element, Integer index) {
    if (element == null
        || TextUtils.isEmpty(parentRef)) {
      return;
    }
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    task.args = new ArrayList<>();
    task.args.add(parentRef);
    task.args.add(element);
    task.args.add(index);
    msg.what = WXDomHandler.MsgType.WX_DOM_ADD_DOM;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * Move the DomElement to the specified parent as its given n-th child.
   * @param ref reference of the node to be moved.
   * @param parentRef reference of the parent.
   * @param index the expected index that the dom in its new parent
   */
  public void moveElement(String ref, String parentRef, Integer index) {
    if (TextUtils.isEmpty(ref)
        || TextUtils.isEmpty(parentRef)) {
      return;
    }
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    task.args = new ArrayList<>();
    task.args.add(ref);
    task.args.add(parentRef);
    task.args.add(index);
    msg.what = WXDomHandler.MsgType.WX_DOM_MOVE_DOM;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * Add eventListener for the specified {@link WXDomObject}
   * @param ref reference of the node
   * @param type the type of the event listener to be added.
   */
  public void addEvent(String ref, String type) {
    if (TextUtils.isEmpty(ref) || TextUtils.isEmpty(type)) {
      return;
    }
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    task.args = new ArrayList<>();
    task.args.add(ref);
    task.args.add(type);
    msg.what = WXDomHandler.MsgType.WX_DOM_ADD_EVENT;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * Remove eventListener for the specified {@link WXDomObject}
   * @param ref reference of the node
   * @param type the type of the event listener to be removed.
   */
  public void removeEvent(String ref, String type) {
    if (TextUtils.isEmpty(ref) || TextUtils.isEmpty(type)) {
      return;
    }
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    task.args = new ArrayList<>();
    task.args.add(ref);
    task.args.add(type);
    msg.what = WXDomHandler.MsgType.WX_DOM_REMOVE_EVENT;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * Notify the {@link WXDomManager} that creation of dom tree is finished.
   * This notify is given by JS.
   */
  public void createFinish() {
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    msg.what = WXDomHandler.MsgType.WX_DOM_CREATE_FINISH;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * Notify the {@link WXDomManager} that refreshing of dom tree is finished.
   * This notify is given by JS.
   */
  public void refreshFinish() {
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    msg.what = WXDomHandler.MsgType.WX_DOM_REFRESH_FINISH;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  public void updateFinish() {
    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    msg.what = WXDomHandler.MsgType.WX_DOM_UPDATE_FINISH;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }


  /**
   * Scroll the specified {@link WXDomObject} to given offset in given duration
   * @param ref reference of specified dom object
   * @param options scroll option, like {offset:0, duration:300}
   */
  public void scrollToElement(String ref, JSONObject options) {
    if (TextUtils.isEmpty(ref) ) {
      return;
    }

    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    task.args = new ArrayList<>();
    task.args.add(ref);
    task.args.add(options);
    msg.what = WXDomHandler.MsgType.WX_DOM_SCROLLTO;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  public void addRule(String type, JSONObject options) {
    if (TextUtils.isEmpty(type) || options == null) {
      return;
    }

    Message msg = Message.obtain();
    WXDomTask task = new WXDomTask();
    task.instanceId = mWXSDKInstance.getInstanceId();
    task.args = new ArrayList<>();
    task.args.add(type);
    task.args.add(options);
    msg.what = WXDomHandler.MsgType.WX_DOM_ADD_RULE;
    msg.obj = task;
    WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
  }

  /**
   * By ref the width and height of the component.
   *
   * @param ref      the refer of component
   * @param callback function id
   */
  public void getComponentRect(String ref, String callback) {
    if (mWXSDKInstance == null) {
      return;
    }
    SimpleJSCallback jsCallback = new SimpleJSCallback(mWXSDKInstance.getInstanceId(), callback);
    if (TextUtils.isEmpty(ref)) {
      Map<String, Object> options = new HashMap<>();
      options.put("result", false);
      options.put("errMsg", "Illegal parameter");
      jsCallback.invoke(options);
      return;
    } else if ("viewport".equalsIgnoreCase(ref)) {
      if (mWXSDKInstance.getContainerView() != null) {
        Map<String, Object> options = new HashMap<>();
        Map<String, Float> sizes = new HashMap<>();
        int[] location = new int[2];
        mWXSDKInstance.getContainerView().getLocationOnScreen(location);
        sizes.put("left", 0f);
        sizes.put("top", 0f);
        sizes.put("right", getWebPxValue(mWXSDKInstance.getContainerView().getWidth()));
        sizes.put("bottom", getWebPxValue(mWXSDKInstance.getContainerView().getHeight()));
        sizes.put("width", getWebPxValue(mWXSDKInstance.getContainerView().getWidth()));
        sizes.put("height", getWebPxValue(mWXSDKInstance.getContainerView().getHeight()));
        options.put("size", sizes);
        options.put("result", true);
        jsCallback.invoke(options);
      } else {
        Map<String, Object> options = new HashMap<>();
        options.put("result", false);
        options.put("errMsg", "Component does not exist");
        jsCallback.invoke(options);
      }
    } else {
      Message msg = Message.obtain();
      WXDomTask task = new WXDomTask();
      task.instanceId = mWXSDKInstance.getInstanceId();
      task.args = new ArrayList<>();
      task.args.add(ref);
      task.args.add(jsCallback);
      msg.what = WXDomHandler.MsgType.WX_COMPONENT_SIZE;
      msg.obj = task;
      WXSDKManager.getInstance().getWXDomManager().sendMessage(msg);
    }
  }

  @NonNull
  private float getWebPxValue(int value) {
    return WXViewUtils.getWebPxByWidth(value,mWXSDKInstance.getInstanceViewPortWidth());
  }
}
