package com.talent.compat.web.core.useage;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.content.res.Resources;
import android.net.Uri;
import android.support.v7.app.AlertDialog;
import android.text.TextUtils;
import android.webkit.WebResourceRequest;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.talent.compat.web.R;
import com.talent.compat.web.core.compat.WebViewCompat;

import java.lang.ref.WeakReference;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Created by wbs on 2018/3/21 0021.
 */

public class DefaultWebViewClient extends WebViewClient {
    //intent schema
    public static final String INTENT_SCHEME = "intent";

    //AliPay schema
    public static final String ALIPAYS_SCHEME = "alipays";

    //sms schema
    public static final String SCHEME_SMS = "sms";

    //WeChat pay scheme ，用于唤醒微信支付
    public static final String WE_CHAT_PAY_SCHEME = "weixin://wap/pay?";

    private static final Set<String> _SCHEMA_LIST = new HashSet<>();

    static {
        _SCHEMA_LIST.add("http");
        _SCHEMA_LIST.add("https");
        _SCHEMA_LIST.add("file");
        _SCHEMA_LIST.add("weixin");
        _SCHEMA_LIST.add(INTENT_SCHEME);
        _SCHEMA_LIST.add(ALIPAYS_SCHEME);
        _SCHEMA_LIST.add(SCHEME_SMS);
        _SCHEMA_LIST.add("mailto");
        _SCHEMA_LIST.add("tel");
        _SCHEMA_LIST.add("geo");
    }

    //缩放
    private static final int CONSTANTS_ABNORMAL_BIG = 7;

    private boolean mAutoDealLink = true;

    private boolean mIsInterceptUnknownUrl = true;

    private int mLinkPageOpenMode;

    private final WeakReference<Activity> mWeakReference;


    public DefaultWebViewClient(Builder builder) {
        mWeakReference = new WeakReference<>(builder.mActivity);
        this.mAutoDealLink = builder.mWebClientHelper;
        mIsInterceptUnknownUrl = builder.mIsInterceptUnknownScheme;
        this.mLinkPageOpenMode = builder.mLinkPageOpenMode;
    }

    @Override
    public boolean shouldOverrideUrlLoading(WebView view, WebResourceRequest request) {
        boolean parentIntercept = super.shouldOverrideUrlLoading(view, request);
        return shouldOverrideUrl(view, request.getUrl().toString(), parentIntercept);
    }

    @Override
    public boolean shouldOverrideUrlLoading(WebView view, String url) {
        boolean parentIntercept = super.shouldOverrideUrlLoading(view, url);
        return shouldOverrideUrl(view, url, parentIntercept);
    }

    private boolean shouldOverrideUrl(WebView view, String url, boolean parentIntercept) {

        if (parentIntercept) {
            return true;
        }
        try {
            Uri uri = Uri.parse(url);
            String schema = uri.getScheme();
            if (!_SCHEMA_LIST.contains(schema)) {
                return handleUnknownUrl(url);
            }
            if (!mAutoDealLink) {
                return false;
            }
            if (handleCommonLink(url)) {
                return true;
            }
            if (url.startsWith(INTENT_SCHEME)) {
                handleIntentUrl(url);
                return true;
            }
            // 微信支付
            if (url.startsWith(WE_CHAT_PAY_SCHEME)) {
                startActivity(url);
                return true;
            }
            if (url.startsWith(ALIPAYS_SCHEME) && lookup(url)) {
                return true;
            }

        } catch (Exception e) {
            return handleUnknownUrl(url);
        }
        return false;
    }

    private boolean handleUnknownUrl(String url) {
        if (mIsInterceptUnknownUrl) {
            if (queryActivitiesNumber(url) > 0 && urlOpenWays(url)) {
                return true;
            }
        }
        return false;
    }

    private boolean handleCommonLink(String url) {
        if (url.startsWith(WebView.SCHEME_TEL)
                || url.startsWith(SCHEME_SMS)
                || url.startsWith(WebView.SCHEME_MAILTO)
                || url.startsWith(WebView.SCHEME_GEO)) {
            try {
                Activity mActivity = null;
                if ((mActivity = mWeakReference.get()) == null) {
                    return false;
                }
                Intent intent = new Intent(Intent.ACTION_VIEW);
                intent.setData(Uri.parse(url));
                mActivity.startActivity(intent);
            } catch (ActivityNotFoundException ignored) {

            }
            return true;
        }
        return false;
    }

    private void handleIntentUrl(String intentUrl) {
        try {

            if (TextUtils.isEmpty(intentUrl) || !intentUrl.startsWith(INTENT_SCHEME)) {
                return;
            }
            lookup(intentUrl);
        } catch (Throwable ignored) {

        }

    }

    private boolean lookup(String url) {
        try {
            Intent intent;
            Activity mActivity = null;
            if ((mActivity = mWeakReference.get()) == null) {
                return true;
            }
            PackageManager packageManager = mActivity.getPackageManager();
            intent = Intent.parseUri(url, Intent.URI_INTENT_SCHEME);
            ResolveInfo info = packageManager.resolveActivity(intent, PackageManager.MATCH_DEFAULT_ONLY);
            // 跳到该应用
            if (info != null) {
                mActivity.startActivity(intent);
                return true;
            }
        } catch (Throwable ignore) {
        }

        return false;
    }

    private void startActivity(String url) {
        try {

            if (mWeakReference.get() == null) {
                return;
            }
            Intent intent = new Intent();
            intent.setAction(Intent.ACTION_VIEW);
            intent.setData(Uri.parse(url));
            mWeakReference.get().startActivity(intent);

        } catch (Exception ignored) {

        }

    }

    private int queryActivitiesNumber(String url) {

        try {
            if (mWeakReference.get() == null) {
                return 0;
            }
            Intent intent = Intent.parseUri(url, Intent.URI_INTENT_SCHEME);
            PackageManager mPackageManager = mWeakReference.get().getPackageManager();
            List<ResolveInfo> mResolveInfos = mPackageManager.queryIntentActivities(intent, PackageManager.MATCH_DEFAULT_ONLY);
            return mResolveInfos == null ? 0 : mResolveInfos.size();
        } catch (URISyntaxException ignore) {
            return 0;
        }
    }

    private boolean urlOpenWays(final String url) {

        switch (mLinkPageOpenMode) {
            // 直接打开其他App
            case EasyWebView.OPEN_PAGE_AS_DIRECT:
                lookup(url);
                return true;
            // 咨询用户是否打开其他App
            case EasyWebView.OPEN_PAGE_AS_ASKING:
                Context context = mWeakReference.get();
                if (context == null) {
                    return false;
                }
                Resources mResources = context.getResources();
                new AlertDialog
                        .Builder(context)//
                        .setMessage(mResources.getString(R.string.web_leave_app_and_go_other_page,
                                WebViewCompat.getApplicationName(context)))//
                        .setTitle(mResources.getString(R.string.web_tips))
                        .setNegativeButton(android.R.string.cancel, new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int which) {
                                dialog.dismiss();
                            }
                        })//
                        .setPositiveButton(mResources.getString(R.string.web_leave), new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int which) {
                                lookup(url);
                            }
                        })
                        .create().show();

                return true;
            // 默认不打开
            default:
                return false;
        }
    }

    @Override
    public void onScaleChanged(WebView view, float oldScale, float newScale) {
        if (newScale - oldScale > CONSTANTS_ABNORMAL_BIG) {
            view.setInitialScale((int) (oldScale / newScale * 100));
        }
    }

    public static Builder asBuilder() {
        return new Builder();
    }

    public static class Builder {

        private Activity mActivity;

        private boolean mWebClientHelper;

        private boolean mIsInterceptUnknownScheme;

        private int mLinkPageOpenMode = EasyWebView.OPEN_PAGE_AS_ASKING;

        private Builder() {

        }

        public Builder component(Activity activity) {
            this.mActivity = activity;
            return this;
        }

        public Builder autoDealLink(boolean auto) {
            this.mWebClientHelper = auto;
            return this;
        }

        public Builder interceptUnknownSchema(boolean interceptUnkownScheme) {
            this.mIsInterceptUnknownScheme = interceptUnkownScheme;
            return this;
        }

        public Builder linkPageOpenMode(int linkPageOpenMode) {
            if (linkPageOpenMode < 0 || linkPageOpenMode > 2) {
                throw new IllegalArgumentException("bad linkPageOpenMode");
            }
            this.mLinkPageOpenMode = linkPageOpenMode;
            return this;
        }

        public DefaultWebViewClient build() {
            return new DefaultWebViewClient(this);
        }

    }

}
