package com.talent.compat.web.core;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.os.Build;
import android.util.AttributeSet;
import android.util.Pair;
import android.webkit.WebChromeClient;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.Toast;

import com.talent.compat.web.core.callback.SupportWebClient;
import com.talent.compat.web.core.callback.WebClient;
import com.talent.compat.web.core.compat.AccessibilityCompat;
import com.talent.compat.web.core.compat.WebViewCompat;
import com.talent.compat.web.core.compat.WebViewConfigCompat;
import com.talent.compat.web.core.compat.WebViewConfigCompat.WebViewConfigHandler;
import com.talent.compat.web.core.js.JellyBeanJsInjection;
import com.talent.compat.web.core.js.JsConstructors;
import com.talent.compat.web.core.js.JsInjection;
import com.talent.compat.web.core.js.UnderSupportJsInjection;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.talent.compat.web.core.compat.WebViewCompat.isWebViewPackageException;

/**
 * Created by wbs on 2018/3/20 0020.
 */

public class CompatWebView extends WebView {

    private final static WebViewConfigHandler mConfigHandler;

    static {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN) {
            mConfigHandler = new WebViewConfigCompat.LowVersionHandler();
        }else if(Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT){
            mConfigHandler = new WebViewConfigCompat.JellyBeanHandler();
        }else {
            mConfigHandler = new WebViewConfigCompat.KitKatHandler();
        }
    }

    private final JsInjection mJsInjection;

    private Map<String, String> mInjectJavaScripts;

    private final SupportWebClient mSupportWebClient;

    private final List<WebClient> mWebClients = new ArrayList<>();

    private final AccessibilityCompat mAccessCompat;

    private boolean mHasConstructed = false;

    private boolean mHasWebViewClient = false;

    private boolean mHasWebChromeClient = false;

    public CompatWebView(Context context) {
        this(context, null);
    }

    public CompatWebView(Context context, AttributeSet attrs) {
        this(context, attrs, android.R.attr.webViewStyle);
    }

    public CompatWebView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            mJsInjection = new JellyBeanJsInjection();
        } else {
            mJsInjection = new UnderSupportJsInjection();
        }
        mSupportWebClient = new SupportWebClient();
        registerWebClient(mSupportWebClient);
        mAccessCompat = new AccessibilityCompat(this);
        WebViewCompat.removeSearchBoxJavaBridge(this);
        mHasConstructed = true;
    }

    /**
     * @deprecated use {@link #addJavascript(Object, String)} replaced
     * @hide
     */
    @SuppressLint({"JavascriptInterface", "AddJavascriptInterface"})
    public void addJavascriptInterface(Object object, String name) {
        super.addJavascriptInterface(object, name);
    }

    /**
     * 经过大量的测试，按照以下方式才能保证JS脚本100%注入成功：
     * 1、在第一次loadUrl之前注入JS（在addJavascriptInterface里面注入即可，setWebViewClient和setWebChromeClient要在addJavascriptInterface之前执行）；
     * 2、在webViewClient.onPageStarted中都注入JS；
     * 3、在webChromeClient.onProgressChanged中都注入JS，并且不能通过自检查（onJsPrompt里面判断）JS是否注入成功来减少注入JS的次数，因为网页中的JS可以同时打开多个url导致无法控制检查的准确性；
     * @param object
     * @param name
     */
    public final void addJavascript(Object object, String name) {
        mJsInjection.addJavascriptInterface(this, object, name);
        onJavaScriptInterfaceAdded(object,name);
    }

    public final void addJavascript(String javaScript) {
        if (mInjectJavaScripts == null) {
            mInjectJavaScripts = new HashMap<>();
        }
        mInjectJavaScripts.put(String.valueOf(javaScript.hashCode()), javaScript);

    }

    @Override
    public final void setWebChromeClient(WebChromeClient client) {
        mHasWebChromeClient = true;
        WebChromeClientWrapper wrapper = new WebChromeClientWrapper(this);
        wrapper.setDelegate(client);
        mSupportWebClient.setWebChromeClient(wrapper);
        super.setWebChromeClient(wrapper);
        onWebChromeClientSetup(wrapper);
    }

    @Override
    public final void setWebViewClient(WebViewClient client) {
        mHasWebViewClient = true;
        WebViewClientWrapper wrapper = new WebViewClientWrapper(this,mAccessCompat);
        wrapper.setDelegate(client);
        super.setWebViewClient(wrapper);
        onWebViewClientSetup(wrapper);
    }


    @Override
    public void destroy() {
        setVisibility(GONE);
        mJsInjection.unregister();
        if(mInjectJavaScripts!=null){
            mInjectJavaScripts.clear();
            mInjectJavaScripts = null;
        }
        mWebClients.clear();
        super.setWebChromeClient(null);
        super.setWebViewClient(null);
        removeAllViewsInLayout();
        WebViewCompat.detachWebParent(this);
        mConfigHandler.destroyWebViewConfigCallback();
        if(mHasConstructed){
            mAccessCompat.resetAccessibilityEnabled();
        }
        super.destroy();
    }

    @Override
    public void clearHistory() {
        if (mHasConstructed) {
            super.clearHistory();
        }
    }

    @Override
    public void setOverScrollMode(int mode) {
        try {
            super.setOverScrollMode(mode);
        } catch (Throwable e) {
            Pair<Boolean, String> pair = isWebViewPackageException(e);
            if (pair.first) {
                Toast.makeText(getContext(), pair.second, Toast.LENGTH_SHORT).show();
                destroy();
            } else {
                throw e;
            }
        }
    }

    @Override
    public boolean isPrivateBrowsingEnabled() {
        if (Build.VERSION.SDK_INT == Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1
                && getSettings() == null) {

            return false; // getSettings().isPrivateBrowsingEnabled()
        } else {
            return super.isPrivateBrowsingEnabled();
        }
    }

    //-----------------------
    protected void onWebChromeClientSetup(WebChromeClient client) {

    }
    protected void onJavaScriptInterfaceAdded(Object interfaceObj, String interfaceName) {

    }
    public void onWebViewClientSetup(WebViewClient client) {

    }

    //----------------------
    final void onReceivedTitle(WebView view, String title) {
        for (WebClient client:mWebClients){
            client.onReceivedTitle(view,title);
        }
    }

    final void onPageStarted(WebView view, String url, Bitmap favicon) {
        for (WebClient client:mWebClients){
            client.onPageStarted(view,url,favicon);
        }
    }

    final void onPageFinished(WebView view, String url) {
        for (WebClient client:mWebClients){
            client.onPageFinished(view,url);
        }
    }

    final void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
        for (WebClient client:mWebClients){
            client.onReceivedError(view,errorCode,description,failingUrl);
        }
    }

    final void onProgressChanged(WebView view, int newProgress) {
        for (WebClient client:mWebClients){
            client.onProgressChanged(view,newProgress);
        }
    }
    public void registerWebClient(WebClient client){
        if(client==null){
            return;
        }
        synchronized (mWebClients){
            mWebClients.add(client);
        }
    }

    public void unRegisterWebClient(WebClient client){
        if(client==null){
            return;
        }
        synchronized (mWebClients){
            mWebClients.remove(client);
        }
    }

    public void useDefaultWebViewClient(){
        setWebViewClient(null);
    }
    public void useDefaultWebChromeClient(){
        setWebChromeClient(null);
    }

    public JsInjection getJsInjection(){
        return mJsInjection;
    }

    public void injectJavaScript(){
        if(mInjectJavaScripts==null){
            return;
        }
        for (Map.Entry<String, String> entry : mInjectJavaScripts.entrySet()) {
            this.loadUrl(JsConstructors.buildNotRepeatInjectJS(entry.getKey(), entry.getValue()));
        }
    }

    public boolean isInstallWebViewClient() {
        return mHasWebViewClient;
    }

    public boolean isInstallWebChromeClient() {
        return mHasWebChromeClient;
    }

    public void prepare(){
        if(!isInstallWebViewClient()){
            useDefaultWebViewClient();
        }
        if(!isInstallWebChromeClient()){
            useDefaultWebChromeClient();
        }
    }

}
