package com.taboola.android.plus.core;


import android.os.Build;
import android.support.annotation.RequiresApi;
import android.util.Log;

import com.google.firebase.messaging.FirebaseMessagingService;
import com.google.firebase.messaging.RemoteMessage;
import com.ocamba.hoood.OcambaHoood;
import com.taboola.android.utils.Logger;

import java.util.Map;

import static com.taboola.android.plus.core.TaboolaSdkPlus.getCoreProvider;

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
public abstract class TBLPushManager implements IFeatureModule {
    private static final String TAG = TBLPushManager.class.getSimpleName();

    /**
     * remove key from user tracking data
     * if the key is reserved by sdk+ for internal usage, it won't be removed
     *
     * @param key to remove
     */
    public abstract void removeUserTrackingDataKey(String key);

    /**
     * updates user tracking data.
     * If a tracking key is reserved by sdk+ for internal usage or a key is passed with an empty string
     * as a value this key will be ignored.
     *
     * @param trackingData key/values map
     */
    public abstract void sendUserTrackingData(Map<String, Object> trackingData);

    /**
     * Sets application icon, which will be displayed on push notification
     *
     * @param iconId resource id of icon, which is set as application icon
     * @return {@link TBLPushManager}
     */
    public abstract TBLPushManager setNotificationAppIconId(int iconId);

    /**
     * Sets application name label, which will be displayed on push notification
     *
     * @param applicationName name label, which is set as application name label
     * @return {@link TBLPushManager}
     */
    public abstract TBLPushManager setNotificationAppNameLabel(String applicationName);


    /**
     * Handles received {@link RemoteMessage}
     * NOTE: this method must be called in overridden method
     * {@link FirebaseMessagingService#onMessageReceived(RemoteMessage)}
     *
     * @param remoteMessage provided from {@link FirebaseMessagingService#onMessageReceived(RemoteMessage)}
     */
    public static void handleTaboolaPush(final RemoteMessage remoteMessage) {
        Logger.d(TAG, "handleTaboolaPush() called");

        BridgeInternalCore.getPushNotificationManagerAsync(new PushManagerCallback() {
            @Override
            public void onManagerRetrieved(AbsPushManager pushNotificationsManager) {
                pushNotificationsManager.handleTaboolaPushInternal(remoteMessage);
            }

            @Override
            public void onManagerRetrieveFailed(Throwable throwable) {
                Logger.e(TAG, throwable.toString(), throwable);
            }
        });
    }

    /**
     * NOTE: this method must be called in overridden method
     * {@link FirebaseMessagingService#onNewToken(String)}
     *
     * @param token a new firebase token
     */
    public static void onNewFirebaseToken(final String token) {
        Logger.d(TAG, "onNewFirebaseToken() called with: token = [" + token + "]");
        //if manager is not initialized then cache the token to send it when init is called
        getCoreProvider().getPushNotificationsManager().onNewFirebaseTokenInternal(token);
    }

    /**
     * @param remoteMessage provided in {@link FirebaseMessagingService#onMessageReceived(RemoteMessage)}
     * @return {@code true} if the FCM RemoteMessage is not empty and sent by Taboola, {@code false} otherwise.
     */
    public static boolean isTaboolaMessage(RemoteMessage remoteMessage) {
        return remoteMessage != null &&
                !remoteMessage.getData().isEmpty() &&
                OcambaHoood.notification().isOcambaMessage(remoteMessage);
    }
}