package com.taboola.android.infra.persistence;

import androidx.annotation.NonNull;
import androidx.arch.core.util.Function;
import androidx.core.util.Supplier;

import com.taboola.android.infra.utilities.Subscription;


/**
 * Class for holding value of type T which persists across process restarts and version updates.
 * Created by ronk on 2017-02-13.
 */
public interface PersistentResource<T> extends Supplier<T> {
    @Override
    @NonNull
    T get();

    void set(@NonNull T value);

    @NonNull
    Subscription registerListener(@NonNull OnChangeListener<T> onChangeListener);

    /**
     * applies the given transaction to the stored value, stores the new value and
     * returns the previous value
     */
    default T transact(@NonNull Function<T, T> transaction) {
        synchronized (this) {
            T v = get();
            T result = transaction.apply(v);
            set(result);
            return v;
        }
    }

    void unregisterListener(@NonNull OnChangeListener<T> onChangeListener);

    /**
     * same as {@link #registerListener(OnChangeListener)} with additional call to the provided listener
     * done using the original (initial) value
      */
    @NonNull
    default Subscription subscribe(@NonNull OnChangeListener<T> onChangeListener) {
        Subscription subscription = registerListener(onChangeListener);
        onChangeListener.onValueChanged(this);
        return subscription;
    }

    interface OnChangeListener<T> {
        void onValueChanged(@NonNull PersistentResource<T> persistentResource);
    }

}
