package com.taboola.android.infra.inappupdate.internals;

import android.app.Activity;

import com.google.android.play.core.appupdate.AppUpdateInfo;
import com.google.android.play.core.appupdate.AppUpdateManager;
import com.google.android.play.core.appupdate.AppUpdateOptions;
import com.google.android.play.core.install.model.ActivityResult;
import com.google.android.play.core.tasks.Task;
import com.taboola.android.infra.inappupdate.Configuration;
import com.taboola.android.infra.inappupdate.InAppUpdateException;
import com.taboola.android.infra.inappupdate.TriggerEventsCallback;
import com.taboola.android.infra.inappupdate.UpdateFlowException;
import com.taboola.android.infra.utilities.Executor2;

abstract class UpdateRunner implements Runnable {
    private final Executor2 uiExecutor;
    private final AppUpdateInfo updateInfo;
    private final Configuration configuration;
    private final TriggerEventsCallback callback;
    private final AppUpdateManager appUpdateManager;
    private final Activity activity;

    public UpdateRunner(Executor2 uiExecutor, AppUpdateInfo updateInfo, Configuration configuration, TriggerEventsCallback callback, AppUpdateManager appUpdateManager, Activity activity) {
        this.uiExecutor = uiExecutor;
        this.updateInfo = updateInfo;
        this.configuration = configuration;
        this.callback = callback;
        this.appUpdateManager = appUpdateManager;
        this.activity = activity;
    }

    @Override
    public void run() {
        AppUpdateOptions appUpdateOptions = AppUpdateOptions.defaultOptions(configuration.getUpdateType());
        Task<Integer> flowResult = appUpdateManager.startUpdateFlow(updateInfo, activity, appUpdateOptions);
        flowResult.addOnCompleteListener(uiExecutor::submit, task -> {
            try {
                if (!task.isSuccessful()) {
                    throw new UpdateFlowException("started with disallowed update type", InAppUpdateException.ErrorCode.UPDATE_NOT_STARTED);
                }
                int result = task.getResult();
                if (result != Activity.RESULT_OK) {
                    throw new UpdateFlowException("failed with error " + result, getErrorCode(result));
                }
                callback.onUserAcceptedUpdate();
                startMonitoringUpdateInProgress();

            } catch (InAppUpdateException e) {
                callback.onUpdateFlowFailed(e);
            }
        });

    }

    private InAppUpdateException.ErrorCode getErrorCode(int result) {
        if (result == ActivityResult.RESULT_IN_APP_UPDATE_FAILED) {
            return InAppUpdateException.ErrorCode.UPDATE_FAILED;
        } else if (result == Activity.RESULT_CANCELED) {
            return InAppUpdateException.ErrorCode.UPDATE_CANCELLED;
        }
        return InAppUpdateException.ErrorCode.ERROR_UNEXPECTED;
    }

    protected abstract void startMonitoringUpdateInProgress();
}
