package com.taboola.android.infra.inappupdate.internals;

import android.annotation.SuppressLint;

import androidx.annotation.AnyThread;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;
import androidx.core.util.Consumer;
import androidx.lifecycle.Lifecycle;
import androidx.lifecycle.ProcessLifecycleOwner;

import com.google.android.play.core.appupdate.AppUpdateInfo;
import com.google.android.play.core.install.model.UpdateAvailability;
import com.google.android.play.core.tasks.Task;
import com.taboola.android.infra.inappupdate.Configuration;
import com.taboola.android.infra.inappupdate.InAppUpdateException;
import com.taboola.android.infra.inappupdate.PreconditionNotMetException;
import com.taboola.android.infra.inappupdate.TriggerEventsCallback;
import com.taboola.android.infra.persistence.PersistentResource;
import com.taboola.android.infra.utilities.Executor2;

import java.util.Date;
import java.util.concurrent.TimeUnit;

@SuppressLint("Assert")
abstract class PreconditionsChecker implements Runnable {
    @Nullable
    private final Configuration maybeConfiguration;
    private final TriggerEventsCallback triggerEventsCallback;
    private final Executor2 uiExecutor;
    private final PersistentResource<Long> lastAppearanceTime;
    private final PersistentResource<Integer> appearancesCount;
    private Configuration configuration;

    PreconditionsChecker(@Nullable Configuration configuration,
                         TriggerEventsCallback triggerEventsCallback,
                         Executor2 uiExecutor,
                         PersistentResource<Long> lastAppearanceTime,
                         PersistentResource<Integer> appearancesCount) {
        this.maybeConfiguration = configuration;
        this.triggerEventsCallback = triggerEventsCallback;
        this.uiExecutor = uiExecutor;
        this.lastAppearanceTime = lastAppearanceTime;
        this.appearancesCount = appearancesCount;
    }

    @WorkerThread
    @Override
    public void run() {
        try {
            configuration = checkConfiguration();
            checkVersion();
            checkTotalCap();
            checkAppearancesInterval();
            inquireUpdateAvailability(this::afterUpdateInfoRetrieved);
        } catch (InAppUpdateException e) {
            notifyCallbackFailure(e);
        }
    }


    private void afterUpdateInfoRetrieved(AppUpdateInfo updateInfo) {
        notifyCallback(it -> it.onUpdateAvailable(updateInfo));
        assert updateInfo.updateAvailability() == UpdateAvailability.UPDATE_AVAILABLE;
        try {
            checkVersionStaleness(updateInfo);
            checkUpdateTypeAllowed(updateInfo);
            checkHostAppIsShowing();

            notifyCallback(TriggerEventsCallback::onUpdateDialogShown);

            updateAppearanceData();
            onAllChecksPassed(updateInfo);
        } catch (InAppUpdateException e) {
            notifyCallbackFailure(e);
        }
    }


    protected abstract void onAllChecksPassed(AppUpdateInfo updateInfo);

    private void checkUpdateTypeAllowed(AppUpdateInfo updateInfo) throws InAppUpdateException {
        int wantedType = configuration.getUpdateType();
        if (!updateInfo.isUpdateTypeAllowed(wantedType)) {
            throw new PreconditionNotMetException("wanted update type not allowed: " + wantedType, InAppUpdateException.ErrorCode.PRECONDITION_UPDATE_TYPE);
        }
    }

    private void checkHostAppIsShowing() throws InAppUpdateException {
        boolean appInForeground = isAppInForeground();
        if (!appInForeground) {
            throw new PreconditionNotMetException("app is in background", InAppUpdateException.ErrorCode.PRECONDITION_APP_BACKGROUND);
        }
    }

    @VisibleForTesting
    boolean isAppInForeground() {
        return ProcessLifecycleOwner.get().getLifecycle().getCurrentState().isAtLeast(Lifecycle.State.RESUMED);
    }

    private void checkVersionStaleness(AppUpdateInfo updateInfo) throws InAppUpdateException {
        Integer clientVersionStalenessDays = updateInfo.clientVersionStalenessDays();
        if (clientVersionStalenessDays != null && configuration.getAvailabilityDelayDays() > clientVersionStalenessDays) {
            throw new PreconditionNotMetException("availability delay", InAppUpdateException.ErrorCode.PRECONDITION_AVAILABILITY_DELAY);
        }
    }

    @AnyThread
    private void notifyCallbackFailure(InAppUpdateException exception) {
        notifyCallback(it -> it.onPreconditionsCheckFailed(exception));
    }

    private void notifyCallback(Consumer<TriggerEventsCallback> what) {
        uiExecutor.submit(() -> what.accept(triggerEventsCallback));
    }

    @SuppressLint("Assert")
    private void inquireUpdateAvailability(Consumer<AppUpdateInfo> onUpdateAvailable) {
        getAppUpdateInfoTask().addOnCompleteListener(task -> {
            if (!task.isSuccessful()) {
                Exception exception = task.getException();
                assert exception != null;
                notifyCallbackFailure(new InAppUpdateException(exception, InAppUpdateException.ErrorCode.PRECONDITION_NO_UPDATE_INFO));
                return;
            }
            AppUpdateInfo appUpdateInfo = task.getResult();
            int availability = appUpdateInfo.updateAvailability();
            switch (availability) {
                case UpdateAvailability.UPDATE_AVAILABLE:
                    onUpdateAvailable.accept(appUpdateInfo);
                    break;
                case UpdateAvailability.DEVELOPER_TRIGGERED_UPDATE_IN_PROGRESS:
                    startMonitoringUpdateInProgress();
                    break;
                case UpdateAvailability.UNKNOWN:
                    //fall-through
                case UpdateAvailability.UPDATE_NOT_AVAILABLE:
                    notifyCallbackFailure(new InAppUpdateException("no app update available: status=" + availability, InAppUpdateException.ErrorCode.PRECONDITION_NO_UPDATE_AVAILABLE));
                    break;
                default:
                    assert false : "unhandled case " + availability;
                    notifyCallbackFailure(new InAppUpdateException("unexpected error", InAppUpdateException.ErrorCode.ERROR_UNEXPECTED));
            }
        });
    }

    protected abstract void startMonitoringUpdateInProgress();

    @NonNull
    protected abstract Task<AppUpdateInfo> getAppUpdateInfoTask();

    @NonNull
    private Configuration checkConfiguration() throws InAppUpdateException {
        if (maybeConfiguration == null) {
            throw new PreconditionNotMetException("no configuration", InAppUpdateException.ErrorCode.PRECONDITION_CONFIGURATION_NOT_SET);
        }
        maybeConfiguration.validate();
        return maybeConfiguration;
    }

    private void checkVersion() throws InAppUpdateException {
        AppData appData = getAppData();

        if (appData.getVersionCode() >= configuration.getTargetVersion()) {
            throw new PreconditionNotMetException("target version lower than current", InAppUpdateException.ErrorCode.PRECONDITION_TARGET_VERSION);
        }
    }

    @NonNull
    protected abstract AppData getAppData() throws InAppUpdateException;

    private void checkTotalCap() throws InAppUpdateException {
        int appearancesCount = getAppearancesCount();
        if (appearancesCount >= configuration.getTotalAppearancesCap()) {
            throw new PreconditionNotMetException("total appearances cap exceeded", InAppUpdateException.ErrorCode.PRECONDITION_TOTAL_CAP);
        }
    }

    private void checkAppearancesInterval() throws InAppUpdateException {
        Date nextPermittedAppearanceTime = getNextPermittedAppearanceTime();
        if (nextPermittedAppearanceTime.after(now())) {
            throw new PreconditionNotMetException("not allowed to appear before " + nextPermittedAppearanceTime, InAppUpdateException.ErrorCode.PRECONDITION_APPEARANCE_INTERVAL);
        }
    }

    @VisibleForTesting
    Date now() {
        return new Date();
    }

    @NonNull
    private Date getNextPermittedAppearanceTime() {
        long lastAppearanceTimeEpoch = lastAppearanceTime.get();
        long nextPermittedAppearanceTimeEpoch = lastAppearanceTimeEpoch + TimeUnit.MINUTES.toMillis(configuration.getMinimalAppearanceIntervalMinutes());
        return new Date(nextPermittedAppearanceTimeEpoch);
    }


    private int getAppearancesCount() {
        return appearancesCount.get();
    }

    private void updateAppearanceData() {
        lastAppearanceTime.set(now().getTime());
        appearancesCount.transact(it -> it + 1);
    }
}
