package com.taboola.android.api;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.taboola.android.utils.Logger;

import java.util.List;
import java.util.Map;

public interface PublicApi {

    interface PublicTaboolaApi {

        /**
         * Initializes TaboolaApi. Must be called before any other method of the SDK.
         * Typically you would want to do it in you Application class in OnCreate()
         *
         * @param applicationContext Application Context
         * @param publisherId        The publisher id in the Taboola system
         * @param apiKey             The API key that was provided by Taboola for the application
         */
        TaboolaApi init(Context applicationContext, String publisherId, String apiKey);

        /**
         * Requests recommendation items.
         *
         * @param request  {@link TBRecommendationsRequest} with at least one {@link TBPlacementRequest}
         * @param callback callback to pass result into
         */
        void fetchRecommendations(final TBRecommendationsRequest request,
                                  final TBRecommendationRequestCallback callback);

        /**
         * Request next batch of the recommendation items for specified placement.
         * Requested amount of recommendation items will be the same as in the previous request.
         *
         * @param placement placement to request next recommendation items for
         * @param callback  callback to pass result into
         */
        void getNextBatchForPlacement(TBPlacement placement,
                                      final TBRecommendationRequestCallback callback);

        /**
         * Request next batch of the recommendation items for specified placement.
         *
         * @param placement placement to request next recommendation items for
         * @param count     amount of recommendation items to fetch
         * @param callback  callback to pass result into
         */
        void getNextBatchForPlacement(TBPlacement placement, int count,
                                      final TBRecommendationRequestCallback callback);

        /**
         * Shows popup with information about Taboola
         * <p>Must be called when user clicks on Attribution view (View with "By Taboola" text).
         *
         * @param context is used to display popup
         */
        void handleAttributionClick(Context context);

        /**
         * Sdk allows to intercept recommendation clicks, and block default click handling for organic items.
         * If you wand to block the default behavior, and display the recommendation content on your own,
         * return {@code false} in {@link TaboolaOnClickListener#onItemClick(String, String, String, boolean)}
         * <p> Note: The return value will be ignored for non-organic items.
         * <p>The listener will be called every time a user clicks a recommendation, right before triggering the default behavior
         */
        TaboolaApi setOnClickListener(TaboolaOnClickListener onClickListener);

        /**
         * To avoid accidental user clicks, the TB views will ignore clicks that were done in the first
         * X milliseconds after the view became visible.
         * The default is 300 ms.
         * <p> (Keep in mind that visibility check also has a 400 ms delay)
         *
         * @param onClickIgnoreTimeMs time in milliseconds
         */
        TaboolaApi setOnClickIgnoreTimeMs(int onClickIgnoreTimeMs);

        /**
         * Sets log level for the SDK. You can find logLevel constants in the {@link Logger} class, for
         * example Logger.DEBUG
         */
        TaboolaApi setLogLevel(int logLevel);

        /**
         * @param placeholderDrawable A placeholder drawable to be used while the image is being loaded.
         */
        TaboolaApi setImagePlaceholder(Drawable placeholderDrawable);
    }

    interface PublicTBView {

        /**
         * Triggers OnClick event on the RecommendationItem.
         * Can be used if you want to open ad by a different action (e.g. swipe)
         * <p>  Ignores visibility requirements for the click.
         */
        void handleClick();
    }

    interface PublicTBPlacement {

        /**
         * @return list of RecommendationItems for this placement
         */
        List<TBRecommendationItem> getItems();

        /**
         * Asynchronously loads in cache thumbnail images in all RecommendationItems in this Placement
         * using default caching mechanism from Picasso, which is:
         * <ul>
         * <li>LRU memory cache of 15% the available application RAM</li>
         * <li>Disk cache of 2% storage space up to 50MB but no less than 5MB. (Note: this is only
         * available on API 14+ <em>or</em> if you are using a standalone library that provides a disk
         * cache on all API levels like OkHttp)</li>
         * <li>Three download threads for disk and network access.</li>
         * </ul>
         */
        void prefetchThumbnails();
    }

    interface PublicTBPlacementRequest {

        /**
         * @param name A text string, uniquely identifying the placement
         */
        TBPlacementRequest setName(String name);

        /**
         * @param recCount A non-negative integer specifying how many recommendations are requested.
         *                 The API will not return more recommendations than this, though if there are
         *                 not enough good quality recommendations to satisfy the request,
         *                 fewer items might be returned.
         */
        TBPlacementRequest setRecCount(int recCount);

        /**
         * Note that this parameter does not influence the type of sponsored content
         * returned – the mix of the sponsored content types is currently determined
         * by a server side configuration.
         * <p>Valid values:
         * <li>video -  return only video content </li>
         * <li>text -  return only textual (stories) content </li>
         * <li>photo -  return only photo / galleries content </li>
         * <li>mix - (default)  return a mix of several content types. In case a mix is
         * requested, the exact mix is determined by server side
         * configuration. Different configurations can be used for
         * different source.type and placement parameters.
         * </li>
         *
         * @param targetType Type of recommended organic content to return.
         */
        TBPlacementRequest setTargetType(String targetType);

        /**
         * Sets the size (in pixels) of the thumbnails returned for recommendations.
         * Both parameters (height and width) should either appear together or not
         * appear at all.
         * <p>In case the thumbnail size parameters are specified, the thumbnail would
         * always preserve its original aspect ratio – it will be scaled to the required
         * size, centered and cropped if needed.
         * <p>In addition, in case the thumbnail contains a face, we will detect that face
         * by default and ensure it is contained within the generated thumbnail.
         *
         * @param width
         * @param height
         */
        TBPlacementRequest setThumbnailSize(int width, int height);
    }

    interface PublicTBRecommendationItem {

        /**
         * Returns TBImageView that contains Thumbnail image of the recommendation item.
         * If PlacementRequest contained Thumbnail size, is will be set even before image is actually
         * loaded.
         *
         * @param activityContext Activity context
         * @return {@link TBImageView}
         * @see TaboolaApi#setImagePlaceholder(Drawable)
         * @see TBRecommendationItem#prefetchThumbnail()
         */
        @NonNull
        TBImageView getThumbnailView(Context activityContext);

        /**
         * Returns styled TBTextView that contains the Title of the recommended item.
         *
         * @param activityContext Activity context
         * @return {@link TBTextView}
         */
        @NonNull
        TBTextView getTitleView(Context activityContext);

        /**
         * If branding text is available returns styled TBTextView that contains the branding text for
         * the network/sponsored item.
         * <p> If branding text is not available returns null
         *
         * @param activityContext Activity context
         * @return {@link TBTextView} or {@code null}
         */
        @Nullable
        TBTextView getBrandingView(Context activityContext);

        /**
         * If description text is available returns styled TBTextView that contains description for
         * the recommended item
         * <p> If description text is not available returns null
         *
         * @param activityContext Activity context
         * @return {@link TBTextView} or {@code null}
         */
        @Nullable
        TBTextView getDescriptionView(Context activityContext);

        /**
         * Asynchronously loads in cache thumbnail image of this RecommendationItem using default caching
         * mechanism from Picasso, which is:
         * <ul>
         * <li>LRU memory cache of 15% the available application RAM</li>
         * <li>Disk cache of 2% storage space up to 50MB but no less than 5MB. (Note: this is only
         * available on API 14+ <em>or</em> if you are using a standalone library that provides a disk
         * cache on all API levels like OkHttp)</li>
         * <li>Three download threads for disk and network access.</li>
         * </ul>
         */
        void prefetchThumbnail();

        /**
         * Triggers OnClick event on the RecommendationItem.
         * Can be used if you want to open ad by a different action (e.g. swipe)
         * <p>  Ignores visibility requirements for the click.
         *
         * @param context Activity context
         */
        void handleClick(Context context);
    }

    interface PublicTBRecommendationsRequest {

        /**
         * adds specified {@link TBPlacementRequest} to this {@link TBRecommendationsRequest}.
         * Maximum allowed amount is 12 {@link TBPlacementRequest}  per {@link TBRecommendationsRequest}.
         * All placements must have unique names.
         *
         * @param placementRequest request to add
         */
        TBRecommendationsRequest addPlacementRequest(TBPlacementRequest placementRequest);

        /**
         * Sets the type of the content the recommendations will be placed next to.
         * <p>The following values are supported:
         * <li> video -  when the recommendations are placed near a video</li>
         * <li> text - when the recommendations are placed near a textual piece such as an article / story</li>
         * <li> photo -  when the recommendations are placed near a photo or gallery </li>
         * <li> home - when the recommendations are placed on the app entry (home) screen </li>
         * <li> section - when the recommendations are placed on a screen representing a “section front” or” topic” </li>
         * <li> search -  when the recommendations are placed on a search results page</li>
         *
         * @param sourceType
         */
        TBRecommendationsRequest setSourceType(String sourceType);

        /**
         * @param url A fully qualified (http:// or https://) publicly accessible URL that contains
         *            the content and/or meta data for the current source item (the piece of
         *            content the recommendations are going to be placed next to).
         */
        TBRecommendationsRequest setPageUrl(String url);

        /**
         * @param userReferrer The referrer (HTTP header) from the request that led to the current page.
         */
        TBRecommendationsRequest setUserReferrer(String userReferrer);

        /**
         * @param userUnifiedId An opaque, anonymized and unique identifier of the user in the
         *                      publisher’s eco-system. This identifier should be identical cross
         *                      application and device (e.g. hashed e-mail, or login)
         */
        TBRecommendationsRequest setUserUnifiedId(String userUnifiedId);
    }

    interface PublicTBRecommendationsResponse {

        /**
         * @return a map with the Placements where a Key is the Placements name and the Value
         * is the {@link TBPlacement} object
         */
        Map<String, TBPlacement> getPlacementsMap();
    }
}