package com.surveymonkey.surveymonkeyandroidsdk.utils;

import android.app.Activity;
import android.content.pm.PackageManager;
import android.os.Handler;
import android.widget.Toast;
import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import java.util.ArrayList;
import java.util.List;

public class PermissionHandler {

    final static String TAG = PermissionHandler.class.getSimpleName();
    final static int REQUEST_CODE = 32;
    private boolean mVisible;
    private List<Permission> mPermissions = new ArrayList<>();

    private Listener mListener;

    private final Activity mActivity;

    public PermissionHandler(Activity activity) {
        mActivity = activity;
    }

    public void setRequestPermissions(List<Permission> permissions) {
        mPermissions = permissions;
    }

    public void request(final Listener listener) {
        // this may bring up permission dialog, use "post" to allow caller
        // to properly dismiss its own dialog beforehand.
        new Handler().post(() -> _go(listener));
    }

    public boolean isVisible() {
        return mVisible;
    }

    private void _go(Listener listener) {

        boolean allTrue = true;
        List<String> requestPermissions = new ArrayList<>();
        for (Permission p : mPermissions) {
            if (ContextCompat.checkSelfPermission(mActivity, p.type) != PackageManager.PERMISSION_GRANTED) {
                allTrue = false;
                requestPermissions.add(p.type);
            }
        }

        if (allTrue) {
            listener.onGranted();
            return;
        }

        mListener = listener;
        mVisible = true;

        int requestSize = requestPermissions.size();
        ActivityCompat.requestPermissions(mActivity, requestPermissions.toArray(new String[requestSize]), REQUEST_CODE);
    }

    final public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        if (requestCode != REQUEST_CODE) {
            return;
        }

        boolean allPermissionsGranted = true;
        Permission deniedPermission = null;

        // Loop through permissions and make sure they are all granted
        for (int i = 0; i < grantResults.length; i++) {
            if (grantResults[i] != PackageManager.PERMISSION_GRANTED) {
                allPermissionsGranted = false;
                if (deniedPermission == null) {
                    // Find the permission that was denied
                    for (Permission p: mPermissions) {
                        if (p.type.equals(permissions[i])) {
                            deniedPermission = p;
                        }
                    }
                }
            }
        }

        if (allPermissionsGranted) {
            mVisible = false;
            mListener.onGranted();
            return;
        }

        boolean showAppSetting = false;

        if (deniedPermission != null) {
            boolean neverAskAgainSelected = !ActivityCompat.shouldShowRequestPermissionRationale(mActivity, deniedPermission.type);
            if (neverAskAgainSelected) {
                showAppSetting = true;
            }

            if (!showAppSetting) {
                mVisible = false;
                Toast.makeText(mActivity, deniedPermission.deniedMessageId, Toast.LENGTH_SHORT).show();
            }
        }

        if (mListener != null) {
            mListener.onDenied(showAppSetting);
        }

    }

    public interface Listener {
        void onGranted();
        void onDenied(boolean showSettings);
    }

    public static class Permission {
        String type;
        int rationaleTitleId;
        int rationaleMessageId;
        int deniedMessageId;

        public Permission() {
        }

        public Permission setType(String type) {
            this.type = type;
            return this;
        }

        public Permission setRationaleMessageIds(int rationaleTitleId, int rationaleMessageId) {
            this.rationaleTitleId = rationaleTitleId;
            this.rationaleMessageId = rationaleMessageId;
            return this;
        }

        public Permission setDeniedMessageId(int deniedMessageId) {
            this.deniedMessageId = deniedMessageId;
            return this;
        }
    }
}
