package com.surveymonkey.surveymonkeyandroidsdk;


import android.app.Activity;
import android.app.AlertDialog;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.res.Resources;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.TextView;

import com.surveymonkey.surveymonkeyandroidsdk.loaders.RetrieveSPageTask;
import com.surveymonkey.surveymonkeyandroidsdk.utils.SMConstants;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;

import static com.surveymonkey.surveymonkeyandroidsdk.SMFeedbackActivity.startActivityForResult;

public class SurveyMonkey {
    private static final String SM_BASE_URL = "https://www.surveymonkey.com/r/";
    private static final String SURVEY_STATUS = "survey_status";
    private static final String HTML = "html";
    private static final String COLLECTOR_CLOSED = "collector_closed";
    private static final long THREE_DAYS = 259200000L;
    private static final long THREE_MONTHS = 7884000000L;
    private static final long THREE_WEEKS = 1814400000L;

    private Activity mContext;
    private String mCollectorHash;
    private JSONObject mCustomVariables;
    private int mRequestCode;
    private String mSPageHTML;

    public SurveyMonkey() {
    }

    /**
     * Use the onStart method to present an AlertDialog prompt to the user encouraging them to take your in-app feedback survey.
     * By default, the user will be prompted to take a feedback survey 3 days after installing the app. If they decline to take the survey, they'll be prompted again in 3 weeks. Finally, if they consent to take the survey, they'll be prompted to take it again in 3 months.
     *
     * @param activity      The activity that will spawn the SMFeedbackActivity - <strong>Note:</strong> The spawning activity must implement onActivityResult to capture the respondent JSONObject or SMError object returned by SurveyMonkey
     * @param appName       The name of your app - used in the survey prompt dialog e.g. "Enjoying the SurveyMonkey app?"
     * @param requestCode   If greater than or equal to 0, this code will be returned in onActivityResult() when the activity exits.
     * @param collectorHash The collector hash that points to your Mobile SDK Collector - found on the Collector Details page of your chosen collector on SurveyMonkey.com
     *                      <p>Once the user completes your survey, a JSONObject containing the user's response will be returned to the spawning activity in onActivityResult
     *                      <p><strong>Note:</strong> If there are problems collecting the user's response, the SDK will return an SMError object with one of the domains and error codes listed in the SMError documentation
     * @see com.surveymonkey.surveymonkeyandroidsdk.utils.SMError
     */
    public void onStart(final Activity activity, String appName, final int requestCode, final String collectorHash, JSONObject... customVariables) {
        Resources res = activity.getResources();
        onStart(activity, requestCode, collectorHash, String.format(res.getString(R.string.prompt_title_text), appName), res.getString(R.string.prompt_message_text), THREE_DAYS, THREE_WEEKS, THREE_MONTHS, customVariables);
    }

    /**
     * Use the onStart method to present a customized AlertDialog prompt (at various intervals that you define) to the user encouraging them to take your in-app feedback survey.
     *
     * @param activity             The activity that will spawn the SMFeedbackActivity - <strong>Note:</strong> The spawning activity must implement onActivityResult to capture the respondent JSONObject or SMError object returned by SurveyMonkey
     * @param requestCode          If greater than or equal to 0, this code will be returned in onActivityResult() when the activity exits.
     * @param collectorHash        The collector hash that points to your Mobile SDK Collector - found on the Collector Details page of your chosen collector on SurveyMonkey.com
     *                             <p>Once the user completes your survey, a JSONObject containing the user's response will be returned to the spawning activity in onActivityResult
     *                             <p><strong>Note:</strong> If there are problems collecting the user's response, the SDK will return an SMError object with one of the domains and error codes listed in the SMError documentation
     * @param alertTitleText       The title of the survey prompt dialog displayed to the user
     * @param alertBodyText        The body of the survey prompt dialog displayed to the user
     * @param afterInstallInterval The amount of time (in milliseconds) to wait to first prompt the user to take your survey after they install your app
     * @param afterDeclineInterval The amount of time (in milliseconds) to wait to prompt the user to take your survey after they decline to take it
     * @param afterAcceptInterval  The amount of time (in milliseconds) to wait to prompt the user to take your survey after they consent to take it
     * @param customVariables (Optional) PLATINUM USERS ONLY: If you've set up custom variables in Create, you can pass a JSON object with the format {"key":"value",...} here.
     * @see com.surveymonkey.surveymonkeyandroidsdk.utils.SMError
     */
    public void onStart(final Activity activity, final int requestCode, final String collectorHash, final String alertTitleText, final String alertBodyText, final long afterInstallInterval, final long afterDeclineInterval, final long afterAcceptInterval, final JSONObject... customVariables) {
        final Context appContext = activity.getApplicationContext();
        SharedPreferences prefs = appContext.getSharedPreferences(SMConstants.PREF_NAME, Context.MODE_PRIVATE);
        final long currentDate = new Date().getTime();
        if (isNetworkConnected(appContext)) {
            long promptDate = prefs.getLong(SMConstants.PROMPT_DATE, 0);
            if (promptDate == 0) {
                prefs.edit().putLong(SMConstants.PROMPT_DATE, currentDate + afterInstallInterval).commit();
            } else if (promptDate < currentDate) {
                mCollectorHash = collectorHash;
                RetrieveSPageTask sPageTask = new RetrieveSPageTask() {
                    @Override
                    protected void onPostExecute(JSONObject data) {
                        try {
                            if (data != null) {
                                JSONObject sdkData = data.getJSONObject(SURVEY_STATUS);
                                mSPageHTML = data.getString(HTML);
                                if (!sdkData.getBoolean(COLLECTOR_CLOSED)) {
                                    AlertDialog.Builder builder = new AlertDialog.Builder(activity);
                                    LayoutInflater inflater = activity.getLayoutInflater();
                                    View view = inflater.inflate(R.layout.fragment_dialog, null);
                                    TextView headerTextView = (TextView) view.findViewById(R.id.dialog_header_text_line);
                                    headerTextView.setText(alertTitleText);
                                    headerTextView.setVisibility(View.VISIBLE);
                                    TextView messageTextView = (TextView) view.findViewById(R.id.dialog_first_text_line);
                                    messageTextView.setText(alertBodyText);
                                    messageTextView.setVisibility(View.VISIBLE);
                                    builder.setView(view);


                                    builder.setPositiveButton(R.string.action_give_feedback, new DialogInterface.OnClickListener() {
                                        public void onClick(DialogInterface dialog, int which) {
                                            SharedPreferences prefs = appContext.getSharedPreferences(SMConstants.PREF_NAME, Context.MODE_PRIVATE);
                                            prefs.edit().putLong(SMConstants.PROMPT_DATE, currentDate + afterAcceptInterval).commit();
                                            startSMFeedbackActivityForResult(activity, requestCode, collectorHash, customVariables);
                                        }
                                    });
                                    builder.setNegativeButton(R.string.action_not_now, new DialogInterface.OnClickListener() {
                                        public void onClick(DialogInterface dialog, int which) {
                                            SharedPreferences prefs = appContext.getSharedPreferences(SMConstants.PREF_NAME, Context.MODE_PRIVATE);
                                            prefs.edit().putLong(SMConstants.PROMPT_DATE, currentDate + afterDeclineInterval).commit();
                                        }
                                    });
                                    builder.create().show();
                                }
                            }
                        } catch (JSONException e) {
                            e.printStackTrace();
                        }
                    }
                };
                sPageTask.execute(buildURL());
            }
        }
        else {
            prefs.edit().putLong(SMConstants.PROMPT_DATE, currentDate + afterInstallInterval).commit();
        }
    }


    /**
     * Use the startSMFeedbackActivityForResult method to present a survey to the user.
     *
     * @param context       The activity that will spawn the SMFeedbackActivity - <strong>Note:</strong> The spawning activity must implement onActivityResult to capture the respondent JSONObject or SMError object returned by SurveyMonkey
     * @param requestCode   If greater than or equal to 0, this code will be returned in onActivityResult() when the activity exits.
     * @param collectorHash The collector hash that points to your Mobile SDK Collector - found on the Collector Details page of your chosen collector on SurveyMonkey.com
     *                      <p>Once the user completes your survey, a JSONObject containing the user's response will be returned to the spawning activity in onActivityResult
     *                      <p><strong>Note:</strong> If there are problems collecting the user's response, the SDK will return an SMError object with one of the domains and error codes listed in the SMError documentation
     * @param customVariables (Optional) PLATINUM USERS ONLY: If you've set up custom variables in Create, you can pass a JSON object with the format {"key":"value",...} here.
     * @see com.surveymonkey.surveymonkeyandroidsdk.utils.SMError
     */
    public void startSMFeedbackActivityForResult(final Activity context, int requestCode, String collectorHash, JSONObject... customVariables) {
        mContext = context;
        mRequestCode = requestCode;
        mCustomVariables = customVariables.length > 0 ? customVariables[0] : null;
        mCollectorHash = collectorHash;
        RetrieveSPageTask sPageTask = new RetrieveSPageTask() {
            @Override
            protected void onPostExecute(JSONObject data) {
                try {
                    if (data != null) {
                        JSONObject sdkData = data.getJSONObject(SURVEY_STATUS);
                        mSPageHTML = data.getString(HTML);
                        if (!sdkData.getBoolean(COLLECTOR_CLOSED)) {
                            startActivityForResult(mContext, mRequestCode, buildURL(), mSPageHTML);
                        } else {
                            //Collector is closed, so start activity with no html so we can provide an error
                            startActivityForResult(mContext, mRequestCode, buildURL(), null);
                        }
                    } else {
                        //Collector is closed, so start activity with no html so we can provide an error
                        startActivityForResult(mContext, mRequestCode, buildURL(), null);
                    }
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        };
        sPageTask.execute(buildURL());

    }

    private static boolean isNetworkConnected(Context context) {
        ConnectivityManager manager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetInfo = manager.getActiveNetworkInfo();
        return activeNetInfo != null && activeNetInfo.isConnected();
    }

    private String buildURL() {
        String url = SM_BASE_URL + mCollectorHash;
        if (mCustomVariables != null) {
            url += JSONToQueryString(mCustomVariables);
        }
        return url;
    }

    private String JSONToQueryString(JSONObject jsonObject) {
        String queryString = "?";
        ArrayList<String> params = new ArrayList<>();
        Iterator<String> keys = jsonObject.keys();

        while( keys.hasNext() ) {
            try {
                String key = keys.next();
                    if ( jsonObject.get(key) instanceof String ) {
                        params.add(String.format("%1$s=%2$s", key, jsonObject.getString(key)));
                    }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        queryString += TextUtils.join("&", params);
        return queryString;
    }

}
