package com.supcon.mes.nfc.controller;

import android.app.Activity;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.nfc.FormatException;
import android.nfc.NdefMessage;
import android.nfc.NdefRecord;
import android.nfc.NfcAdapter;
import android.nfc.Tag;
import android.nfc.tech.Ndef;
import android.nfc.tech.NdefFormatable;
import android.text.TextUtils;
import android.util.Log;

import com.google.gson.Gson;
import com.supcon.common.view.base.controller.BaseDataController;
import com.supcon.common.view.util.ToastUtils;
import com.supcon.mes.nfc.util.Byte2StringUtil;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * NFCController 高频NFC工具处理类
 * created by zhangwenshuai1 2019/2/27
 */
public class NFCController extends BaseDataController {

    private static final String TAG = "NFC_TAG";

    private NfcAdapter mNfcAdapter;
    private PendingIntent mPendingIntent;

    private OnNFCListener onNFCListener;

    public NFCController(Context context) {
        super(context);
    }

    @Override
    public void onInit() {
        super.onInit();

        setup(context);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
    }

    @Override
    public void initListener() {
        super.initListener();
    }

    @Override
    public void initData() {
        super.initData();
    }

    /**
     *  获取NFC适配器
     * @param context
     * @author zhangwenshuai1
     *  2018/6/27
     *
     */
    public void setup(Context context){

        if (!checkNFC(context)){
            return;
        }

        //创建PendingIntent对象,当检测到一个Tag标签就会执行此Intent; new Intent(context,context.getClass())自己的activity内接收intent
        if (mPendingIntent == null)
            mPendingIntent = PendingIntent.getActivity(context,0,new Intent(context,context.getClass()),0);
    }

    /**
     * 检测NFC功能,创建PendingIntent对象
     * @param context
     * @author zhangwenshuai1
     * 2018/6/27
     *
     */
    public boolean checkNFC(Context context){

        mNfcAdapter = NfcAdapter.getDefaultAdapter(context);

        //判断设备是否支持NFC功能
        if (mNfcAdapter == null){
            ToastUtils.show(context,"设备不支持NFC功能!",3000);
            return false;
        }

        //判断设备NFC功能是否打开
        if (!mNfcAdapter.isEnabled()){
            ToastUtils.show(context,"请到系统设置中打开NFC功能!",3000);
            return false;
        }

        return true;
    }

    /**
     * 拦截NFC,设置处理优于所有其他NFC的处理,呈现在用户界面的最前面
     * @param activity
     * @author zhangwenshuai1
     * 2018/6/27
     *
     */
    public void onResumeNFC(Activity activity){
        if (mNfcAdapter != null && mPendingIntent != null)
            mNfcAdapter.enableForegroundDispatch(activity, mPendingIntent, null, null);
    }

    /**
     * 恢复默认状态
     * @param activity
     * @author zhangwenshuai1
     * 2018/6/27
     *
     */
    public void onPauseNFC(Activity activity){
        if (mNfcAdapter != null)
            mNfcAdapter.disableForegroundDispatch(activity);
    }

    /**
     * 获取到Tag对象
     * @param intent
     * @author zhangwenshuai1
     * 2018/6/27
     *
     */
    public String dealNFCTag(Intent intent){
        String action = intent.getAction();
        if (TextUtils.isEmpty(action)) {
            return "获取标签信息失败!";
        }

        //1.获取Tag对象
        Tag tag = intent.getParcelableExtra(NfcAdapter.EXTRA_TAG);
        if (tag == null)
            return "获取标签对象为空!";

        //2.获取Ndef的实例
        String nfcMsg = "";
        if (action.equals(NfcAdapter.ACTION_NDEF_DISCOVERED)) {
            Ndef ndef = Ndef.get(tag);
            nfcMsg = fireNdefEvent(ndef);
        } else if (action.equals(NfcAdapter.ACTION_TECH_DISCOVERED)) {
            for (String tagTech : tag.getTechList()) {
                Log.d(TAG, tagTech);
                if (tagTech.equals(NdefFormatable.class.getName())) {
                    nfcMsg = fireTagEvent(tag);
                } else if (tagTech.equals(Ndef.class.getName())) {
                    Ndef ndef = Ndef.get(tag);
                    nfcMsg = fireNdefEvent(ndef);
                }
            }
        }
        if (action.equals(NfcAdapter.ACTION_TAG_DISCOVERED)) {
            nfcMsg = fireTagEvent(tag);
        }
        if (onNFCListener != null){
            onNFCListener.onNFCReceived(nfcMsg);
        }
        return nfcMsg;

    }

    /**
     * 解析Ndef信息
     * @param ndef
     * @author zhangwenshuai1
     * 2018/6/19
     */
    private String fireNdefEvent(Ndef ndef) {
        if (ndef != null) {
            try {
                ndef.connect();

                byte[] bytes = ndef.getTag().getId();
                String id = Byte2StringUtil.bytesToHexString(bytes);
                Log.d(TAG, id);

                NdefMessage ndefMessage = ndef.getNdefMessage();   //附加信息
                if (ndefMessage != null) {
                    NdefRecord[] ndefRecords = ndefMessage.getRecords();
                    String textRecord = "";
                    for (NdefRecord ndefRecord : ndefRecords) {

                        textRecord = parseTextRecord(ndefRecord);//读取写入的文本信息
                        //                    Log.d(TAG, textRecord);

                        //                byte[] payload = ndefRecord.getPayload();
                        //                String resultStr = new String(payload);  //会带有无需要的前两个字节
                        //                Log.d(TAG, resultStr);
                    }

                    Map<String,String> jsonObject = new HashMap<>();
                    jsonObject.put("id",id);
                    jsonObject.put("type",ndef.getType());
                    jsonObject.put("maxsize",ndef.getMaxSize() +" bytes");
                    jsonObject.put("textRecord",textRecord);

                    Gson gson = new Gson();
                    String nfc = gson.toJson(jsonObject);

                    return nfc;
                }

            } catch (IOException | FormatException e) {
                e.printStackTrace();
            } finally {
                try {
                    ndef.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return "";
    }

    /**
     * 解析tag信息
     * @param tag
     * @author zhangwenshuai1
     * 2018/6/19
     */
    public String fireTagEvent(Tag tag) {
        if (tag != null) {
            byte[] bytes = tag.getId();
            String id = Byte2StringUtil.bytesToHexString(bytes);
            Log.d(TAG, id);

            /*JSONObject jsonObject = Util.tagToJSON(tag);
            Log.d(TAG, jsonObject.toString());
            try {
                JSONArray jsonArray = jsonObject.getJSONArray("id");
                byte[] bytes1 = Util.jsonToByteArray(jsonArray);
                String message1 = bytesToHexString(bytes1);
                Log.d(TAG, message1);
            } catch (JSONException e) {
                e.printStackTrace();
            }*/
            String[] techList = tag.getTechList();
            List<String> techTypes = new ArrayList<>();
            for (String tech : techList){
                techTypes.add(tech);
            }

            Map<String,Object> jsonObject = new HashMap<>();
            jsonObject.put("id",id);
            jsonObject.put("techTypes",techTypes);

            Gson gson = new Gson();
            String nfc = gson.toJson(jsonObject);

            return nfc;

        }
        return "";
    }

    /**
     * 解析NDEF文本数据，从第三个字节开始，后面的文本数据
     *
     * @param ndefRecord
     * @return
     */
    private static String parseTextRecord(NdefRecord ndefRecord) {
        /**
         * 判断数据是否为NDEF格式
         */
        //判断TNF
        if (ndefRecord.getTnf() != NdefRecord.TNF_WELL_KNOWN) {
            return null;
        }
        //判断可变的长度的类型 注意：写入的是Text文本此处代码Arrays.equals(ndefRecord.getType(), NdefRecord.RTD_TEXT) = true，否则会return null
        if (!Arrays.equals(ndefRecord.getType(), NdefRecord.RTD_TEXT)) {
            return null;
        }
        try {
            //获得字节数组，然后进行分析
            byte[] payload = ndefRecord.getPayload();
            //下面开始NDEF文本数据第一个字节，状态字节
            //判断文本是基于UTF-8还是UTF-16的，取第一个字节"位与"上16进制的80，16进制的80也就是最高位是1，
            //其他位都是0，所以进行"位与"运算后就会保留最高位
            String textEncoding = ((payload[0] & 0x80) == 0) ? "UTF-8" : "UTF-16";
            //3f最高两位是0，第六位是1，所以进行"位与"运算后获得第六位
            int languageCodeLength = payload[0] & 0x3f;
            //下面开始NDEF文本数据第二个字节，语言编码
            //获得语言编码
            String languageCode = new String(payload, 1, languageCodeLength, "US-ASCII");
            //下面开始NDEF文本数据后面的字节，解析出文本
            String textRecord = new String(payload, languageCodeLength + 1,
                    payload.length - languageCodeLength - 1, textEncoding);
            return textRecord;
        } catch (Exception e) {
            throw new IllegalArgumentException();
        }
    }



    public void setOnNFCListener (OnNFCListener onNFCListener){
        this.onNFCListener = onNFCListener;
    }

    public interface OnNFCListener {
        void onNFCReceived(String nfc);
    }


}
