package com.supcon.common.view.view.picker.framework;

import android.content.Context;
import android.graphics.Color;
import android.support.annotation.ColorInt;
import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.StringRes;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.supcon.common.view.util.ConvertUtils;

/**
 * 带确定及取消按钮的弹窗
 *
 * @param <V> the type parameter
 */
@SuppressWarnings("WeakerAccess")
public abstract class ConfirmPopup<V extends View> extends BasicPopup<View> {
    /**
     * The Top line visible.
     */
    protected boolean topLineVisible = true;
    /**
     * The Top line color.
     */
    protected int topLineColor = 0xFF33B5E5;
    /**
     * The Top line height pixels.
     */
    protected int topLineHeightPixels = 1;//px
    /**
     * The Top background color.
     */
    protected int topBackgroundColor = Color.WHITE;
    /**
     * The Top height.
     */
    protected int topHeight = 40;//dp
    /**
     * The Top padding.
     */
    protected int topPadding = 15;//dp
    /**
     * The Content left and right padding.
     */
    protected int contentLeftAndRightPadding = 0;//dp
    /**
     * The Content top and bottom padding.
     */
    protected int contentTopAndBottomPadding = 0;//dp
    /**
     * The Cancel visible.
     */
    protected boolean cancelVisible = true;
    /**
     * The Cancel text.
     */
    protected CharSequence cancelText = "";
    /**
     * The Submit text.
     */
    protected CharSequence submitText = "";
    /**
     * The Title text.
     */
    protected CharSequence titleText = "";
    /**
     * The Cancel text color.
     */
    protected int cancelTextColor = 0xFF33B5E5;
    /**
     * The Submit text color.
     */
    protected int submitTextColor =  0xFF33B5E5;
    /**
     * The Title text color.
     */
    protected int titleTextColor = Color.BLACK;
    /**
     * The Pressed text color.
     */
    protected int pressedTextColor = 0XFF0288CE;
    /**
     * The Cancel text size.
     */
    protected int cancelTextSize = 0;
    /**
     * The Submit text size.
     */
    protected int submitTextSize = 0;
    /**
     * The Title text size.
     */
    protected int titleTextSize = 0;
    /**
     * The Background color.
     */
    protected int backgroundColor = Color.WHITE;
    /**
     * The Cancel button.
     */
    protected TextView cancelButton, /**
     * The Submit button.
     */
    submitButton, /**
     * The Swap button.
     */
    swapButton;
    /**
     * The Title view.
     */
    protected View titleView;
    /**
     * The Header view.
     */
    protected View headerView, /**
     * The Center view.
     */
    centerView, /**
     * The Footer view.
     */
    footerView;

    /**
     * Instantiates a new Confirm popup.
     *
     * @param activity the activity
     */
    public ConfirmPopup(Context activity) {
        super(activity);
        cancelText = activity.getString(android.R.string.cancel);
        submitText = activity.getString(android.R.string.ok);
    }

    /**
     * 设置顶部标题栏下划线颜色
     *
     * @param topLineColor the top line color
     */
    public void setTopLineColor(@ColorInt int topLineColor) {
        this.topLineColor = topLineColor;
    }

    /**
     * 设置顶部标题栏下划线高度，单位为px
     *
     * @param topLineHeightPixels the top line height pixels
     */
    public void setTopLineHeight(int topLineHeightPixels) {
        this.topLineHeightPixels = topLineHeightPixels;
    }

    /**
     * 设置顶部标题栏背景颜色
     *
     * @param topBackgroundColor the top background color
     */
    public void setTopBackgroundColor(@ColorInt int topBackgroundColor) {
        this.topBackgroundColor = topBackgroundColor;
    }

    /**
     * 设置顶部标题栏高度（单位为dp）
     *
     * @param topHeight the top height
     */
    public void setTopHeight(@IntRange(from = 10, to = 80) int topHeight) {
        this.topHeight = topHeight;
    }

    /**
     * 设置顶部按钮左边及右边边距（单位为dp）
     *
     * @param topPadding the top padding
     */
    public void setTopPadding(int topPadding) {
        this.topPadding = topPadding;
    }

    /**
     * 设置顶部标题栏下划线是否显示
     *
     * @param topLineVisible the top line visible
     */
    public void setTopLineVisible(boolean topLineVisible) {
        this.topLineVisible = topLineVisible;
    }

    /**
     * 设置内容上下左右边距（单位为dp）
     *
     * @param leftAndRight the left and right
     * @param topAndBottom the top and bottom
     */
    public void setContentPadding(int leftAndRight, int topAndBottom) {
        this.contentLeftAndRightPadding = leftAndRight;
        this.contentTopAndBottomPadding = topAndBottom;
    }

    /**
     * 设置顶部标题栏取消按钮是否显示
     *
     * @param cancelVisible the cancel visible
     */
    public void setCancelVisible(boolean cancelVisible) {
        if (null != cancelButton) {
            cancelButton.setVisibility(cancelVisible ? View.VISIBLE : View.GONE);
        } else {
            this.cancelVisible = cancelVisible;
        }
    }

    /**
     * 设置顶部标题栏取消按钮文字
     *
     * @param cancelText the cancel text
     */
    public void setCancelText(CharSequence cancelText) {
        if (null != cancelButton) {
            cancelButton.setText(cancelText);
        } else {
            this.cancelText = cancelText;
        }
    }

    /**
     * 设置顶部标题栏取消按钮文字
     *
     * @param textRes the text res
     */
    public void setCancelText(@StringRes int textRes) {
        setCancelText(activity.getString(textRes));
    }

    /**
     * 设置顶部标题栏确定按钮文字
     *
     * @param submitText the submit text
     */
    public void setSubmitText(CharSequence submitText) {
        if (null != submitButton) {
            submitButton.setText(submitText);
        } else {
            this.submitText = submitText;
        }
    }

    /**
     * 设置顶部标题栏确定按钮文字
     *
     * @param textRes the text res
     */
    public void setSubmitText(@StringRes int textRes) {
        setSubmitText(activity.getString(textRes));
    }

    /**
     * 设置顶部标题栏标题文字
     *
     * @param titleText the title text
     */
    public void setTitleText(CharSequence titleText) {
        if (titleView != null && titleView instanceof TextView) {
            ((TextView) titleView).setText(titleText);
        } else {
            this.titleText = titleText;
        }
    }

    /**
     * 设置顶部标题栏标题文字
     *
     * @param textRes the text res
     */
    public void setTitleText(@StringRes int textRes) {
        setTitleText(activity.getString(textRes));
    }

    /**
     * 设置顶部标题栏取消按钮文字颜色
     *
     * @param cancelTextColor the cancel text color
     */
    public void setCancelTextColor(@ColorInt int cancelTextColor) {
        if (null != cancelButton) {
            cancelButton.setTextColor(cancelTextColor);
        } else {
            this.cancelTextColor = cancelTextColor;
        }
    }

    /**
     * 设置顶部标题栏确定按钮文字颜色
     *
     * @param submitTextColor the submit text color
     */
    public void setSubmitTextColor(@ColorInt int submitTextColor) {
        if (null != submitButton) {
            submitButton.setTextColor(submitTextColor);
        } else {
            this.submitTextColor = submitTextColor;
        }
    }

    /**
     * 设置顶部标题栏标题文字颜色
     *
     * @param titleTextColor the title text color
     */
    public void setTitleTextColor(@ColorInt int titleTextColor) {
        if (null != titleView && titleView instanceof TextView) {
            ((TextView) titleView).setTextColor(titleTextColor);
        } else {
            this.titleTextColor = titleTextColor;
        }
    }

    /**
     * 设置按下时的文字颜色
     *
     * @param pressedTextColor the pressed text color
     */
    public void setPressedTextColor(int pressedTextColor) {
        this.pressedTextColor = pressedTextColor;
    }

    /**
     * 设置顶部标题栏取消按钮文字大小（单位为sp）
     *
     * @param cancelTextSize the cancel text size
     */
    public void setCancelTextSize(@IntRange(from = 10, to = 40) int cancelTextSize) {
        this.cancelTextSize = cancelTextSize;
    }

    /**
     * 设置顶部标题栏确定按钮文字大小（单位为sp）
     *
     * @param submitTextSize the submit text size
     */
    public void setSubmitTextSize(@IntRange(from = 10, to = 40) int submitTextSize) {
        this.submitTextSize = submitTextSize;
    }

    /**
     * 设置顶部标题栏标题文字大小（单位为sp）
     *
     * @param titleTextSize the title text size
     */
    public void setTitleTextSize(@IntRange(from = 10, to = 40) int titleTextSize) {
        this.titleTextSize = titleTextSize;
    }

    /**
     * 设置选择器主体背景颜色
     *
     * @param backgroundColor the background color
     */
    public void setBackgroundColor(@ColorInt int backgroundColor) {
        this.backgroundColor = backgroundColor;
    }

    /**
     * Sets title view.
     *
     * @param titleView the title view
     */
    public void setTitleView(View titleView) {
        this.titleView = titleView;
    }

    /**
     * Gets title view.
     *
     * @return the title view
     */
    public View getTitleView() {
        if (null == titleView) {
            throw new NullPointerException("please call show at first");
        }
        return titleView;
    }

    /**
     * Gets cancel button.
     *
     * @return the cancel button
     */
    public TextView getCancelButton() {
        if (null == cancelButton) {
            throw new NullPointerException("please call show at first");
        }
        return cancelButton;
    }

    /**
     * Gets submit button.
     *
     * @return the submit button
     */
    public TextView getSubmitButton() {
        if (null == submitButton) {
            throw new NullPointerException("please call show at first");
        }
        return submitButton;
    }

    /**
     * Sets header view.
     *
     * @param headerView the header view
     */
    public void setHeaderView(View headerView) {
        this.headerView = headerView;
    }

    /**
     * Sets footer view.
     *
     * @param footerView the footer view
     */
    public void setFooterView(View footerView) {
        this.footerView = footerView;
    }

    /**
     * @see #makeHeaderView()
     * @see #makeCenterView()
     * @see #makeFooterView()
     */
    @Override
    protected final View makeContentView() {
        LinearLayout rootLayout = new LinearLayout(activity);
        rootLayout.setLayoutParams(new LinearLayout.LayoutParams(MATCH_PARENT, MATCH_PARENT));
        rootLayout.setBackgroundColor(backgroundColor);
        rootLayout.setOrientation(LinearLayout.VERTICAL);
        rootLayout.setGravity(Gravity.CENTER);
        rootLayout.setPadding(10, 0, 10, 0);
        rootLayout.setClipToPadding(false);
        View headerView = makeHeaderView();
        if (headerView != null) {
            rootLayout.addView(headerView);
        }
        if (topLineVisible) {
            View lineView = new View(activity);
            LinearLayout.LayoutParams lp = new LinearLayout.LayoutParams(MATCH_PARENT, topLineHeightPixels);
            lp.topMargin = ConvertUtils.toPx(activity, 0);
            lp.bottomMargin = ConvertUtils.toPx(activity, 5);
            lineView.setLayoutParams(lp);
            lineView.setBackgroundColor(topLineColor);
            rootLayout.addView(lineView);
        }
        if (centerView == null) {
            centerView = makeCenterView();
        }
        int lr = 0;
        int tb = 0;
        if (contentLeftAndRightPadding > 0) {
            lr = ConvertUtils.toPx(activity, contentLeftAndRightPadding);
        }
        if (contentTopAndBottomPadding > 0) {
            tb = ConvertUtils.toPx(activity, contentTopAndBottomPadding);
        }
        centerView.setPadding(lr, tb, lr, tb);
        ViewGroup vg = (ViewGroup) centerView.getParent();
        if (vg != null) {
            //IllegalStateException: The specified child already has a parent
            vg.removeView(centerView);
        }
        rootLayout.addView(centerView, new LinearLayout.LayoutParams(MATCH_PARENT, 0, 1.0f));
        View footerView = makeFooterView();
        if (footerView != null) {
            rootLayout.addView(footerView);
        }
        return rootLayout;
    }

    /**
     * Make header view view.
     *
     * @return the view
     */
    @Nullable
    protected View makeHeaderView() {
        if (null != headerView) {
            return headerView;
        }
        RelativeLayout topButtonLayout = new RelativeLayout(activity);
        int height = ConvertUtils.toPx(activity, topHeight);
        int p = ConvertUtils.toPx(activity, 10);
        topButtonLayout.setLayoutParams(new RelativeLayout.LayoutParams(MATCH_PARENT, height));
        topButtonLayout.setBackgroundColor(topBackgroundColor);
        topButtonLayout.setGravity(Gravity.CENTER_VERTICAL);

        cancelButton = new TextView(activity);
        cancelButton.setVisibility(cancelVisible ? View.VISIBLE : View.GONE);
        RelativeLayout.LayoutParams cancelParams = new RelativeLayout.LayoutParams(WRAP_CONTENT, MATCH_PARENT);
        cancelParams.addRule(RelativeLayout.ALIGN_PARENT_LEFT, RelativeLayout.TRUE);
        cancelParams.addRule(RelativeLayout.CENTER_VERTICAL, RelativeLayout.TRUE);

        cancelButton.setLayoutParams(cancelParams);
        cancelButton.setBackgroundColor(Color.TRANSPARENT);
        cancelButton.setGravity(Gravity.CENTER);
        int padding = ConvertUtils.toPx(activity, topPadding);
        cancelButton.setPadding(padding, 0, padding, 0);
        if (!TextUtils.isEmpty(cancelText)) {
            cancelButton.setText(cancelText);
        }
        cancelButton.setTextColor(ConvertUtils.toColorStateList(cancelTextColor, pressedTextColor));
        if (cancelTextSize != 0) {
            cancelButton.setTextSize(cancelTextSize);
        }
        cancelButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dismiss();
                onCancel();
            }
        });
        topButtonLayout.addView(cancelButton);

        if (null == titleView) {
            TextView textView = new TextView(activity);
            RelativeLayout.LayoutParams titleParams = new RelativeLayout.LayoutParams(WRAP_CONTENT, WRAP_CONTENT);
            int margin = ConvertUtils.toPx(activity, topPadding);
            titleParams.leftMargin = margin;
            titleParams.rightMargin = margin;
            titleParams.addRule(RelativeLayout.CENTER_HORIZONTAL, RelativeLayout.TRUE);
            titleParams.addRule(RelativeLayout.CENTER_VERTICAL, RelativeLayout.TRUE);
            textView.setLayoutParams(titleParams);
            textView.setGravity(Gravity.CENTER);
            if (!TextUtils.isEmpty(titleText)) {
                textView.setText(titleText);
            }
            textView.setTextColor(titleTextColor);
            if (titleTextSize != 0) {
                textView.setTextSize(titleTextSize);
            }
            titleView = textView;
        }
        topButtonLayout.addView(titleView);

        submitButton = new TextView(activity);
        RelativeLayout.LayoutParams submitParams = new RelativeLayout.LayoutParams(WRAP_CONTENT, MATCH_PARENT);
        submitParams.addRule(RelativeLayout.ALIGN_PARENT_RIGHT, RelativeLayout.TRUE);
        submitParams.addRule(RelativeLayout.CENTER_VERTICAL, RelativeLayout.TRUE);
        submitButton.setLayoutParams(submitParams);
        submitButton.setBackgroundColor(Color.TRANSPARENT);
        submitButton.setGravity(Gravity.CENTER);
        submitButton.setPadding(padding, 0, padding, 0);
        if (!TextUtils.isEmpty(submitText)) {
            submitButton.setText(submitText);
        }
        submitButton.setTextColor(ConvertUtils.toColorStateList(submitTextColor, pressedTextColor));
        if (submitTextSize != 0) {
            submitButton.setTextSize(submitTextSize);
        }
        submitButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dismiss();
                onSubmit();
            }
        });
        topButtonLayout.addView(submitButton);

        return topButtonLayout;
    }

    /**
     * Make center view v.
     *
     * @return the v
     */
    @NonNull
    protected abstract V makeCenterView();

    /**
     * Make footer view view.
     *
     * @return the view
     */
    @Nullable
    protected View makeFooterView() {
        if (null != footerView) {
            return footerView;
        }
        return null;
    }

    /**
     * On submit.
     */
    protected void onSubmit() {

    }

    /**
     * On cancel.
     */
    protected void onCancel() {

    }

}
