/*
 * XWSSClientPipe.java
 *
 * Created on October 26, 2006, 12:53 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package com.sun.xml.xwss;

import com.sun.xml.ws.api.SOAPVersion;
import com.sun.xml.ws.api.WSBinding;
import com.sun.xml.ws.api.WSService;
import com.sun.xml.ws.api.message.Messages;
import com.sun.xml.ws.api.message.Packet;
import com.sun.xml.ws.api.model.wsdl.WSDLBoundOperation;
import com.sun.xml.ws.api.model.wsdl.WSDLPort;
import com.sun.xml.ws.api.pipe.Pipe;
import com.sun.xml.ws.api.pipe.PipeCloner;
import com.sun.xml.wss.ProcessingContext;
import com.sun.xml.wss.XWSSecurityException;
import com.sun.xml.wss.impl.MessageConstants;
import com.sun.xml.wss.impl.PolicyTypeUtil;
import com.sun.xml.wss.impl.PolicyViolationException;
import com.sun.xml.wss.impl.ProcessingContextImpl;
import com.sun.xml.wss.impl.SecurableSoapMessage;
import com.sun.xml.wss.impl.SecurityAnnotator;
import com.sun.xml.wss.impl.SecurityRecipient;
import com.sun.xml.wss.impl.WssSoapFaultException;
import com.sun.xml.wss.impl.XWSSecurityRuntimeException;
import com.sun.xml.wss.impl.config.ApplicationSecurityConfiguration;
import com.sun.xml.wss.impl.config.DeclarativeSecurityConfiguration;
import com.sun.xml.wss.impl.configuration.StaticApplicationContext;
import com.sun.xml.wss.impl.misc.SecurityUtil;
import com.sun.xml.wss.impl.policy.SecurityPolicy;
import java.net.URL;
import javax.xml.namespace.QName;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPConstants;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPFactory;
import javax.xml.soap.SOAPFault;
import javax.xml.soap.SOAPMessage;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.soap.SOAPFaultException;
import org.w3c.dom.Node;

/**
 * This pipe is used to handle XWSS 2.0 Style Message Security using
 * Security Configuration files
 * @author kumar.jayanti
 */
public class XWSSClientPipe  implements Pipe   {
    
    protected Pipe nextPipe = null;
    protected WSDLPort port = null;
    protected WSService service = null;
    protected WSBinding binding =null;
    
    protected  SOAPFactory soapFactory = null;
    protected  MessageFactory messageFactory = null;
    protected SOAPVersion soapVersion = null;
    protected boolean isSOAP12 = false;
    
    private static final String MESSAGE_SECURITY_CONFIGURATION =
            SecurityConfiguration.MESSAGE_SECURITY_CONFIGURATION;
    
    private static final String CONTEXT_WSDL_OPERATION =
            "com.sun.xml.ws.wsdl.operation";
    
    /** Creates a new instance of XWSSClientPipe */
    public XWSSClientPipe(WSDLPort prt, WSService svc, WSBinding bnd, Pipe nextP) {
       
        nextPipe = nextP;
        port = prt;
        service = svc;
        binding = bnd;
        
        soapVersion = bnd.getSOAPVersion();
        isSOAP12 = (soapVersion == SOAPVersion.SOAP_12) ? true : false;
        soapFactory = soapVersion.saajSoapFactory;
        messageFactory = soapVersion.saajMessageFactory;
        
    }
    
    public XWSSClientPipe(XWSSClientPipe that) {
        
        this.nextPipe = that.nextPipe;
        this.binding = that.binding;
        this.port = that.port;
        this.service = that.service;
        this.soapFactory = that.soapFactory;
        this.messageFactory = that.messageFactory;
        this.soapVersion = that.soapVersion;
        this.isSOAP12 = that.isSOAP12;
        
    }

    public Packet process(Packet packet) {
        SecurityConfiguration sConfig =
                (SecurityConfiguration)packet.invocationProperties.get(MESSAGE_SECURITY_CONFIGURATION);
        
        if (sConfig == null) {
            //No Security case here...
            //now look for "client_security_config.xml file from META-INF/ classpath"
            String configUrl = "META-INF/client_security_config.xml";
            URL url = SecurityUtil.loadFromClasspath(configUrl);
            if (url != null) {
                try {
                    sConfig = new SecurityConfiguration(url);
                    packet.invocationProperties.put(MESSAGE_SECURITY_CONFIGURATION,sConfig);
                } catch (XWSSecurityException e) {
                    throw new XWSSecurityRuntimeException(e);
                }
            } else {
                return nextPipe.process(packet);
            }
        }
        
        Packet ret = secureRequest(packet);
        ret =  nextPipe.process(ret);
        //could be one-way
        if (ret == null || ret.getMessage() == null) {
            return ret;
        }
        return validateResponse(ret);
    }

    public void preDestroy() {
    }

    public Pipe copy(PipeCloner pipeCloner) {
        Pipe clonedNextPipe = null;
        if (nextPipe != null) {
            clonedNextPipe = pipeCloner.copy(nextPipe);
        }
        Pipe copied = new XWSSClientPipe(this);
        ((XWSSClientPipe)copied).setNextPipe(clonedNextPipe);
        pipeCloner.add(this, copied);
        return copied;
    }
    
    public void setNextPipe(Pipe pipe) {
        nextPipe = pipe;
    }

    // client side incoming request handling code
    public Packet validateResponse(Packet packet) {
        try {
            SecurityConfiguration sConfig = (SecurityConfiguration)
            packet.invocationProperties.get(MESSAGE_SECURITY_CONFIGURATION);
            
            if (sConfig == null) {
                return packet;
            }
            SOAPMessage message = null;
            try {
                message =  packet.getMessage().readAsSOAPMessage();
            } catch (SOAPException ex) {
                throw new WebServiceException(ex);
            }
            
            String operation = (String)packet.invocationProperties.get(CONTEXT_WSDL_OPERATION);
            
            StaticApplicationContext sContext =
                    getPolicyContext(packet, sConfig);
            sContext.setOperationIdentifier(operation);
            
            ApplicationSecurityConfiguration config =
                    sConfig.getSecurityConfiguration();
            
            SecurityPolicy policy = config.getSecurityConfiguration(sContext);
            
            ProcessingContext context = new ProcessingContextImpl(packet.invocationProperties);
            context.setPolicyContext(sContext);
            context.setSOAPMessage(message);
            
            if (PolicyTypeUtil.declarativeSecurityConfiguration(policy)) {
                context.setSecurityPolicy(
                        ((DeclarativeSecurityConfiguration)policy).
                        receiverSettings());
            } else {
                context.setSecurityPolicy(policy);
            }
            
            context.setSecurityEnvironment(sConfig.getSecurityEnvironment());
            context.isInboundMessage(true);
            
            if (config.retainSecurityHeader()) {
                context.retainSecurityHeader(true);
            }
            
            SecurityRecipient.validateMessage(context);
            
            /* TODO: not sure if this is needed
            if (messageContext.get("javax.security.auth.Subject") != null) {
                messageContext.setScope("javax.security.auth.Subject", MessageContext.Scope.APPLICATION); 
            }*/
            packet.setMessage(Messages.create(context.getSOAPMessage()));
            return packet;
            
        } catch (com.sun.xml.wss.impl.WssSoapFaultException soapFaultException) {
            throw getSOAPFaultException(soapFaultException, isSOAP12);
        } catch (com.sun.xml.wss.XWSSecurityException xwse) {
            QName qname = null;
            
            if (xwse.getCause() instanceof PolicyViolationException)
                qname = MessageConstants.WSSE_RECEIVER_POLICY_VIOLATION;
            else
                qname = MessageConstants.WSSE_FAILED_AUTHENTICATION;
            
            com.sun.xml.wss.impl.WssSoapFaultException wsfe =
                    SecurableSoapMessage.newSOAPFaultException(
                    qname, xwse.getMessage(), xwse);
            throw getSOAPFaultException(wsfe, isSOAP12);
        }
    }
    
    // client side request sending hook
    public Packet secureRequest(Packet packet) {
        
        ProcessingContext context = null;
        SOAPMessage message = null;
        try {
            message = packet.getMessage().readAsSOAPMessage();
        }catch (SOAPException ex) {
            throw new WebServiceException(ex);
        }
        
        try {
            //TODO: whether property on BindingProvider.RequestContext is available here
            SecurityConfiguration sConfig = 
                    (SecurityConfiguration)packet.invocationProperties.get(MESSAGE_SECURITY_CONFIGURATION);
            
            if (sConfig == null) {
                return packet;
            }
            
            WSDLBoundOperation op =  null;
            if (port != null) {
                op = packet.getMessage().getOperation(port);
            }        
            
            QName operationQName =  null;
            if (op != null) {
                operationQName = op.getName();
            }
            
            String operation =  null;
            try {
                if (operationQName == null) {
                    operation = getOperationName(message);
                } else {
                    operation = operationQName.toString();
                }
            } catch (XWSSecurityException e) {
                throw new WebServiceException(e);
            }

            packet.invocationProperties.put(CONTEXT_WSDL_OPERATION, operation);
            
            StaticApplicationContext sContext =
                    getPolicyContext(packet, sConfig);
            sContext.setOperationIdentifier(operation);
            
            ApplicationSecurityConfiguration config =
                    sConfig.getSecurityConfiguration();
            
            SecurityPolicy policy = config.getSecurityConfiguration(sContext);
            
            context = new ProcessingContextImpl(packet.invocationProperties);
            
            context.setPolicyContext(sContext);
            
            if (PolicyTypeUtil.declarativeSecurityConfiguration(policy)) {
                context.setSecurityPolicy(
                        ((DeclarativeSecurityConfiguration)policy).senderSettings());
            } else {
                context.setSecurityPolicy(policy);
            }
            
            context.setSecurityEnvironment(sConfig.getSecurityEnvironment());
            context.isInboundMessage(false);
            context.setSOAPMessage(message);
            SecurityAnnotator.secureMessage(context);
            packet.setMessage(Messages.create(context.getSOAPMessage()));
            return packet;
            
        } catch (com.sun.xml.wss.impl.WssSoapFaultException soapFaultException) {
            throw new WebServiceException(soapFaultException);
        } catch (com.sun.xml.wss.XWSSecurityException xwse) {
            // log the exception here
            throw new WebServiceException(xwse);
        }
        
    }
    
    
    private String getOperationName(SOAPMessage message) throws XWSSecurityException {
        Node node = null;
        String key = null;
        SOAPBody body = null;
        
        if (message != null) {
            try {
                body = message.getSOAPBody();
            } catch  (SOAPException ex) {
                throw new XWSSecurityException(ex);       
            }
        } else {
            throw new XWSSecurityException(
                    "SOAPMessage in message context is null");
        }
        
        if (body != null)
            node = body.getFirstChild();
        else
            throw new XWSSecurityException(
                    "No body element identifying an operation is found");
        
        StringBuffer tmp = new StringBuffer("");
        String operation = "";
        
        for (; node != null; node = node.getNextSibling())
            tmp.append("{" + node.getNamespaceURI() + "}" + node.getLocalName() + ":");
        operation = tmp.toString();
        if(operation.length()> 0){
            return operation.substring(0, operation.length()-1);
        }else{
            return operation;
        }
    }

    public SOAPFaultException getSOAPFaultException(
            WssSoapFaultException sfe, boolean isSOAP12) {
        
        SOAPFault fault = null;
        try {
            if (isSOAP12) {
                fault = soapFactory.createFault(sfe.getFaultString(),SOAPConstants.SOAP_SENDER_FAULT);
                
                fault.appendFaultSubcode(sfe.getFaultCode());
            } else {
                fault = soapFactory.createFault(sfe.getFaultString(), sfe.getFaultCode());
            }
        } catch (Exception e) {
            throw new RuntimeException(this + ": Internal Error while trying to create a SOAPFault");
        }
        return new SOAPFaultException(fault);
    }

    
     private StaticApplicationContext getPolicyContext(Packet packet, SecurityConfiguration config) {
    
        // assumed to contain single nested container
        ApplicationSecurityConfiguration appconfig =
                config.getSecurityConfiguration();
        
        StaticApplicationContext iContext =
                (StaticApplicationContext)appconfig.getAllContexts().next();
        StaticApplicationContext sContext =
                new StaticApplicationContext(iContext);
        
        QName portQname = null;
        if (port != null) {
            portQname = port.getName();
        }
        String prt = null;
        
        if (portQname == null) {
            prt = "";
        } else {
            prt = portQname.toString();
        }
        
        sContext.setPortIdentifier(prt);
        return sContext;
    }
    
}
