/*
 * Copyright (c) 1997, 2023 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Distribution License v. 1.0, which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

package com.sun.tools.ws.wsdl.document;

import com.sun.tools.ws.api.wsdl.TWSDLExtensible;
import com.sun.tools.ws.api.wsdl.TWSDLExtension;
import com.sun.tools.ws.wsdl.framework.*;
import com.sun.tools.ws.wscompile.ErrorReceiver;
import org.xml.sax.Locator;

import javax.xml.namespace.QName;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Entity corresponding to the "service" WSDL element.
 *
 * @author WS Development Team
 */
@SuppressWarnings({"deprecation"})
public class Service extends GlobalEntity implements TWSDLExtensible {

    public Service(Defining defining, Locator locator, ErrorReceiver errReceiver) {
        super(defining, locator, errReceiver);
        _ports = new ArrayList<>();
        _helper = new ExtensibilityHelper();
    }

    public void add(Port port) {
        port.setService(this);
        _ports.add(port);
    }

    public Iterator <Port> ports() {
        return _ports.iterator();
    }

    @Override
    public Kind getKind() {
        return Kinds.SERVICE;
    }

    @Override
    public QName getElementName() {
        return WSDLConstants.QNAME_SERVICE;
    }

    public Documentation getDocumentation() {
        return _documentation;
    }

    public void setDocumentation(Documentation d) {
        _documentation = d;
    }

    @Override
    public void withAllSubEntitiesDo(EntityAction action) {
        for (Port port : _ports) {
            action.perform(port);
        }
        _helper.withAllSubEntitiesDo(action);
    }

    public void accept(WSDLDocumentVisitor visitor) throws Exception {
        visitor.preVisit(this);
        for (Port port : _ports) {
            port.accept(visitor);
        }
        _helper.accept(visitor);
        visitor.postVisit(this);
    }

    @Override
    public void validateThis() {
        if (getName() == null) {
            failValidation("validation.missingRequiredAttribute", "name");
        }
    }

    @Override
    public String getNameValue() {
        return getName();
    }

    @Override
    public String getNamespaceURI() {
        return getDefining().getTargetNamespaceURI();
    }

    @Override
    public QName getWSDLElementName() {
        return getElementName();
    }

    @Override
    public void addExtension(TWSDLExtension e) {
        _helper.addExtension(e);
    }

    @Override
    public Iterable<TWSDLExtension> extensions() {
        return _helper.extensions();
    }

    @Override
    public TWSDLExtensible getParent() {
        return null;  //To change body of implemented methods use File | Settings | File Templates.
    }

    private ExtensibilityHelper _helper;
    private Documentation _documentation;
    private List <Port> _ports;
}
