package com.stripe.android.stripe3ds2.transaction

import android.content.Intent
import android.os.Bundle
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import com.stripe.android.stripe3ds2.observability.DefaultErrorReporter
import com.stripe.android.stripe3ds2.observability.Stripe3ds2ErrorReporterConfig
import com.stripe.android.stripe3ds2.views.ChallengeActivity
import com.stripe.android.stripe3ds2.views.ChallengeViewArgs
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * An Activity that makes the initial challenge request. If successful, will start
 * [ChallengeActivity] using [ChallengeContract]. Will always finish with a [ChallengeResult]
 * result.
 *
 * This Activity does not show UI.
 */
internal class TransactionActivity : AppCompatActivity() {

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val args = runCatching {
            requireNotNull(TransactionContract.Args.fromIntent(intent))
        }.getOrElse {
            finishWithResult(
                ChallengeResult.RuntimeError(
                    it,
                    null,
                    IntentData.EMPTY
                )
            )
            return
        }

        val errorReporter = DefaultErrorReporter(
            application,
            Stripe3ds2ErrorReporterConfig(args.sdkTransactionId),
            Dispatchers.IO,
            Logger.get(args.enableLogging)
        )

        val viewModel by viewModels<TransactionViewModel> {
            TransactionViewModelFactory(
                args.sdkTransactionId,
                args.sdkReferenceNumber,
                args.isLiveMode,
                args.challengeParameters,
                args.rootCerts,
                args.sdkKeyPair,
                Logger.get(args.enableLogging),
                errorReporter,
                Dispatchers.IO
            )
        }

        val challengeLauncher = registerForActivityResult(ChallengeContract()) {
            finishWithResult(it)
        }

        lifecycleScope.launch {
            when (val challengeRequestResult = viewModel.startChallenge()) {
                is ChallengeRequestResult.Success -> {
                    // don't invoke `challengeLauncher` if the Activity is finishing - this means
                    // that `finishWithResult` was called already.
                    if (!isFinishing) {
                        val creqExecutorConfig = challengeRequestResult.creqExecutorConfig
                        challengeLauncher.launch(
                            ChallengeViewArgs(
                                challengeRequestResult.cresData,
                                challengeRequestResult.creqData,
                                args.uiCustomization,
                                creqExecutorConfig,
                                StripeChallengeRequestExecutor.Factory(creqExecutorConfig),
                                args.timeoutMins,
                                args.intentData
                            )
                        )
                    }
                }
                is ChallengeRequestResult.ProtocolError -> {
                    finishWithResult(
                        ChallengeResult.ProtocolError(
                            challengeRequestResult.data,
                            null,
                            args.intentData
                        )
                    )
                }
                is ChallengeRequestResult.RuntimeError -> {
                    finishWithResult(
                        ChallengeResult.RuntimeError(
                            challengeRequestResult.throwable,
                            null,
                            args.intentData
                        )
                    )
                }
                is ChallengeRequestResult.Timeout -> {
                    finishWithResult(
                        ChallengeResult.Timeout(
                            null,
                            null,
                            args.intentData
                        )
                    )
                }
            }
        }
    }

    private fun finishWithResult(
        challengeResult: ChallengeResult
    ) {
        setResult(
            RESULT_OK,
            Intent().putExtras(challengeResult.toBundle())
        )
        finish()
    }
}
