package com.stripe.android.stripe3ds2.transaction

import android.app.Activity
import android.app.Dialog
import com.nimbusds.jose.JOSEException
import com.stripe.android.stripe3ds2.exceptions.InvalidInputException
import com.stripe.android.stripe3ds2.init.ui.StripeUiCustomization
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.security.MessageTransformer
import com.stripe.android.stripe3ds2.transactions.ChallengeRequestData
import com.stripe.android.stripe3ds2.transactions.ChallengeResponseData
import com.stripe.android.stripe3ds2.transactions.ErrorData
import com.stripe.android.stripe3ds2.views.ProgressViewFactory
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.json.JSONException
import java.security.KeyPair
import java.security.PublicKey
import java.security.cert.CertificateException
import java.security.cert.X509Certificate
import java.text.ParseException

internal class StripeTransaction(
    private val areqParamsFactory: AuthenticationRequestParametersFactory,
    private val progressViewFactory: ProgressViewFactory,
    private val challengeStatusReceiverProvider: ChallengeStatusReceiverProvider,
    private val messageVersionRegistry: MessageVersionRegistry,
    private val sdkReferenceNumber: String,
    private val jwsValidator: JwsValidator,
    private val protocolErrorEventFactory: ProtocolErrorEventFactory,
    private val directoryServerId: String,
    private val directoryServerPublicKey: PublicKey,
    private val directoryServerKeyId: String?,
    override val sdkTransactionId: SdkTransactionId,
    private val sdkKeyPair: KeyPair,
    private val isLiveMode: Boolean,
    private val rootCerts: List<X509Certificate>,
    private val messageTransformer: MessageTransformer,
    private val uiCustomization: StripeUiCustomization?,
    private val brand: ProgressViewFactory.Brand,
    private val logger: Logger,
    private val errorReporter: ErrorReporter
) : Transaction {
    override var initialChallengeUiType: String? = null

    private val acsDataParser = AcsDataParser(errorReporter)

    override suspend fun createAuthenticationRequestParameters(): AuthenticationRequestParameters {
        return areqParamsFactory.create(
            directoryServerId,
            directoryServerPublicKey,
            directoryServerKeyId,
            sdkTransactionId,
            sdkKeyPair.public
        )
    }

    override suspend fun doChallenge(
        activity: Activity,
        challengeParameters: ChallengeParameters,
        challengeStatusReceiver: ChallengeStatusReceiver,
        timeoutMins: Int
    ) {
        doChallenge(
            host = Stripe3ds2ActivityStarterHost(activity),
            challengeParameters = challengeParameters,
            challengeStatusReceiver = challengeStatusReceiver,
            timeoutMins = timeoutMins
        )
    }

    override suspend fun doChallenge(
        host: Stripe3ds2ActivityStarterHost,
        challengeParameters: ChallengeParameters,
        challengeStatusReceiver: ChallengeStatusReceiver,
        timeoutMins: Int
    ) {
        logger.info("Starting challenge flow.")

        runCatching {
            if (timeoutMins < MIN_TIMEOUT) {
                throw InvalidInputException("Timeout must be at least 5 minutes")
            }

            challengeStatusReceiverProvider.put(sdkTransactionId, challengeStatusReceiver)

            // will throw exception if acsSignedContent fails verification
            val (acsUrl, acsEphemPubKey) = getAcsData(
                requireNotNull(challengeParameters.acsSignedContent),
                isLiveMode,
                rootCerts
            )

            val creqData = createCreqData(challengeParameters)

            val errorRequestExecutor =
                StripeErrorRequestExecutor.Factory().create(acsUrl, errorReporter)

            val creqExecutorConfig = ChallengeRequestExecutor.Config(
                messageTransformer,
                sdkReferenceNumber,
                sdkKeyPair.private.encoded,
                acsEphemPubKey.encoded,
                acsUrl,
                creqData
            )

            val result =
                StripeChallengeRequestExecutor.Factory()
                    .create(creqExecutorConfig, errorReporter)
                    .execute(creqData)

            onResult(
                result,
                host,
                creqExecutorConfig,
                challengeStatusReceiver,
                errorRequestExecutor,
                timeoutMins
            )
        }.onFailure {
            errorReporter.reportError(it)
            logger.error("Exception during challenge flow.", it)

            challengeStatusReceiver.runtimeError(RuntimeErrorEvent(it))
        }
    }

    private suspend fun onResult(
        result: ChallengeRequestResult,
        host: Stripe3ds2ActivityStarterHost,
        creqExecutorConfig: ChallengeRequestExecutor.Config,
        challengeStatusReceiver: ChallengeStatusReceiver,
        errorRequestExecutor: ErrorRequestExecutor,
        timeoutMins: Int
    ) {
        when (result) {
            is ChallengeRequestResult.Success -> onResultSuccess(
                result.creqData,
                result.cresData,
                host,
                creqExecutorConfig,
                timeoutMins
            )
            is ChallengeRequestResult.ProtocolError -> onProtocolErrorResult(
                result.data,
                challengeStatusReceiver,
                errorRequestExecutor
            )
            is ChallengeRequestResult.RuntimeError -> onRuntimeErrorResult(
                result.throwable,
                challengeStatusReceiver
            )
            is ChallengeRequestResult.Timeout -> onTimeoutResult(
                result.data,
                challengeStatusReceiver,
                errorRequestExecutor
            )
        }
    }

    private suspend fun onResultSuccess(
        creqData: ChallengeRequestData,
        cresData: ChallengeResponseData,
        host: Stripe3ds2ActivityStarterHost,
        creqExecutorConfig: ChallengeRequestExecutor.Config,
        timeoutMins: Int
    ) {
        handleChallengeResponse(
            host,
            creqData,
            cresData,
            uiCustomization!!,
            creqExecutorConfig,
            timeoutMins
        )
    }

    private fun onProtocolErrorResult(
        data: ErrorData,
        challengeStatusReceiver: ChallengeStatusReceiver,
        errorRequestExecutor: ErrorRequestExecutor
    ) {
        sendErrorData(errorRequestExecutor, data)

        challengeStatusReceiver.protocolError(
            protocolErrorEventFactory.create(data)
        )
    }

    private fun onTimeoutResult(
        data: ErrorData,
        challengeStatusReceiver: ChallengeStatusReceiver,
        errorRequestExecutor: ErrorRequestExecutor
    ) {
        sendErrorData(errorRequestExecutor, data)

        challengeStatusReceiver.runtimeError(RuntimeErrorEvent(data))
    }

    private fun onRuntimeErrorResult(
        throwable: Throwable,
        challengeStatusReceiver: ChallengeStatusReceiver
    ) {
        challengeStatusReceiver.runtimeError(
            RuntimeErrorEvent(throwable)
        )
    }

    @Throws(InvalidInputException::class)
    override fun getProgressView(currentActivity: Activity): Dialog {
        return progressViewFactory.create(currentActivity, brand, uiCustomization!!)
    }

    override fun close() {}

    @Throws(
        ParseException::class, JOSEException::class, JSONException::class,
        CertificateException::class
    )
    private fun getAcsData(
        acsSignedContent: String,
        isLiveMode: Boolean,
        rootCerts: List<X509Certificate>
    ): AcsData {
        return acsDataParser.parse(
            jwsValidator.getPayload(acsSignedContent, isLiveMode, rootCerts)
        )
    }

    private suspend fun handleChallengeResponse(
        host: Stripe3ds2ActivityStarterHost,
        creqData: ChallengeRequestData,
        cresData: ChallengeResponseData,
        uiCustomization: StripeUiCustomization,
        creqExecutorConfig: ChallengeRequestExecutor.Config,
        timeoutMins: Int
    ) = withContext(Dispatchers.Main) {
        initialChallengeUiType = cresData.uiType?.code

        // start the initial challenge screen
        ChallengeStarter(
            host,
            creqData,
            cresData,
            uiCustomization,
            creqExecutorConfig,
            timeoutMins
        ).start()
    }

    private fun sendErrorData(
        errorRequestExecutor: ErrorRequestExecutor,
        errorData: ErrorData
    ) {
        errorRequestExecutor.executeAsync(errorData)
    }

    private fun createCreqData(
        challengeParameters: ChallengeParameters
    ) = ChallengeRequestData(
        acsTransId = requireNotNull(challengeParameters.acsTransactionId),
        threeDsServerTransId = requireNotNull(challengeParameters.threeDsServerTransactionId),
        sdkTransId = sdkTransactionId,
        messageVersion = messageVersionRegistry.current
    )

    private companion object {
        private const val MIN_TIMEOUT = 5
    }
}
