package com.stripe.android.stripe3ds2.views

import android.os.Bundle
import android.os.Parcelable
import androidx.core.os.bundleOf
import com.stripe.android.stripe3ds2.exceptions.InvalidInputException
import com.stripe.android.stripe3ds2.init.ui.StripeUiCustomization
import com.stripe.android.stripe3ds2.transaction.ChallengeRequestExecutor
import com.stripe.android.stripe3ds2.transaction.ErrorRequestExecutor
import com.stripe.android.stripe3ds2.transactions.ChallengeRequestData
import com.stripe.android.stripe3ds2.transactions.ChallengeResponseData
import kotlinx.parcelize.Parcelize

@Parcelize
internal data class ChallengeViewArgs(
    val cresData: ChallengeResponseData,
    val creqData: ChallengeRequestData,
    val uiCustomization: StripeUiCustomization,
    val creqExecutorConfig: ChallengeRequestExecutor.Config,
    val creqExecutorFactory: ChallengeRequestExecutor.Factory,
    val errorExecutorFactory: ErrorRequestExecutor.Factory,
    val timeoutMins: Int
) : Parcelable {
    internal val sdkTransactionId get() = creqData.sdkTransId

    /**
     * Convert the fields to a [Bundle]
     */
    fun toBundle() = bundleOf(
        EXTRA_CREQ_DATA to creqData,
        EXTRA_CRES_DATA to cresData,
        EXTRA_UI_CUSTOMIZATION to uiCustomization,
        EXTRA_CREQ_EXECUTOR_CONFIG to creqExecutorConfig,
        EXTRA_CREQ_EXECUTOR_FACTORY to creqExecutorFactory,
        EXTRA_ERROR_EXECUTOR_FACTORY to errorExecutorFactory,
        EXTRA_TIMEOUT to timeoutMins
    )

    companion object {
        internal const val EXTRA_CREQ_DATA = "extra_creq_data"
        internal const val EXTRA_CRES_DATA = "extra_cres_data"
        internal const val EXTRA_UI_CUSTOMIZATION = "extra_ui_customization"
        internal const val EXTRA_CREQ_EXECUTOR_CONFIG = "extra_creq_executor_config"
        internal const val EXTRA_CREQ_EXECUTOR_FACTORY = "extra_creq_executor_factory"
        internal const val EXTRA_ERROR_EXECUTOR_FACTORY = "extra_error_executor_factory"
        internal const val EXTRA_TIMEOUT = "extra_timeout"

        /**
         * Create a [ChallengeViewArgs] from Intent extras
         */
        fun create(extras: Bundle?): ChallengeViewArgs {
            if (extras == null) {
                throw InvalidInputException("Intent extras required")
            }

            val cresData =
                extras.getParcelable<ChallengeResponseData>(EXTRA_CRES_DATA)
                    ?: throw InvalidInputException("ChallengeResponseData is required")

            val creqData =
                extras.getSerializable(EXTRA_CREQ_DATA) as ChallengeRequestData

            val uiCustomization =
                extras.getParcelable<StripeUiCustomization>(EXTRA_UI_CUSTOMIZATION)
                    ?: throw InvalidInputException("UiCustomization is required")

            val creqExecutorConfig =
                extras.getSerializable(EXTRA_CREQ_EXECUTOR_CONFIG)
                    as ChallengeRequestExecutor.Config

            val creqExecutorFactory =
                extras.getSerializable(EXTRA_CREQ_EXECUTOR_FACTORY)
                    as ChallengeRequestExecutor.Factory

            val errorExecutorFactory =
                extras.getSerializable(EXTRA_ERROR_EXECUTOR_FACTORY) as ErrorRequestExecutor.Factory

            return ChallengeViewArgs(
                cresData,
                creqData,
                uiCustomization,
                creqExecutorConfig,
                creqExecutorFactory,
                errorExecutorFactory,
                extras.getInt(EXTRA_TIMEOUT, MIN_TIMEOUT)
            )
        }

        private const val MIN_TIMEOUT = 5
    }
}
