package com.stripe.android.stripe3ds2.views

import android.app.Application
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.liveData
import androidx.lifecycle.viewModelScope
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.transaction.ChallengeAction
import com.stripe.android.stripe3ds2.transaction.ChallengeActionHandler
import com.stripe.android.stripe3ds2.transaction.ChallengeRequestResult
import com.stripe.android.stripe3ds2.transaction.TransactionTimer
import com.stripe.android.stripe3ds2.transactions.ChallengeResponseData
import com.stripe.android.stripe3ds2.utils.ImageCache
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.launch
import kotlin.coroutines.CoroutineContext

internal class ChallengeActivityViewModel(
    application: Application,
    private val challengeActionHandler: ChallengeActionHandler,
    private val transactionTimer: TransactionTimer,
    errorReporter: ErrorReporter,
    private val imageCache: ImageCache = ImageCache.Default,
    workContext: CoroutineContext
) : AndroidViewModel(application) {
    private val densityDpi = application.resources.displayMetrics.densityDpi
    private val imageRepository = ImageRepository(errorReporter, workContext)

    private val _refreshUi = MutableLiveData<Unit>()
    val refreshUi: LiveData<Unit> = _refreshUi

    private val _submitClicked = MutableLiveData<ChallengeAction>()
    val submitClicked: LiveData<ChallengeAction> = _submitClicked

    private val _shouldFinish = MutableLiveData<Unit>()
    val shouldFinish: LiveData<Unit> = _shouldFinish

    private val _challengeText = MutableLiveData<String>()
    val challengeText: LiveData<String> = _challengeText

    private val _challengeRequestResult = OnInactiveAwareMutableLiveData<ChallengeRequestResult>()
    val challengeRequestResult: LiveData<ChallengeRequestResult> = _challengeRequestResult

    private val _nextScreen = OnInactiveAwareMutableLiveData<ChallengeResponseData>()
    val nextScreen: LiveData<ChallengeResponseData> = _nextScreen

    var shouldRefreshUi: Boolean = false

    internal val transactionTimerJob: Job

    init {
        transactionTimerJob = viewModelScope.launch {
            transactionTimer.start()
        }
    }

    fun getTimeout() = liveData {
        emit(
            transactionTimer.timeout.firstOrNull { isTimeout -> isTimeout }
        )
    }

    fun getImage(
        imageData: ChallengeResponseData.Image?
    ) = liveData {
        emit(
            imageRepository.getImage(
                imageData?.getUrlForDensity(densityDpi)
            )
        )
    }

    fun submit(action: ChallengeAction) {
        viewModelScope.launch {
            _challengeRequestResult.postValue(challengeActionHandler.submit(action))
        }
    }

    fun stopTimer() {
        transactionTimerJob.cancel()
    }

    fun onMemoryEvent() {
        imageCache.clear()
    }

    fun onRefreshUi() {
        _refreshUi.value = Unit
    }

    fun onSubmitClicked(challengeAction: ChallengeAction) {
        _submitClicked.value = challengeAction
    }

    fun onFinish() {
        _shouldFinish.postValue(Unit)
    }

    fun updateChallengeText(text: String) {
        _challengeText.value = text
    }

    fun onNextScreen(cres: ChallengeResponseData) {
        _nextScreen.value = cres
    }

    /**
     * A `MutableLiveData` that sets the value to `null` when `onInactive()` is invoked.
     */
    private class OnInactiveAwareMutableLiveData<T> : MutableLiveData<T>() {
        override fun onInactive() {
            super.onInactive()
            value = null
        }
    }

    internal class Factory(
        private val application: Application,
        private val challengeActionHandler: ChallengeActionHandler,
        private val transactionTimer: TransactionTimer,
        private val errorReporter: ErrorReporter,
        private val workContext: CoroutineContext
    ) : ViewModelProvider.Factory {
        override fun <T : ViewModel?> create(modelClass: Class<T>): T {

            return ChallengeActivityViewModel(
                application,
                challengeActionHandler,
                transactionTimer,
                errorReporter,
                workContext = workContext
            ) as T
        }
    }
}
