package com.stripe.android.stripe3ds2.transaction

import com.stripe.android.stripe3ds2.init.ui.StripeUiCustomization
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.security.EphemeralKeyPairGenerator
import com.stripe.android.stripe3ds2.security.MessageTransformerFactory
import com.stripe.android.stripe3ds2.views.ProgressViewFactory
import java.security.PublicKey
import java.security.cert.X509Certificate

/**
 * Factory to create a [Transaction] for
 * [com.stripe.android.stripe3ds2.service.ThreeDS2Service.createTransaction].
 */
internal fun interface TransactionFactory {
    fun create(
        directoryServerId: String,
        rootCerts: List<X509Certificate>,
        directoryServerPublicKey: PublicKey,
        keyId: String?,
        sdkTransactionId: SdkTransactionId,
        uiCustomization: StripeUiCustomization?,
        isLiveMode: Boolean,
        brand: ProgressViewFactory.Brand
    ): Transaction

    class Default @JvmOverloads internal constructor(
        private val areqParamsFactory: AuthenticationRequestParametersFactory,
        private val ephemeralKeyPairGenerator: EphemeralKeyPairGenerator,
        private val messageVersionRegistry: MessageVersionRegistry,
        private val sdkReferenceNumber: String,
        private val errorReporter: ErrorReporter,
        private val logger: Logger = Logger.noop(),
        private val progressViewFactory: ProgressViewFactory = ProgressViewFactory.Default(),
        private val jwsValidator: JwsValidator = JwsValidator.Default(errorReporter),
        private val challengeStatusReceiverProvider: ChallengeStatusReceiverProvider =
            ChallengeStatusReceiverProvider.Default
    ) : TransactionFactory {
        private val protocolErrorEventFactory = ProtocolErrorEventFactory()
        private val messageTransformerFactory = MessageTransformerFactory(errorReporter)

        override fun create(
            directoryServerId: String,
            rootCerts: List<X509Certificate>,
            directoryServerPublicKey: PublicKey,
            keyId: String?,
            sdkTransactionId: SdkTransactionId,
            uiCustomization: StripeUiCustomization?,
            isLiveMode: Boolean,
            brand: ProgressViewFactory.Brand
        ): Transaction {
            val sdkKeyPair = ephemeralKeyPairGenerator.generate()
            return StripeTransaction(
                areqParamsFactory,
                progressViewFactory,
                challengeStatusReceiverProvider,
                messageVersionRegistry,
                sdkReferenceNumber,
                jwsValidator,
                protocolErrorEventFactory,
                directoryServerId, directoryServerPublicKey, keyId, sdkTransactionId,
                sdkKeyPair, isLiveMode, rootCerts, messageTransformerFactory.create(isLiveMode),
                uiCustomization,
                brand,
                logger,
                errorReporter
            )
        }
    }
}
