package com.stripe.android.stripe3ds2.views

import android.app.Application
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.liveData
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.transaction.ChallengeAction
import com.stripe.android.stripe3ds2.transaction.ChallengeActionHandler
import com.stripe.android.stripe3ds2.transaction.TransactionTimer
import com.stripe.android.stripe3ds2.transactions.ChallengeResponseData
import kotlinx.coroutines.flow.firstOrNull
import kotlin.coroutines.CoroutineContext

internal class ChallengeActivityViewModel(
    application: Application,
    private val challengeActionHandler: ChallengeActionHandler,
    private val transactionTimer: TransactionTimer,
    errorReporter: ErrorReporter,
    workContext: CoroutineContext
) : AndroidViewModel(application) {
    private val densityDpi = application.resources.displayMetrics.densityDpi
    private val imageRepository = ImageRepository(errorReporter, workContext)

    fun getTimeout() = liveData {
        emit(
            transactionTimer.timeout.firstOrNull { isTimeout -> isTimeout }
        )
    }

    fun getImage(
        imageData: ChallengeResponseData.Image?
    ) = liveData {
        emit(
            imageRepository.getImage(
                imageData?.getUrlForDensity(densityDpi)
            )
        )
    }

    fun submit(action: ChallengeAction) = liveData {
        emit(challengeActionHandler.submit(action))
    }

    fun stopTimer() {
        transactionTimer.stop()
    }

    internal class Factory(
        private val application: Application,
        private val challengeActionHandler: ChallengeActionHandler,
        private val transactionTimer: TransactionTimer,
        private val errorReporter: ErrorReporter,
        private val workContext: CoroutineContext
    ) : ViewModelProvider.Factory {
        override fun <T : ViewModel?> create(modelClass: Class<T>): T {
            return ChallengeActivityViewModel(
                application,
                challengeActionHandler,
                transactionTimer,
                errorReporter,
                workContext
            ) as T
        }
    }
}
