package com.stripe.android.stripe3ds2.security

import com.nimbusds.jose.jwk.KeyUse
import com.stripe.android.stripe3ds2.exceptions.SDKRuntimeException

internal enum class DirectoryServer(
    val ids: List<String>,
    val algorithm: Algorithm,
    val fileName: String,
    val keyUse: KeyUse? = KeyUse.SIGNATURE
) {
    // public key for each algorithm defined in
    // "UL Test Harness Specification - UL 3DS Self Test Platform"
    TestRsa(
        listOf("F000000000"),
        Algorithm.RSA,
        "ds-test-rsa.txt"
    ),
    TestEc(
        listOf("F000000001"),
        Algorithm.EC,
        "ds-test-ec.txt"
    ),

    Visa(
        listOf("A000000003"),
        Algorithm.RSA,
        "ds-visa.crt"
    ),

    Mastercard(
        listOf("A000000004"),
        Algorithm.RSA,
        "ds-mastercard.crt"
    ),

    Amex(
        listOf("A000000025"),
        Algorithm.RSA,
        "ds-amex.pem"
    ),

    Discover(
        listOf("A000000152", "A000000324"),
        Algorithm.RSA,
        "ds-discover.cer",
        keyUse = null
    );

    val isCertificate: Boolean
        get() = CERTIFICATE_EXTENSIONS.any { fileName.endsWith(it) }

    companion object {
        private val CERTIFICATE_EXTENSIONS = setOf(".crt", ".cer", ".pem")

        fun lookup(directoryServerId: String): DirectoryServer {
            val directoryServer = values().firstOrNull {
                it.ids.contains(directoryServerId)
            }
            return directoryServer ?: throw SDKRuntimeException(
                IllegalArgumentException("Unknown directory server id: $directoryServerId")
            )
        }
    }
}
