/*
 * Decompiled with CFR 0.152.
 */
package com.sourceclear.util.io;

import com.google.common.base.Optional;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.io.ByteStreams;
import com.sourceclear.api.data.git.GitCommit;
import com.sourceclear.api.data.git.GitUser;
import com.sourceclear.api.data.git.MetaGit;
import com.sourceclear.api.data.git.RefType;
import com.sourceclear.util.config.RepoUtils;
import com.sourceclear.util.io.SrcclrIo;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.Proxy;
import java.net.URI;
import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.eclipse.jgit.annotations.NonNull;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.errors.RevWalkException;
import org.eclipse.jgit.errors.StopWalkException;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.BranchConfig;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.RepositoryBuilder;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.revwalk.filter.RevFilter;
import org.eclipse.jgit.transport.RemoteConfig;
import org.eclipse.jgit.transport.URIish;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class GitUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(GitUtils.class);
    protected static final long ONE_DAY_IN_MS = 86400000L;
    private static final Set<String> CI_BRANCHES = ImmutableSet.of((Object)"GIT_BRANCH", (Object)"bamboo_planRepository_branch");
    private static final ImmutableList<String> TRAVIS_BRANCHES_ORDER_PREFERENCE = ImmutableList.of((Object)"TRAVIS_PULL_REQUEST_BRANCH", (Object)"TRAVIS_BRANCH");
    private static final String TRAVIS_PULL_REQUEST_SLUG = "TRAVIS_PULL_REQUEST_SLUG";
    private static final String TRAVIS_REPO_SLUG = "TRAVIS_REPO_SLUG";
    private static final Set<String> CI_ORIGINS = ImmutableSet.of((Object)"GIT_URL", (Object)"bamboo_planRepository_repositoryUrl");
    public static final String GENERIC_SCM_ENV_PREFIX = "SRCCLR_SCM_";
    public static final String GITLESS_ENV_KEY = "__WITHOUT_GIT__";

    private static File goUpToGitRepo(File origFile) throws IOException {
        for (File file = origFile = origFile.getAbsoluteFile(); file != null; file = file.getParentFile()) {
            if (!file.isDirectory()) continue;
            File[] files = file.listFiles();
            if (files == null) {
                throw new IOException(String.format("Encountered an unexpected error searching for the .git metadata directory in %s", file));
            }
            for (File child : files) {
                if (!".git".equals(child.getName()) || !child.isDirectory() || !child.canRead()) continue;
                File[] childChildren = child.listFiles();
                if (childChildren == null) {
                    LOGGER.warn("Could not evaluate .git metadata directory candidate {} while searching from file {} due to a null file listing. Skipping that directory.", (Object)child, (Object)origFile);
                    continue;
                }
                for (File childChild : childChildren) {
                    if (!"HEAD".equals(childChild.getName())) continue;
                    return file;
                }
            }
        }
        throw new IOException(String.format("The file %s is not a git repo nor the child directory of a git repo.", origFile));
    }

    static Repository getRepoFromProjDir(@Nonnull File projectDir) throws IOException {
        File repoDir = GitUtils.goUpToGitRepo(projectDir);
        RepositoryBuilder repoBuilder = new RepositoryBuilder();
        repoBuilder.setWorkTree(repoDir);
        return repoBuilder.build();
    }

    public static boolean isOpenRepository(String urlIsh) {
        try {
            String location = GitUtils.convertGitToHttps(urlIsh);
            URL locationUrl = new URL(location);
            List<String> noProxyDomains = GitUtils.getNoProxyDomains();
            boolean noProxy = GitUtils.shouldNotProxyForUrl(locationUrl, noProxyDomains);
            HttpURLConnection conn = (HttpURLConnection)(noProxy ? locationUrl.openConnection(Proxy.NO_PROXY) : locationUrl.openConnection());
            conn.setRequestMethod("HEAD");
            conn.setConnectTimeout(10000);
            conn.setDoInput(true);
            conn.setDoOutput(false);
            int status = conn.getResponseCode();
            try (InputStream stream = conn.getInputStream();){
                ByteStreams.copy((InputStream)stream, (OutputStream)ByteStreams.nullOutputStream());
            }
            catch (FileNotFoundException ignored) {
                LOGGER.debug("{}.FNFE.RC={}", (Object)location, (Object)status);
                try (InputStream errorStream = conn.getErrorStream();){
                    ByteStreams.copy((InputStream)errorStream, (OutputStream)ByteStreams.nullOutputStream());
                }
            }
            conn.disconnect();
            return GitUtils.testHttpResponse(urlIsh, status);
        }
        catch (Exception ex) {
            LOGGER.debug(String.format("Couldn't fetch repo location for %s", urlIsh));
            return true;
        }
    }

    protected static String convertGitToHttps(String urlIsh) {
        if (urlIsh.startsWith("http")) {
            return urlIsh;
        }
        if (urlIsh.startsWith("git@")) {
            return urlIsh.replaceFirst(":", "/").replaceFirst("^git@", "https://");
        }
        if (urlIsh.startsWith("ssh:")) {
            String tmp = urlIsh.replaceFirst("^ssh:", "").replaceFirst("git@", "").replaceFirst(":", "/");
            tmp = "https:" + tmp;
            return tmp;
        }
        return "https://" + urlIsh;
    }

    private static boolean testHttpResponse(String urlish, int status) {
        if (status == 404 || status == 401) {
            LOGGER.debug("Repo doesn't appear public, a GET returned {} for {}", (Object)status, (Object)urlish);
            return false;
        }
        return true;
    }

    public static MetaGit getGitMetaData(@Nonnull File projectDir, @Nonnegative int days, Set<CommitInfoPerms> commitInfoPerms) throws Exception {
        return GitUtils.getGitMetaData(projectDir, days, System.getenv(), commitInfoPerms);
    }

    @Nonnull
    public static MetaGit getGitMetaData(@Nonnull File projectDir, @Nonnegative int days, @Nonnull Map<String, String> env, Set<CommitInfoPerms> commitInfoPerms) throws Exception {
        Optional<String> branchName;
        if (days < 0) {
            throw new IllegalArgumentException("Days must be greater than or equal to 0");
        }
        GitUtils.verifyOrThrow(projectDir);
        if (env.containsKey(GITLESS_ENV_KEY)) {
            String projectUri;
            String subPath;
            String refType;
            boolean haveRef = false;
            boolean haveRefName = false;
            boolean haveUri = false;
            MetaGit.Builder builder = new MetaGit.Builder();
            String ref = GenericScmEnvNames.REF.getOrNull(env);
            if (ref != null) {
                builder.withRefName(ref);
                haveRefName = true;
            }
            if ((refType = GenericScmEnvNames.REF_TYPE.getOrNull(env)) != null) {
                builder.withRefType(RefType.valueOf(refType.toUpperCase()));
            } else {
                builder.withRefType(RefType.COMMIT);
            }
            String rev = GenericScmEnvNames.REV.getOrNull(env);
            if (rev != null) {
                builder.withHead(rev);
                haveRef = true;
            }
            if ((subPath = GenericScmEnvNames.SUB_PATH.getOrNull(env)) != null) {
                builder.withSubPath(subPath);
            }
            if ((projectUri = GenericScmEnvNames.URI.getOrNull(env)) != null) {
                builder.withRemote(URI.create(projectUri));
                haveUri = true;
            }
            if (haveRef && haveRefName && haveUri) {
                builder.withLocalPath(projectDir.toURI());
                List<GitCommit> commitList = Collections.emptyList();
                builder.withCommits(commitList);
                MetaGit result = builder.build();
                LOGGER.debug("Short-circuiting MetaGit in favor of the environment, result = {}", (Object)result);
                return result;
            }
        }
        Repository repo = GitUtils.getRepoFromProjDir(projectDir);
        Path repoPath = repo.getDirectory().toPath();
        Path scanPath = projectDir.toPath();
        Path subPath = repoPath.getParent().toAbsolutePath().relativize(scanPath.toAbsolutePath());
        Pair<String, RefType> refPair = GitUtils.findRef(repo, env);
        URIish remoteUri = GitUtils.getRemoteUrl(repo, env);
        boolean isOpen = remoteUri != null && !"file".equals(remoteUri.getScheme()) && GitUtils.isOpenRepository(remoteUri.toString());
        List<GitCommit> commits = Collections.emptyList();
        if (!isOpen && !commitInfoPerms.contains((Object)CommitInfoPerms.SKIP_ALL)) {
            commits = GitUtils.getCommits(repo, (long)days * 86400000L, commitInfoPerms);
        }
        RefType refType = (RefType)((Object)refPair.getRight());
        String refName = (String)refPair.getLeft();
        URI remote = remoteUri == null ? null : URI.create(remoteUri.toString());
        if (refType == RefType.COMMIT && (branchName = GitUtils.tryGetBranchName(projectDir)).isPresent()) {
            refType = RefType.BRANCH;
            refName = (String)branchName.get();
        }
        return new MetaGit.Builder().withSubPath(subPath.toString()).withLocalPath(projectDir.toURI()).withRemote(remote).withHead(GitUtils.getCommitHash(repo)).withRefName(refName).withRefType(refType).withCommits(commits).withCommitHistory(days).build();
    }

    @Nonnull
    public static List<GitCommit> getCommits(@Nonnull Repository repo, final @Nonnegative long duration, Set<CommitInfoPerms> commitInfoPerms) throws IOException {
        LinkedList<GitCommit> commits = new LinkedList<GitCommit>();
        final long currentStamp = System.currentTimeMillis();
        RevWalk walk = new RevWalk(repo);
        walk.markStart(walk.parseCommit((AnyObjectId)repo.resolve("HEAD")));
        if (duration > 0L) {
            walk.setRevFilter(new RevFilter(){

                public boolean include(RevWalk revWalk, RevCommit revCommit) throws StopWalkException, IOException {
                    long commitStamp = (long)revCommit.getCommitTime() * 1000L;
                    return currentStamp - duration < commitStamp;
                }

                public RevFilter clone() {
                    return null;
                }
            });
        }
        try {
            boolean collectEmail = !commitInfoPerms.contains((Object)CommitInfoPerms.SKIP_EMAILS);
            boolean collectName = !commitInfoPerms.contains((Object)CommitInfoPerms.SKIP_NAMES);
            for (RevCommit revCommit : walk) {
                GitUser author = GitUtils.toGitUser(revCommit.getAuthorIdent(), collectEmail, collectName);
                GitUser committer = GitUtils.toGitUser(revCommit.getCommitterIdent(), collectEmail, collectName);
                commits.add(new GitCommit(author, committer, new Date((long)revCommit.getCommitTime() * 1000L), revCommit.getId().getName()));
            }
        }
        catch (RevWalkException ex) {
            LOGGER.info("Couldn't perform repo walk", (Throwable)ex);
        }
        Collections.reverse(commits);
        return commits;
    }

    @Deprecated
    @Nonnull
    public static Pair<String, String> parseRepoUrl(@Nullable String repoUrl) {
        return RepoUtils.parseRepoUrl(repoUrl);
    }

    @NonNull
    public static Pair<String, RefType> findRef(@Nonnull Repository repo, @Nonnull Map<String, String> env) throws Exception {
        for (String var : CI_BRANCHES) {
            if (!env.containsKey(var)) continue;
            String branch = env.get(var).replaceAll("^.+/", "");
            return Pair.of((Object)branch, (Object)((Object)RefType.BRANCH));
        }
        Optional<String> branch = GitUtils.tryTravisBranchNames(env);
        if (branch.isPresent()) {
            Optional<String> prSlug = GitUtils.getTravisPrSlugIfDifferentFromRepoSlug(env);
            String fullBranchName = prSlug.isPresent() ? String.format("%s/%s", prSlug.get(), branch.get()) : (String)branch.get();
            return Pair.of((Object)fullBranchName, (Object)((Object)RefType.BRANCH));
        }
        String tagName = GitUtils.getTagName(repo);
        if (tagName != null) {
            return Pair.of((Object)tagName, (Object)((Object)RefType.TAG));
        }
        String name = repo.getBranch();
        String fullName = repo.getFullBranch();
        if (name == null || fullName == null) {
            return Pair.of((Object)"Unknown", (Object)((Object)RefType.COMMIT));
        }
        RefType refType = name.equals(fullName) ? RefType.COMMIT : RefType.BRANCH;
        return Pair.of((Object)name, (Object)((Object)refType));
    }

    @Nullable
    private static String getTagName(@Nonnull Repository repository) throws Exception {
        String commitHash = GitUtils.getCommitHash(repository);
        if (commitHash == null) {
            return null;
        }
        ObjectId commitId = ObjectId.fromString((String)commitHash);
        List list = Git.wrap((Repository)repository).tagList().call();
        for (Ref tag : list) {
            Ref peeledTag = repository.peel(tag);
            ObjectId object = peeledTag.getPeeledObjectId() != null ? peeledTag.getPeeledObjectId() : peeledTag.getObjectId();
            if (!commitId.equals((AnyObjectId)object)) continue;
            return Repository.shortenRefName((String)peeledTag.getName());
        }
        return null;
    }

    @Nullable
    public static URIish getRemoteUrl(@Nonnull Repository repo, @Nonnull Map<String, String> env) throws Exception {
        URIish uri = null;
        for (String var : CI_ORIGINS) {
            if (!env.containsKey(var)) continue;
            uri = new URIish(env.get(var));
            break;
        }
        if (uri == null) {
            RemoteConfig remoteConfig;
            List uris;
            String branch = repo.getBranch();
            StoredConfig config = repo.getConfig();
            BranchConfig branchConfig = new BranchConfig((Config)config, branch);
            String remote = branchConfig.getRemote();
            if (remote == null) {
                remote = "origin";
            }
            if (!(uris = (remoteConfig = new RemoteConfig((Config)config, remote)).getURIs()).isEmpty()) {
                uri = ((URIish)uris.get(0)).setPass(null);
            }
        }
        return uri == null ? null : GitUtils.ensureScheme(uri);
    }

    @Nullable
    public static String getRepoUrl(@Nonnull Repository repo, @Nonnull Map<String, String> env) throws Exception {
        URIish uri = GitUtils.getRemoteUrl(repo, env);
        if (uri == null) {
            return null;
        }
        String host = uri.getHost();
        if (host != null) {
            String path = uri.getPath();
            if (path.charAt(0) != '/') {
                path = '/' + path;
            }
            return host + path;
        }
        return uri.toString();
    }

    @Nullable
    public static String getRepoUrl(@Nonnull File projectDir, @Nonnull Map<String, String> env) throws Exception {
        GitUtils.verifyOrThrow(projectDir);
        Repository repo = GitUtils.getRepoFromProjDir(projectDir);
        return GitUtils.getRepoUrl(repo, env);
    }

    @Nullable
    public static String getCommitHash(@Nonnull Repository repo) throws Exception {
        ObjectId head = repo.resolve("HEAD");
        if (head == null) {
            return null;
        }
        return head.getName();
    }

    @Nullable
    public static String getCommitHash(@Nonnull File projectDir) throws Exception {
        GitUtils.verifyOrThrow(projectDir);
        Repository repo = GitUtils.getRepoFromProjDir(projectDir);
        return GitUtils.getCommitHash(repo);
    }

    @Nullable
    public static String getBranch(@Nonnull File projectDir) throws Exception {
        GitUtils.verifyOrThrow(projectDir);
        Repository repo = GitUtils.getRepoFromProjDir(projectDir);
        return repo.getBranch();
    }

    public static List<String> runGit(@Nonnull File projectDir, String ... gitArgs) throws Exception {
        GitUtils.verifyOrThrow(projectDir);
        List<List<String>> output = GitUtils.runGitWithStderr(projectDir, gitArgs);
        if (output.isEmpty()) {
            return Collections.emptyList();
        }
        return output.get(0);
    }

    public static List<List<String>> runGitWithStderr(@Nonnull File projectDir, String ... gitArgs) throws Exception {
        List errorLines;
        List outputLines;
        GitUtils.verifyOrThrow(projectDir);
        File gitDir = new File(projectDir, ".git");
        if (!gitDir.exists()) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("No .git found at \"{}\"; aborting working-copy command: {}", (Object)gitDir, (Object)StringUtils.join((Object[])gitArgs, (char)' '));
            }
            return Collections.emptyList();
        }
        ArrayList<String> cmd = new ArrayList<String>(3);
        cmd.add("git");
        cmd.add("--git-dir");
        cmd.add(gitDir.getAbsolutePath());
        cmd.addAll(Arrays.asList(gitArgs));
        Process proc = new ProcessBuilder(new String[0]).command(cmd).directory(gitDir.getParentFile()).start();
        int rc = proc.waitFor();
        try (InputStream inStream = proc.getInputStream();
             InputStream errorStream = proc.getErrorStream();){
            outputLines = IOUtils.readLines((InputStream)inStream, (String)"UTF-8");
            errorLines = IOUtils.readLines((InputStream)errorStream, (String)"UTF-8");
        }
        if (0 != rc) {
            LOGGER.warn("<<{}>> did not pan out; rc={}\n[stdout]\n{}\n[stderr]\n{}\n", new Object[]{StringUtils.join(cmd, (char)' '), rc, StringUtils.join((Iterable)outputLines, (char)'\n'), StringUtils.join((Iterable)errorLines, (char)'\n')});
            return Collections.emptyList();
        }
        if (outputLines.isEmpty() && errorLines.isEmpty()) {
            LOGGER.warn("Odd, rc=0 but no output from git command");
            return Collections.emptyList();
        }
        return Arrays.asList(outputLines, errorLines);
    }

    private static void verifyOrThrow(File projectDir) {
        if (!projectDir.exists()) {
            throw new IllegalArgumentException(projectDir + " does not exist");
        }
        if (!projectDir.canRead()) {
            throw new IllegalArgumentException(projectDir + " cannot be read");
        }
        if (!projectDir.isDirectory()) {
            throw new IllegalArgumentException(projectDir + " is not a directory");
        }
    }

    protected static GitUser toGitUser(PersonIdent person, boolean collectEmail, boolean collectName) {
        String email = collectEmail ? person.getEmailAddress() : "";
        String name = collectName ? person.getName() : "";
        return new GitUser(name, email);
    }

    @Nonnull
    static URIish ensureScheme(@Nonnull URIish uri) {
        if (Strings.isNullOrEmpty((String)uri.getScheme())) {
            if (uri.toString().matches("^[^/]+:.+")) {
                return uri.setScheme("ssh");
            }
            return uri.setScheme("file");
        }
        return uri;
    }

    public static boolean isRepoClean(File projectDir) {
        String[] lines;
        String procError;
        String procOutput;
        int rc;
        ArrayList<String> command = new ArrayList<String>(3);
        command.add("git");
        command.add("status");
        command.add("--porcelain");
        try {
            Process gitStatusProc = new ProcessBuilder(new String[0]).command(command).directory(projectDir).start();
            gitStatusProc.getOutputStream().close();
            String[] stringArray = null;
            try (InputStream stdOutStream = gitStatusProc.getInputStream();
                 InputStream stdErrStream = gitStatusProc.getErrorStream();){
                Future<String> futureStdOut = SrcclrIo.readAsync(stdOutStream);
                Future<String> futureStdErr = SrcclrIo.readAsync(stdErrStream);
                rc = gitStatusProc.waitFor();
                procOutput = futureStdOut.get();
                procError = futureStdErr.get();
            }
            catch (Throwable object) {
                stringArray = object;
                throw object;
            }
        }
        catch (IOException | InterruptedException | ExecutionException e) {
            throw new RuntimeException(String.format("Exception running %s: %s", StringUtils.join(command, (String)" "), e.getMessage()));
        }
        LOGGER.debug(String.format("%s stdout: %s", StringUtils.join(command, (String)" "), procOutput));
        LOGGER.debug(String.format("%s stderr: %s", StringUtils.join(command, (String)" "), procError));
        if (rc != 0) {
            LOGGER.debug(String.format("rc of %s is non-zero. value: %d", StringUtils.join(command, (String)" "), rc));
            throw new RuntimeException(String.format("Error running %s in %s. Return code: %d. Error message: %s", StringUtils.join(command, (String)" "), projectDir, rc, procError));
        }
        if (StringUtils.isEmpty((CharSequence)procOutput.trim())) {
            return true;
        }
        String untrackedFileGitChars = "?? ";
        for (String line : lines = procOutput.split("[\\r\\n]+")) {
            if (line.trim().startsWith("?? ")) continue;
            return false;
        }
        return true;
    }

    public static boolean isValidRemote(URI remote) {
        return remote != null && !"file".equals(remote.getScheme());
    }

    private GitUtils() {
    }

    static Optional<String> tryTravisBranchNames(@Nonnull Map<String, String> env) {
        for (String var : TRAVIS_BRANCHES_ORDER_PREFERENCE) {
            String branch;
            if (!env.containsKey(var) || !StringUtils.isNotBlank((CharSequence)(branch = env.get(var)))) continue;
            return Optional.of((Object)branch);
        }
        return Optional.absent();
    }

    static Optional<String> getTravisPrSlugIfDifferentFromRepoSlug(@Nonnull Map<String, String> env) {
        String prSlug = Strings.nullToEmpty((String)env.get(TRAVIS_PULL_REQUEST_SLUG)).trim();
        String repoSLug = Strings.nullToEmpty((String)env.get(TRAVIS_REPO_SLUG)).trim();
        return !prSlug.isEmpty() && !prSlug.equals(repoSLug) ? Optional.of((Object)prSlug) : Optional.absent();
    }

    static Optional<String> tryGetBranchName(@Nonnull File projectDir) throws Exception {
        List<List<String>> outAndErr = GitUtils.runGitWithStderr(projectDir, "describe", "--all");
        if (outAndErr.isEmpty()) {
            return Optional.absent();
        }
        List<String> output = outAndErr.get(0);
        if (output.isEmpty()) {
            return Optional.absent();
        }
        for (String line : output) {
            if (!line.startsWith("heads/")) continue;
            return Optional.of((Object)line.replaceFirst("heads/", ""));
        }
        return Optional.absent();
    }

    static boolean shouldNotProxyForUrl(URL url, Collection<String> noProxyDomainList) {
        String host = url.getHost().toLowerCase();
        for (String domain : noProxyDomainList) {
            if (domain.equals("*")) {
                return true;
            }
            if (domain.startsWith(".")) {
                if (host.endsWith(domain)) {
                    return true;
                }
                if (!domain.substring(1).equalsIgnoreCase(host)) continue;
                return true;
            }
            if (!host.equalsIgnoreCase(domain)) continue;
            return true;
        }
        return false;
    }

    private static List<String> getNoProxyDomains() {
        String[] values;
        ArrayList<String> results = new ArrayList<String>();
        String noProxyValue = Strings.nullToEmpty((String)System.getenv("NO_PROXY"));
        String noProxyValue2 = Strings.nullToEmpty((String)System.getenv("no_proxy"));
        String allValues = noProxyValue + "," + noProxyValue2;
        for (String value : values = allValues.split(",")) {
            String trimmed = value.trim();
            if (trimmed.isEmpty()) continue;
            results.add(trimmed.toLowerCase());
        }
        return results;
    }

    public static enum GenericScmEnvNames {
        REF("REF"),
        REF_TYPE("REF_TYPE"),
        REV("REV"),
        SUB_PATH("SUB_PATH"),
        URI("URI");

        private String envName;

        private GenericScmEnvNames(String name) {
            this.envName = GitUtils.GENERIC_SCM_ENV_PREFIX + name;
        }

        @Nonnull
        public String getEnvironmentName() {
            return this.envName;
        }

        @Nullable
        public String getOrNull(@Nonnull Map<String, String> env) {
            return Strings.emptyToNull((String)env.get(this.envName));
        }
    }

    public static enum CommitInfoPerms {
        SKIP_ALL,
        SKIP_EMAILS,
        SKIP_NAMES;

    }
}

