/*
 * Decompiled with CFR 0.152.
 */
package com.sourceclear.engine.component.git;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.sourceclear.engine.common.logging.LogStream;
import com.sourceclear.engine.common.logging.NoopLogStream;
import com.sourceclear.engine.common.logging.Stage;
import com.sourceclear.engine.component.git.GitCloneException;
import com.sourceclear.scm.core.SCMAuth;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GitCloner {
    private static final String GIT_COMMAND = "git";
    private static final String CLONE_COMMAND = "clone";
    private static final String CHECKOUT_COMMAND = "checkout";
    private static final String GIT_DEPTH_OPTION = "--depth";
    private static final Logger LOGGER = LoggerFactory.getLogger((String)GitCloner.class.getName());
    private static final Map<String, String> GIT_ENVS = ImmutableMap.of((Object)"GIT_TERMINAL_PROMPT", (Object)"0", (Object)"GIT_ASKPASS", (Object)"false");
    private final String repositoryUrl;
    private final SCMAuth scmAuth;
    private final Integer depth;
    private final File destinationFolder;
    private final String ref;
    private final LogStream logStream;

    public static Path getDefaultCloneRoot() {
        return Paths.get(System.getProperty("user.home"), ".srcclr", "scans");
    }

    protected static String createRepoUrl(SCMAuth scmAuth, String repositoryUrl) {
        if (!repositoryUrl.startsWith("http") || scmAuth == null) {
            return repositoryUrl;
        }
        try {
            URL url = new URL(repositoryUrl);
            String userInfo = scmAuth.getAuthToken() != null ? scmAuth.getAuthToken() : String.format("%s:%s", scmAuth.getUsername(), scmAuth.getPassword());
            return new URI(url.getProtocol(), userInfo, url.getHost(), url.getPort(), url.getPath(), null, null).toString();
        }
        catch (MalformedURLException | URISyntaxException ex) {
            throw (IllegalArgumentException)new IllegalArgumentException("Couldn't create URI/URL for " + repositoryUrl).initCause(ex);
        }
    }

    private GitCloner(String repositoryUrl, SCMAuth scmAuth, Integer depth, File destinationFolder, String ref, LogStream logStream) {
        this.repositoryUrl = repositoryUrl;
        this.scmAuth = scmAuth;
        this.depth = depth;
        this.destinationFolder = destinationFolder;
        this.ref = ref;
        this.logStream = logStream;
    }

    public void exec() throws GitCloneException {
        try {
            String cloneOutput;
            List<String> cloneCommand = this.createCloneCommand();
            LOGGER.debug("Executing clone command with " + Joiner.on((String)" ").join(cloneCommand));
            String message = String.format("Cloning %s into a temporary folder for scanning.", this.repositoryUrl);
            this.logStream.log("com.srcclr.evidence.scm.start", Stage.SCM_STAGING, message);
            ProcessBuilder processBuilder = new ProcessBuilder(cloneCommand).redirectErrorStream(true);
            processBuilder.environment().putAll(GIT_ENVS);
            Process cloneProcess = processBuilder.start();
            cloneProcess.getOutputStream().close();
            try (InputStream stream = cloneProcess.getInputStream();){
                cloneOutput = this.readProcessInputStream(stream);
            }
            int term = cloneProcess.waitFor();
            if (term != 0) {
                this.logStream.logFatal("com.srcclr.evidence.scm.issue", Stage.SCM_STAGING, "Exit code from Git clone: " + term, null);
                LOGGER.debug("Clone failed with:\n" + cloneOutput);
                throw new GitCloneException(String.format("Git clone failed for %s, please verify the repository URL and permissions", this.repositoryUrl));
            }
            this.checkoutRef();
            this.logStream.log("com.srcclr.evidence.scm.end", Stage.SCM_STAGING, "Clone complete.");
        }
        catch (IOException | InterruptedException ex) {
            String errorMsgPartial = this.ref != null ? " and checkout to branch/commit " + this.ref : "";
            this.logStream.logFatal("com.srcclr.evidence.scm.issue", Stage.SCM_STAGING, "Unable to clone repo" + errorMsgPartial, ex.getMessage());
            throw new GitCloneException("Error during clone and checkout process: " + ex.getMessage(), ex);
        }
    }

    private void checkoutRef() throws IOException, InterruptedException, GitCloneException {
        if (this.ref == null) {
            LOGGER.debug("No reference specified. No need to git checkout.");
        } else {
            String checkoutOutput;
            LOGGER.debug("User specified ref: " + this.ref);
            List<String> checkoutCommand = this.createCheckoutCommand();
            LOGGER.debug("Executing checkout command with " + Joiner.on((String)" ").join(checkoutCommand));
            String message = "Performing a git checkout on ref " + this.ref;
            this.logStream.log("com.srcclr.evidence.scm.start", Stage.SCM_STAGING, message);
            ProcessBuilder checkoutProcessBuilder = new ProcessBuilder(checkoutCommand).redirectErrorStream(true);
            File workingDir = this.destinationFolder != null ? this.destinationFolder : new File(".");
            LOGGER.debug("working directory for checkout command process: " + workingDir.getAbsolutePath());
            checkoutProcessBuilder.directory(workingDir);
            checkoutProcessBuilder.environment().putAll(GIT_ENVS);
            Process checkoutProcess = checkoutProcessBuilder.start();
            checkoutProcess.getOutputStream().close();
            try (InputStream stream = checkoutProcess.getInputStream();){
                checkoutOutput = this.readProcessInputStream(stream);
            }
            int term = checkoutProcess.waitFor();
            if (term != 0) {
                this.logStream.logFatal("com.srcclr.evidence.scm.issue", Stage.SCM_STAGING, "Exit code from Git checkout: " + term, null);
                LOGGER.debug("Couldn't check out ref {}:\n {}", (Object)this.ref, (Object)checkoutOutput);
                throw new GitCloneException(String.format("An error occurred checking out ref '%s', please verify it exists", this.ref));
            }
        }
    }

    protected String getGitCloneCommand() {
        String command = "";
        for (String cmd : this.createCloneCommand()) {
            command = command + cmd + " ";
        }
        return command.trim();
    }

    private String readProcessInputStream(InputStream is) throws IOException {
        String line;
        StringBuilder sb = new StringBuilder();
        BufferedReader reader = new BufferedReader(new InputStreamReader(is));
        while ((line = reader.readLine()) != null) {
            sb.append(line).append("\n");
            LOGGER.info(line);
        }
        return sb.toString();
    }

    private List<String> createCloneCommand() {
        ArrayList commands = Lists.newArrayList((Object[])new String[]{GIT_COMMAND, CLONE_COMMAND});
        String url = GitCloner.createRepoUrl(this.scmAuth, this.repositoryUrl);
        if (this.ref != null) {
            LOGGER.info("Ref {} was specified, ignoring clone depth. Full clone will occur.", (Object)this.ref);
        } else {
            LOGGER.info("No ref was specified. Cloning latest commit of default branch.");
            if (this.depth > 0) {
                LOGGER.info("Cloning {} with a depth of {}.", (Object)this.repositoryUrl, (Object)this.depth);
                commands.addAll(Lists.newArrayList((Object[])new String[]{GIT_DEPTH_OPTION, this.depth.toString()}));
            } else {
                LOGGER.info("Performing a full clone of {}.", (Object)this.repositoryUrl);
            }
        }
        commands.add(url);
        if (this.destinationFolder == null) {
            commands.add(".");
        } else {
            commands.add(this.destinationFolder.getAbsolutePath());
        }
        return commands;
    }

    public List<String> createCheckoutCommand() {
        if (this.ref != null) {
            ArrayList commands = Lists.newArrayList((Object[])new String[]{GIT_COMMAND, CHECKOUT_COMMAND});
            commands.add(this.ref);
            return commands;
        }
        return new ArrayList<String>();
    }

    public static class GitCloneBuilder {
        private static final Integer DEFAULT_CLONE_DEPTH = 1;
        private static final String DEFAULT_BRANCH = null;
        private final String repositoryUrl;
        private SCMAuth scmAuth;
        private String ref;
        private Integer depth;
        private File destinationFolder;
        private LogStream logStream = new NoopLogStream();

        public GitCloneBuilder(String repositoryUrl) {
            this.repositoryUrl = repositoryUrl;
            this.depth = DEFAULT_CLONE_DEPTH;
            this.ref = DEFAULT_BRANCH;
        }

        public GitCloneBuilder withScmAuth(SCMAuth scmAuth) {
            this.scmAuth = scmAuth;
            return this;
        }

        public GitCloneBuilder withDestinationFolder(File destinationFolder) {
            this.destinationFolder = destinationFolder;
            return this;
        }

        public GitCloneBuilder withRef(String ref) {
            this.ref = ref;
            return this;
        }

        public GitCloneBuilder withLogStream(LogStream logStream) {
            this.logStream = logStream;
            return this;
        }

        public GitCloneBuilder withDepth(int depth) {
            this.depth = depth;
            return this;
        }

        public GitCloner build() {
            return new GitCloner(this.repositoryUrl, this.scmAuth, this.depth, this.destinationFolder, this.ref, this.logStream);
        }
    }
}

