/*
 * Decompiled with CFR 0.152.
 */
package com.sourceclear.engine.component;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.io.ByteStreams;
import com.google.common.io.PatternFilenameFilter;
import com.sourceclear.api.data.evidence.CollectionErrorType;
import com.sourceclear.bytecode.CompFingerprint;
import com.sourceclear.bytecode.HashedCompFingerprint;
import com.sourceclear.engine.common.FileTypeVisitor;
import com.sourceclear.engine.common.logging.LogStream;
import com.sourceclear.engine.component.CollectionException;
import com.srcclr.sdk.LibraryGraph;
import com.srcclr.sdk.LibraryGraphContainer;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileSystemLoopException;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.io.output.NullOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class JarUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(JarUtils.class);
    public static final ImmutableSet<String> IGNORED_FOLDERS = ImmutableSet.of((Object)"target", (Object)"build", (Object)"bin");
    public static final Set<String> JAR_EXTENSIONS = Sets.newHashSet((Object[])new String[]{".jar"});
    private static final Path TEST_RESOURCES = Paths.get("src", "test");

    public static boolean containsJarFile(File projectPath, Set<String> ignoredFolders) throws CollectionException {
        FileTypeVisitor fileTypeVisitor = new FileTypeVisitor(JAR_EXTENSIONS, ignoredFolders, LOGGER);
        try {
            Files.walkFileTree(projectPath.toPath(), fileTypeVisitor);
            return !fileTypeVisitor.getFiles().isEmpty();
        }
        catch (IOException e) {
            throw new CollectionException(CollectionErrorType.IO, "Error searching project folder for jar files: " + e.getMessage()).initCause(e);
        }
    }

    public static boolean containsJarFile(File projectPath) throws CollectionException {
        return JarUtils.containsJarFile(projectPath, IGNORED_FOLDERS);
    }

    public static LibraryGraphContainer collect(File projectRoot, Set<String> ignoredFolders, LogStream logStream) throws CollectionException {
        Path projectPath = projectRoot.toPath();
        JarVisitor jarVisitor = new JarVisitor(projectPath, ignoredFolders);
        LibraryGraphContainer.Builder graphsBuilder = new LibraryGraphContainer.Builder();
        try {
            Files.walkFileTree(projectPath, EnumSet.of(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, jarVisitor);
            graphsBuilder.withGraphs(jarVisitor.getGraphs());
        }
        catch (IOException ex) {
            LOGGER.error("Couldn't walk the file tree for {}, jar graphs was not collected", (Object)projectRoot, (Object)ex);
        }
        LibraryGraphContainer graphs = graphsBuilder.build();
        if (graphs.getGraphs().size() > 0) {
            LOGGER.debug("Discovered and analyzed {} jar files", (Object)graphs.getGraphs().size());
        }
        return graphs;
    }

    public static LibraryGraphContainer collect(File projectRoot, LogStream logStream) throws CollectionException {
        return JarUtils.collect(projectRoot, IGNORED_FOLDERS, logStream);
    }

    public static LibraryGraphContainer collectSingleFolder(File projectRoot, LogStream logstream) {
        File[] files;
        LibraryGraphContainer.Builder builder = new LibraryGraphContainer.Builder();
        for (File f : files = projectRoot.listFiles((FilenameFilter)new PatternFilenameFilter(".+\\.jar"))) {
            builder.withGraphs(JarAnalyzer.analyze(f, f.getAbsolutePath().toString()));
        }
        return builder.build();
    }

    private JarUtils() {
    }

    public static class JarVisitor
    extends SimpleFileVisitor<Path> {
        private final Path projectRoot;
        private final Set<LibraryGraph> graphs = new HashSet<LibraryGraph>();
        private final Set<String> ignoredFolders;

        public JarVisitor(Path projectRoot) {
            this(projectRoot, (Set<String>)IGNORED_FOLDERS);
        }

        public JarVisitor(Path projectRoot, Set<String> ignoredFolders) {
            this.projectRoot = projectRoot;
            this.ignoredFolders = ignoredFolders;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException ex) {
            if (ex instanceof FileSystemLoopException) {
                LOGGER.debug("File system loop detected at {}, ignoring branch.", (Object)file, (Object)ex);
            } else {
                LOGGER.debug("Failed to search {}, ignoring branch.", (Object)file, (Object)ex);
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dirPath, BasicFileAttributes attrs) {
            String folderName = dirPath.getFileName().toString();
            return this.ignoredFolders.contains(folderName) || dirPath.toFile().isHidden() || dirPath.endsWith(TEST_RESOURCES) ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path filePath, BasicFileAttributes attrs) throws IOException {
            File file = filePath.toFile();
            if (file.getName().endsWith(".jar")) {
                this.graphs.addAll(this.collectGraphs(file));
            }
            return FileVisitResult.CONTINUE;
        }

        private Set<LibraryGraph> collectGraphs(File file) {
            String relativeFileName = this.projectRoot.relativize(file.toPath()).toString();
            return JarAnalyzer.analyze(file, relativeFileName);
        }

        public Set<LibraryGraph> getGraphs() {
            return this.graphs;
        }
    }

    public static class JarAnalyzer {
        public static Set<LibraryGraph> analyze(File file, String relativeFileName) {
            HashSet<LibraryGraph> graphs = new HashSet<LibraryGraph>();
            try (FileInputStream jarInputStream = new FileInputStream(file);
                 DigestInputStream sha1Stream = new DigestInputStream(jarInputStream, MessageDigest.getInstance("SHA-1"));
                 DigestInputStream sha2Stream = new DigestInputStream(sha1Stream, MessageDigest.getInstance("SHA-256"));){
                LibraryGraph.Builder builder = new LibraryGraph.Builder();
                try {
                    CompFingerprint fingerprint = new CompFingerprint();
                    fingerprint.read(sha2Stream);
                    HashedCompFingerprint hash = new HashedCompFingerprint(fingerprint);
                    builder.withBytecodeHash(hash.hash());
                }
                catch (Exception ex) {
                    LOGGER.info("Could not compute bytecode hash for {}: {}, ignoring.", (Object)relativeFileName, (Object)ex.getMessage());
                }
                ByteStreams.copy((InputStream)sha2Stream, (OutputStream)new NullOutputStream());
                LibraryGraph graph = builder.withFilename(relativeFileName).withSha1(Hex.encodeHexString((byte[])sha1Stream.getMessageDigest().digest())).withSha2(Hex.encodeHexString((byte[])sha2Stream.getMessageDigest().digest())).build();
                graphs.add(graph);
                LOGGER.debug("Analyzed {} (sha2={}, bytecode={})", new Object[]{relativeFileName, graph.getSha2(), graph.getBytecodeHash()});
            }
            catch (Exception ex) {
                LOGGER.info("Could not scan {}", (Object)file, (Object)ex);
            }
            return graphs;
        }
    }
}

