/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.styx.util;

import com.spotify.styx.model.Workflow;
import com.spotify.styx.model.WorkflowConfiguration;
import com.spotify.styx.model.WorkflowId;
import com.spotify.styx.util.DockerImageValidator;
import com.spotify.styx.util.TimeUtil;
import com.spotify.styx.util.WorkflowValidator;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.springframework.expression.ParseException;
import org.springframework.expression.spel.standard.SpelExpressionParser;

public class BasicWorkflowValidator
implements WorkflowValidator {
    private static final int MAX_ID_LENGTH = 256;
    private static final int MAX_DOCKER_ARGS_TOTAL = 1000000;
    private static final int MAX_RESOURCES = 5;
    private static final int MAX_RESOURCE_LENGTH = 256;
    private static final int MAX_COMMIT_SHA_LENGTH = 256;
    private static final int MAX_SECRET_NAME_LENGTH = 253;
    private static final int MAX_SECRET_MOUNT_PATH_LENGTH = 1024;
    private static final int MAX_SERVICE_ACCOUNT_LENGTH = 256;
    private static final int MAX_RETRY_CONDITION_LENGTH = 256;
    private static final int MAX_ENV_VARS = 128;
    private static final int MAX_ENV_SIZE = 16384;
    private static final Pattern VALID_EMAIL_ADDRESS_REGEX = Pattern.compile("^[_A-Za-z0-9-+]+(\\.[_A-Za-z0-9-]+)*@[A-Za-z0-9-]+(\\.[A-Za-z0-9]+)*(\\.[A-Za-z]{2,})$", 2);
    private static final Duration MIN_RUNNING_TIMEOUT = Duration.ofMinutes(1L);
    private final DockerImageValidator dockerImageValidator;

    public BasicWorkflowValidator(DockerImageValidator dockerImageValidator) {
        this.dockerImageValidator = Objects.requireNonNull(dockerImageValidator);
    }

    @Override
    public List<String> validateWorkflow(Workflow workflow) {
        WorkflowId workflowId = workflow.id();
        WorkflowConfiguration cfg = workflow.configuration();
        ArrayList<String> e = new ArrayList<String>();
        String componentId = workflowId.componentId();
        if (componentId.isEmpty()) {
            e.add("component id cannot be empty");
        } else if (componentId.contains("#")) {
            e.add("component id cannot contain #");
        }
        if (workflowId.id().isEmpty()) {
            e.add("workflow id cannot be empty");
        }
        if (!workflowId.id().equals(cfg.id())) {
            e.add("workflow id mismatch");
        }
        WorkflowValidator.upperLimit(e, cfg.id().length(), 256, "id too long");
        WorkflowValidator.upperLimit(e, cfg.commitSha().map(String::length).orElse(0), 256, "commitSha too long");
        WorkflowValidator.upperLimit(e, cfg.secret().map(s2 -> s2.name().length()).orElse(0), 253, "secret name too long");
        WorkflowValidator.upperLimit(e, cfg.secret().map(s2 -> s2.mountPath().length()).orElse(0), 1024, "secret mount path too long");
        WorkflowValidator.upperLimit(e, cfg.serviceAccount().map(String::length).orElse(0), 256, "service account too long");
        WorkflowValidator.upperLimit(e, cfg.resources().size(), 5, "too many resources");
        WorkflowValidator.upperLimit(e, cfg.env().size(), 128, "too many env vars");
        WorkflowValidator.upperLimit(e, cfg.env().entrySet().stream().mapToInt(entry -> ((String)entry.getKey()).length() + ((String)entry.getValue()).length()).sum(), 16384, "env too big");
        WorkflowValidator.upperLimit(e, cfg.retryCondition().map(String::length).orElse(0), 256, "retry condition too long");
        cfg.dockerImage().ifPresent(image -> this.dockerImageValidator.validateImageReference((String)image).stream().map(s2 -> "invalid image: " + s2).forEach(e::add));
        cfg.resources().stream().map(String::length).forEach(v -> WorkflowValidator.upperLimit(e, v, 256, "resource name too long"));
        cfg.dockerArgs().ifPresent(args -> {
            int dockerArgs = args.size() + args.stream().mapToInt(String::length).sum();
            WorkflowValidator.upperLimit(e, dockerArgs, 1000000, "docker args is too large");
        });
        cfg.offset().ifPresent(offset -> {
            try {
                TimeUtil.addOffset(ZonedDateTime.now(), offset);
            }
            catch (DateTimeParseException ex) {
                e.add(String.format("invalid offset: %s", ex.getMessage()));
            }
        });
        try {
            TimeUtil.cron(cfg.schedule());
        }
        catch (IllegalArgumentException ex) {
            e.add("invalid schedule");
        }
        cfg.runningTimeout().ifPresent(timeout -> WorkflowValidator.lowerLimit(e, timeout, MIN_RUNNING_TIMEOUT, "running timeout is too small"));
        cfg.serviceAccount().ifPresent(serviceAccount -> {
            if (!BasicWorkflowValidator.validateServiceAccount(serviceAccount)) {
                e.add("service account is not a valid email address: " + serviceAccount);
            }
        });
        cfg.retryCondition().ifPresent(retryCondition -> {
            try {
                new SpelExpressionParser().parseRaw((String)retryCondition);
            }
            catch (ParseException ex) {
                e.add(String.format("invalid retry condition: %s", ex.getMessage()));
            }
        });
        return e;
    }

    private static boolean validateServiceAccount(String serviceAccount) {
        Matcher matcher = VALID_EMAIL_ADDRESS_REGEX.matcher(serviceAccount);
        return matcher.matches();
    }
}

