/*
 * JPAstreamer - Express JPA queries with Java Streams
 * Copyright (c) 2020-2022, Speedment, Inc. All Rights Reserved.
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * See: https://github.com/speedment/jpa-streamer/blob/master/LICENSE
 */
package com.speedment.jpastreamer.field.internal;

import com.speedment.jpastreamer.field.FloatField;
import com.speedment.jpastreamer.field.comparator.FloatFieldComparator;
import com.speedment.jpastreamer.field.comparator.NullOrder;
import com.speedment.jpastreamer.field.internal.comparator.FloatFieldComparatorImpl;
import com.speedment.jpastreamer.field.internal.method.GetFloatImpl;
import com.speedment.jpastreamer.field.internal.predicate.floats.*;
import com.speedment.jpastreamer.field.internal.util.CollectionUtil;
import com.speedment.jpastreamer.field.method.FloatGetter;
import com.speedment.jpastreamer.field.method.GetFloat;
import com.speedment.jpastreamer.field.predicate.FieldPredicate;
import com.speedment.jpastreamer.field.predicate.Inclusion;
import com.speedment.jpastreamer.field.predicate.SpeedmentPredicate;

import java.util.Collection;

import static java.util.Objects.requireNonNull;

/**
 * Default implementation of the {@link FloatField}-interface.
 * 
 * Generated by com.speedment.sources.pattern.FieldImplPattern
 * 
 * @param <ENTITY> entity type
 *
 * @author Emil Forslund
 * @since  3.0.0
 */
public final class FloatFieldImpl<ENTITY> implements FloatField<ENTITY> {
    
    private final Class<ENTITY> table;
    private final String columnName;
    private final GetFloat<ENTITY> getter;
    private final boolean unique;

    public FloatFieldImpl(
            Class<ENTITY> table,
            String columnName,
            FloatGetter<ENTITY> getter,
            boolean unique) {
        this.table = requireNonNull(table);
        this.columnName = requireNonNull(columnName);
        this.getter     = new GetFloatImpl<>(this, getter);
        this.unique     = unique;
    }
    
    @Override
    public Class<ENTITY> table() {
        return table;
    }

    @Override
    public GetFloat<ENTITY> getter() {
        return getter;
    }

    @Override
    public boolean isUnique() {
        return unique;
    }

    @Override
    public FloatFieldComparator<ENTITY> comparator() {
        return new FloatFieldComparatorImpl<>(this);
    }
    
    @Override
    public FloatFieldComparator<ENTITY> reversed() {
        return comparator().reversed();
    }
    
    @Override
    public FloatFieldComparator<ENTITY> comparatorNullFieldsFirst() {
        return comparator();
    }
    
    @Override
    public NullOrder getNullOrder() {
        return NullOrder.LAST;
    }
    
    @Override
    public boolean isReversed() {
        return false;
    }
    
    @Override
    public FieldPredicate<ENTITY> equal(Float value) {
        return new FloatEqualPredicate<>(this, value);
    }
    
    @Override
    public FieldPredicate<ENTITY> greaterThan(Float value) {
        return new FloatGreaterThanPredicate<>(this, value);
    }
    
    @Override
    public FieldPredicate<ENTITY> greaterOrEqual(Float value) {
        return new FloatGreaterOrEqualPredicate<>(this, value);
    }
    
    @Override
    public FieldPredicate<ENTITY> between(
            Float start,
            Float end,
            Inclusion inclusion) {
        return new FloatBetweenPredicate<>(this, start, end, inclusion);
    }
    
    @Override
    public FieldPredicate<ENTITY> in(Collection<Float> values) {
        return new FloatInPredicate<>(this, CollectionUtil.collectionToSet(values));
    }
    
    @Override
    public SpeedmentPredicate<ENTITY> notEqual(Float value) {
        return new FloatNotEqualPredicate<>(this, value);
    }
    
    @Override
    public SpeedmentPredicate<ENTITY> lessOrEqual(Float value) {
        return new FloatLessOrEqualPredicate<>(this, value);
    }
    
    @Override
    public SpeedmentPredicate<ENTITY> lessThan(Float value) {
        return new FloatLessThanPredicate<>(this, value);
    }
    
    @Override
    public SpeedmentPredicate<ENTITY> notBetween(
            Float start,
            Float end,
            Inclusion inclusion) {
        return new FloatNotBetweenPredicate<>(this, start, end, inclusion);
    }
    
    @Override
    public SpeedmentPredicate<ENTITY> notIn(Collection<Float> values) {
        return new FloatNotInPredicate<>(this, CollectionUtil.collectionToSet(values));
    }

    @Override
    public String columnName() {
        return columnName;
    }
}
