/*
 * Copyright (C) 2016 Oleg Kan, @Simplaapliko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.simplaapliko.updater;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.util.Log;

import java.util.List;

public class Updater {

    // Inner classes

    public interface OnVersionChangeListener {
        void onVersionChange();
    }

    private static final String TAG = "Updater";

    private Context mContext;
    private int mNewVersionCode;
    private int mOldVersionCode;


    // Constructors

    public Updater(Context context) {
        mContext = context.getApplicationContext();

        try {
            PackageInfo packageInfo = mContext.getPackageManager()
                    .getPackageInfo(mContext.getPackageName(), PackageManager.GET_META_DATA);

            mNewVersionCode = packageInfo.versionCode;
        } catch (PackageManager.NameNotFoundException e) {
            Log.e(TAG, "unable to get version code.", e);
            mNewVersionCode = PreferencesHelper.VERSION_NOT_FOUND;
        }

        mOldVersionCode = new PreferencesHelper(mContext)
                .getVersion();
    }


    // Public API

    public int getNewVersion() {
        return mNewVersionCode;
    }

    public int getOldVersion() {
        return mOldVersionCode;
    }

    public boolean isFirstLaunch() {
        return mOldVersionCode == PreferencesHelper.VERSION_NOT_FOUND;
    }

    /**
     *
     * @return true if app version updated.
     * <b><i>Note: Returns false on the first launch</i></b>
     */
    public boolean isVersionChanged() {
        return mOldVersionCode != PreferencesHelper.VERSION_NOT_FOUND &&
                mNewVersionCode > mOldVersionCode;
    }

    public void onVersionChanged(List<OnVersionChangeListener> listeners) {

        for (OnVersionChangeListener listener : listeners) {
            update(listener);
        }

        new PreferencesHelper(mContext)
                .setVersion(mNewVersionCode);
    }


    // Private API

    private void update(OnVersionChangeListener listener) {
        listener.onVersionChange();
    }

}
