/*
 * Copyright (C) 2014-2015 Oleg Kan, @Simplaapliko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.simplaapliko.apprater;

import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.res.Resources;
import android.net.Uri;
import android.support.v4.app.DialogFragment;
import android.support.v7.app.AppCompatActivity;
import android.widget.Toast;

public final class AppRater {

    /**
     * Default values for Days until 'rate app' dialog will pop up.
     */
    public static final int DAYS_UNTIL_PROMPT = 7;

    /**
     * Default values for Launches until 'rate app' dialog will pop up.
     */
    public static final int LAUNCHES_UNTIL_PROMPT = 10;

    /**
     * Shared preferences variables.
     */
    private static final String PREFS_APP_RATER = "PREFS_APP_RATER";
    private static final String PREF_FIRST_DATE_LAUNCH = "FIRST_DATE_LAUNCH";
    private static final String PREF_LAUNCH_COUNT = "LAUNCH_COUNT";
    private static final String PREF_DO_NOT_SHOW_AGAIN = "DO_NOT_SHOW_AGAIN";

    /**
     * Check if 'rate app' dialog needs to be displayed,
     * based on the first date launch, launch count and do not show again
     * shared preferences.
     */
    public static void appLaunched(Context context) {
        int dayUntilPrompt;
        int launchesUntilPrompt;

        try {
            dayUntilPrompt = context.getResources().getInteger(R.integer.app_rater_days_until_prompt);
        } catch (Resources.NotFoundException ex) {
            // set default values if resource is not found
            dayUntilPrompt = DAYS_UNTIL_PROMPT;
        }

        try {
            launchesUntilPrompt = context.getResources().getInteger(R.integer.app_rater_launches_until_prompt);
        } catch (Resources.NotFoundException ex) {
            // set default values if resource is not found
            launchesUntilPrompt = LAUNCHES_UNTIL_PROMPT;
        }

        SharedPreferences prefs = context.getSharedPreferences(PREFS_APP_RATER, Context.MODE_PRIVATE);
        if (prefs.getBoolean(PREF_DO_NOT_SHOW_AGAIN, false)) {
            return;
        }

        SharedPreferences.Editor editor = prefs.edit();

        // Increment the launch counter
        int launchCount = prefs.getInt(PREF_LAUNCH_COUNT, 0) + 1;
        editor.putInt(PREF_LAUNCH_COUNT, launchCount);

        // Get date of the first launch
        long firstDateLaunch = prefs.getLong(PREF_FIRST_DATE_LAUNCH, 0);
        if (firstDateLaunch == 0) {
            firstDateLaunch = System.currentTimeMillis();
            editor.putLong(PREF_FIRST_DATE_LAUNCH, firstDateLaunch);
        }

        editor.apply();

        // Wait at least n days before opening
        if (launchCount >= launchesUntilPrompt &&
                (System.currentTimeMillis() >= firstDateLaunch + dayUntilPrompt * 24 * 60 * 60 * 1000)) {

            DialogFragment dialog = new RateAppDialog.Builder()
                    .build();
            dialog.show(((AppCompatActivity) context).getSupportFragmentManager(), null);
        }
    }

    /**
     * Open Play Store for the related application.
     */
    public static void rateApp(Context context) {

        SharedPreferences prefs = context.getSharedPreferences(PREFS_APP_RATER, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = prefs.edit();

        editor.putBoolean(PREF_DO_NOT_SHOW_AGAIN, true);
        editor.apply();

        try {
            context.startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("market://details?id=" + context.getPackageName())));
        } catch (ActivityNotFoundException e) {
            context.startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("http://play.google.com/store/apps/details?id=" + context.getPackageName())));
        } catch (Exception e) {
            Toast.makeText(context, context.getString(R.string.unable_to_find_google_play), Toast.LENGTH_LONG).show();
        }
    }

    /**
     * Reset launch counter.
     */
    public static void remindLater(Context context) {

        SharedPreferences prefs = context.getSharedPreferences(AppRater.PREFS_APP_RATER, 0);
        SharedPreferences.Editor editor = prefs.edit();

        editor.putInt(AppRater.PREF_LAUNCH_COUNT, 0);
        long dateFirstLaunch = System.currentTimeMillis();
        editor.putLong(AppRater.PREF_FIRST_DATE_LAUNCH, dateFirstLaunch);
        editor.apply();
    }

    /**
     * Cancel reminders.
     */
    public static void cancelReminders(Context context) {

        SharedPreferences prefs = context.getSharedPreferences(AppRater.PREFS_APP_RATER, 0);
        SharedPreferences.Editor editor = prefs.edit();

        editor.putBoolean(AppRater.PREF_DO_NOT_SHOW_AGAIN, true);
        editor.apply();
    }

    /**
     * Check if this is a first launch
     */
    public static boolean isFirstLaunch(Context context) {
        SharedPreferences prefs = context.getSharedPreferences(PREFS_APP_RATER, Context.MODE_PRIVATE);

        return prefs.getLong(PREF_FIRST_DATE_LAUNCH, 0) == 0;
    }

}