/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.utils;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.provider.Settings;
import android.text.Spannable;
import android.text.SpannableString;
import android.text.style.ForegroundColorSpan;
import android.util.TypedValue;
import android.widget.TextView;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Locale;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.4
 * @since 28/Jan/2016
 */
public class Utils {

    public static final byte BIT_7 = (byte) 0x80;
    public static final byte BIT_6 = (byte) 0x40;
    public static final byte BIT_5 = (byte) 0x20;
    public static final byte BIT_4 = (byte) 0x10;
    public static final byte BIT_3 = (byte) 0x08;
    public static final byte BIT_2 = (byte) 0x04;
    public static final byte BIT_1 = (byte) 0x02;
    public static final byte BIT_0 = (byte) 0x01;
    private static final String TAG = Utils.class.getName();
    private static final String TAG_DIALOG_FRAGMENT = "dialog_fragment";
    private static Utils instance;
    private final Context mContext;

    private Utils(Context context) {
        mContext = context;
    }

    private Utils() {
        this(AppContext.getInstance().getContext());
    }

    public static Utils getInstance() {
        if (instance == null)
            instance = new Utils();
        return instance;
    }

    public static Utils getInstance(Context context) {
        if (instance == null)
            instance = new Utils(context);
        return instance;
    }

    /**
     * Get current date and time
     *
     * @return date and time string format.
     */
    public static String getDateAndTimeString() {
        // date and time;
        Calendar calendar = Calendar.getInstance();
        Date date = calendar.getTime();
        SimpleDateFormat format = new SimpleDateFormat("   dd/MM/yyyy       HH:mm:ss    ", Locale.US);
        return format.format(date);
    }

    /**
     * Get current date and time
     *
     * @return date and time string format.
     */
    public static String getTimeString() {
        // date and time;
        Calendar calendar = Calendar.getInstance();
        Date date = calendar.getTime();
        SimpleDateFormat format = new SimpleDateFormat("HH:mm:ss", Locale.US);
        return format.format(date);
    }

    public static String getCapitalizeFirstLetter(String original) {
        if (original.length() == 0)
            return original;
        return original.substring(0, 1).toUpperCase() + original.substring(1).toLowerCase();
    }

    public static void scrollAmountTextView(TextView view) {
        try {
            final int scrollAmount = view.getLayout().getLineTop(view.getLineCount()) - view.getHeight();
            // if there is no need to scroll, scrollAmount will be <=0
            if (scrollAmount > 0) {
                view.scrollTo(0, scrollAmount);
            } else {
                view.scrollTo(0, 0);
            }
        } catch (NullPointerException ignored) {

        }
    }

    public static byte checkSum(byte[] input) {
        byte sum = 0x00;
        for (int i = 0; i < input.length - 1; i++) {
            sum ^= input[i];
        }
        return sum;
    }


    public static byte checkSum(byte[] data, int start, int length) {
        byte crc = 0x00;
        if (data == null) {
            return 0x00;
        }

        if (start < 0 || length < 1 || start + length > data.length + 1) {
            return 0x00;
        }
        for (int i = start; i < start + length - 1; ++i) {
            crc ^= data[i];
        }
        return crc;
    }

    public static Spannable toSpannable(String text) {
        return new SpannableString(text);
    }

    public static Spannable setSpannable(Spannable text, int color) {
        text.setSpan(new ForegroundColorSpan(color), 0, text.length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
        return text;
    }


    public static Spannable setSpannable(Spannable fulltext, String subtext, int color) {
        int index = fulltext.toString().indexOf(subtext);
        if (index < 0) {
            return fulltext;
        }
        fulltext.setSpan(new ForegroundColorSpan(color), index, index + subtext.length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);

        return fulltext;
    }

    public static Spannable setSpannable(Spannable text, int index, int length, int color) {
        if (index + length > text.length() || index < 0 || length < 0) {
            return text;
        }
        text.setSpan(new ForegroundColorSpan(color), index, index + length, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);

        return text;
    }

    public static boolean checkArraySize(byte[] data, int min_size, int max_size) {
        return data != null && !(min_size < 0 || data.length < min_size || data.length > max_size);
    }

    public static byte[][] chunkSplit(byte[] data, int size) {
        if (checkArraySize(data, 1, data.length)) {
            int dalen = data.length;
            int maxlen = (int) Math.ceil(dalen / (double) size);
            byte[][] buffer = new byte[maxlen--][];

            for (int i = 0; i < maxlen; ++i) {
                buffer[i] = new byte[size];
                System.arraycopy(data, i * size, buffer[i], 0, size);
            }
            if (dalen % size > 0) {
                size = dalen % size;
            }
            buffer[maxlen] = new byte[size];
            System.arraycopy(data, dalen - size, buffer[maxlen], 0, size);
            return buffer;
        }
        return null;
    }

    public static byte[] getPackageLimited(byte cmd, byte[] data, int size) {
        if (data == null) {
            return null;
        }
        int length = Math.min(size, data.length);
        byte[] buffer = new byte[length + 1];
        buffer[0] = cmd;
        System.arraycopy(data, 0, buffer, 1, length);
        return buffer;
    }

    public static byte[][] getPackageSplit(byte cmd, byte[][] data, int size) {
        return getPackageSplit(cmd, cmd, data, size);
    }

    public static byte[][] getPackageSplit(byte cmd, byte lastcmd, byte[][] data, int size) {
        List<byte[]> buffer = new ArrayList<>();
        for (byte[] each : data) {
            if (Utils.checkArraySize(each, 1, each.length)) {
                int dalen = each.length;
                int localSize = size;
                int maxlen = (int) Math.ceil(dalen / (double) localSize) - 1;
                byte[] temp;

                for (int i = 0; i < maxlen; ++i) {
                    temp = new byte[localSize + 1];
                    temp[0] = cmd;
                    System.arraycopy(each, i * localSize, temp, 1, localSize);
                    buffer.add(temp);
                }
                if (dalen % localSize > 0) {
                    localSize = dalen % localSize;
                }
                temp = new byte[localSize + 1];
                temp[0] = lastcmd;
                System.arraycopy(each, dalen - localSize, temp, 1, localSize);
                buffer.add(temp);
            }
        }
        byte[][] array = new byte[buffer.size()][];
        return buffer.toArray(array);
    }

    public String getDeviceId() {
        return Settings.Secure.getString(mContext.getContentResolver(), Settings.Secure.ANDROID_ID);
    }

    public String getVersionName() {
        try {
            PackageInfo pInfo = mContext.getPackageManager().getPackageInfo(mContext.getPackageName(), 0);
            return pInfo.versionName;
        } catch (Exception e) {
            return "1.0";
        }
    }

    /**
     * Convert Dp to Pixel
     *
     * @param dp unit
     * @return Pixel unit
     */
    public int dpToPx(float dp) {
        float px = TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dp, mContext.getResources().getDisplayMetrics());
        return (int) px;
    }

    public List<ApplicationInfo> getInstalledApplication() {
        PackageManager packageManager = mContext.getPackageManager();
        List<ApplicationInfo> apps = packageManager.getInstalledApplications(0);
        Collections.sort(apps, new ApplicationInfo.DisplayNameComparator(packageManager));
        return apps;
    }
}
