/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.utils;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.0
 * @since 29/Jan/2016
 */
public class ParseUtils {

    private static ParseUtils instance;

    public static ParseUtils getInstance() {
        if (instance == null)
            instance = new ParseUtils();
        return instance;
    }

    private ParseUtils() {
    }

    // String to ...

    /**
     * Parse decimal string to hex string.
     *
     * @param string decimal string.
     * @return hex string.
     */
    public static String parseDecimalStringToHexString(String string) {
        return String.format("%X ", new BigInteger(string.getBytes(/* YOUR_CHARSET? */)));
    }

    /**
     * Parse ASCII string to byte array.
     *
     * @param ascii ascii string.
     * @return byte array.
     */
    public static byte[] parseASCIIStringToByteArray(String ascii) {
        int length = ascii.length();
        byte[] b = new byte[length];
        for (int i = 0; i < length; ++i) {
            b[i] = (byte) ascii.charAt(i);
        }
        return b;
    }

    /**
     * Parse hex string to byte array.
     *
     * @param sarray Hex string.
     * @return byte array.
     */
    public static byte[] parseHexStringToByteArray(String sarray) {
        List<String> separated = new ArrayList<>();
        int index = 0;
        while (index < sarray.length()) {
            separated.add(sarray.substring(index,
                    Math.min(index + 2, sarray.length())));
            index += 2;
        }
        byte[] b = new byte[separated.size()];
        for (int i = 0; i < separated.size(); ++i) {
            try {
                b[i] = (byte) Integer.parseInt(separated.get(i), 16);
            } catch (NumberFormatException e) {
                b[i] = (byte) 0xFF;
            }
        }
        return b;
    }

    /**
     * Parse string to byte array, can chooses
     *
     * @param sarray Hex string.
     * @param radix  radix for number
     * @return byte array.
     */
    public static byte[] parseStringToByteArray(String sarray, int radix) {
        List<String> separated = new ArrayList<>();
        int index = 0;
        int width = (int) Math.ceil((Math.log10(0xFF) / Math.log10(radix)));
        while (index < sarray.length()) {
            String s = sarray.substring(index,
                    Math.min(index + width, sarray.length()));
            if (Integer.parseInt(s, radix) > 255) {
                sarray = sarray.substring(0, index) + "0"
                        + sarray.substring(index, sarray.length());
                s = sarray.substring(index,
                        Math.min(index + width, sarray.length()));
            }
            separated.add(s);
            index += width;
        }
        byte[] b = new byte[separated.size()];
        for (int i = 0; i < separated.size(); ++i) {
            try {
                b[i] = (byte) Integer.parseInt(separated.get(i), radix);
            } catch (NumberFormatException e) {
                b[i] = (byte) 0xFF;
            }
        }
        return b;
    }

    // Integer to ...

    /**
     * Parse integer to byte array. It is can limit array.
     *
     * @param intad integer.
     * @return byte array.
     */
    public static byte[] parseIntegerToByteArray(int intad) {
        String HexStr = Integer.toHexString(intad);
        return parseHexStringToByteArray(HexStr);
    }

    /**
     * Parse integer to byte array. It is can limit array.
     *
     * @param intad  integer.
     * @param length length of byte array.
     * @return byte array.
     */
    public static byte[] parseIntegerToByteArray(int intad, int length) {
        String HexStr = Integer.toHexString(intad);
        length <<= 1;
        while (HexStr.length() < length) {
            HexStr = "0" + HexStr;
        }
        if (HexStr.length() > length) {
            HexStr = HexStr.substring(HexStr.length() - length);
        }
        return parseHexStringToByteArray(HexStr);
    }

    // Byte to ...
    public static String parseByteToHexString(byte bytead) {
        return ("" + String.format("%02x", bytead & 0xff)).toUpperCase();
    }

    // Byte Array to ...

    /**
     * Parse byte array to ASCII string.
     *
     * @param barray byte array.
     * @return ASCII.
     */
    public static String parseByteArrayToASCIIString(byte[] barray) {
        StringBuilder strBuilder = new StringBuilder();
        for (byte aBarray : barray) {
            strBuilder.append((char) aBarray);
        }
        return strBuilder.toString();
    }

    /**
     * Parse byte array to hex string.
     *
     * @param barray byte array.
     * @return hex string.
     */
    public static String parseByteArrayToHexString(byte[] barray) {
        StringBuilder hexString = new StringBuilder("");
        if (barray == null) {
            return "";
        }
        for (byte b : barray) {
            hexString.append(String.format("%02x", b & 0xff));
        }
        return hexString.toString();
    }

    /**
     * Parse byte array to string by radix.
     *
     * @param barray byte array
     * @param radix  base number.
     * @return default string.
     */
    public static String parseByteArrayToString(byte[] barray, int radix) {
        StringBuilder string = new StringBuilder("");
        String temp;
        int width = (int) Math.ceil((Math.log10(0xFF) / Math.log10(radix)));
        for (byte b : barray) {
            temp = Integer.toString((int) b & 0xFF, radix);
            while (temp.length() % width != 0) {
                temp = "0" + temp;
            }
            string.append(temp);
        }

        return string.toString();
    }

    public static byte[] concatenateByteArray(byte[]... arrays) {
        // Determine the length of the result array
        int totalLength = 0;
        for (byte[] array : arrays) {
            totalLength += array.length;
        }

        // create the result array
        byte[] result = new byte[totalLength];

        // copy the source arrays into the result array
        int currentIndex = 0;
        for (byte[] array : arrays) {
            System.arraycopy(array, 0, result, currentIndex, array.length);
            currentIndex += array.length;
        }

        return result;
    }

    public static String getOrdinal(int packetNumber) {
        String ordinal;
        if (packetNumber / 10 % 10 == 1) {
            ordinal = packetNumber + "th";
        } else {
            switch (packetNumber % 10) {
                case 1:
                    ordinal = packetNumber + "st";
                    break;
                case 2:
                    ordinal = packetNumber + "nd";
                    break;
                case 3:
                    ordinal = packetNumber + "rd";
                    break;
                default:
                    ordinal = packetNumber + "th";
                    break;
            }
        }
        return ordinal;
    }

}
