/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.mandatories;

import android.content.Context;
import android.content.Intent;
import android.nfc.TagLostException;
import android.nfc.tech.NfcA;

import com.sic.module.nfc.Flag;
import com.sic.module.nfc.Nfc;
import com.sic.module.utils.SICLog;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.2
 * @since 25/Nov/2015
 */
public class NfcTypeA extends Nfc {

    private static final String TAG = NfcTypeA.class.getName();

    private static final String[] TECH_LISTS = new String[]{NfcA.class.getName()};

    private static NfcTypeA instance;
    private final Context mContext;
    protected byte response = Flag.DISAPPEAR;
    private NfcA nfcA;
    private int timeout = 618;
    private List<byte[]> transferData = new ArrayList<>();
    private List<byte[]> receiverData = new ArrayList<>();
    private boolean protection = true;

    protected NfcTypeA(Context context) {
        super(context, null, new String[][]{TECH_LISTS});
        mContext = context;
    }

    public static NfcTypeA getInstance(Context context) {
        if (instance == null)
            instance = new NfcTypeA(context);
        return instance;
    }

    @Override
    public boolean onTagFinding(Intent intent) {
        if (super.onTagFinding(intent)) {
            nfcA = NfcA.get(getTag());
            response = Flag.NFC_SAK;
            return true;
        }
        return false;
    }

    protected Context getContext() {
        return mContext;
    }

    /**
     * Gets a original NfcA object to Provides access to NFC-A (ISO 14443-3A) properties.
     *
     * @return nfcA object
     */
    public NfcA getRawNfcA() {
        return nfcA;
    }

    /**
     * Checks whether a tag exists in a NFC field.
     *
     * @return true, if a tag exists.
     */
    public boolean isTagExists() {
        try {
            if (nfcA.isConnected()) {
                nfcA.close();
            }
            nfcA.connect();
            if (protection) {
                nfcA.close();
            }
            return response != Flag.DISAPPEAR;
        } catch (Exception e) {
            response = Flag.DISAPPEAR;
            return false;
        }
    }

    /**
     * Gets the NFC timeout in milliseconds.
     *
     * @return integer, timeout value in milliseconds
     */
    public int getTimeout() {
        return timeout;
    }

    /**
     * Sets the NFC timeout in milliseconds.
     *
     * @param timeout timeout value in milliseconds
     */
    public void setTimeout(int timeout) {
        this.timeout = timeout;
        nfcA.setTimeout(timeout);
    }

    /**
     * This function is used for check protection from multi-type of using NFC.
     *
     * @return true is auto disconnect every transceive command, if false is not.
     */
    public boolean isProtection() {
        return protection;
    }

    /**
     * This function is used for set protection from multi-type of using NFC.
     *
     * @param protection true is auto disconnect every transceive command, if false is not.
     */
    public void setProtection(boolean protection) {
        this.protection = protection;
    }

    /**
     * This function is used for disconnect the NFC device.
     */
    public void close() {
        protection = true;
        try {
            if (nfcA.isConnected()) {
                nfcA.close();
            }
        } catch (Exception e) {
            e.printStackTrace();
            response = Flag.DISAPPEAR;
        }
    }

    /**
     * This function is used for add data for prepare to send to a tag.
     *
     * @param buffer data for transmission
     * @return last index
     */
    public int addDataStream(byte[]... buffer) {
        if (buffer == null || buffer.length == 0) {
            return -1;
        }
        Collections.addAll(transferData, buffer);
        return transferData.size() - 1;
    }

    /**
     * This function is used for get all data receiver from stream function
     *
     * @return data receiver
     */
    public byte[][] getDataStream() {
        int index = 0;
        int size = receiverData.size();
        byte[][] result = new byte[size][];
        for (byte[] data : receiverData) {
            result[index++] = data.clone();
        }
        return result;
    }

    /**
     * This function is used for get data receiver by index from stream function
     *
     * @param index index of data buffer
     * @return data receiver at index
     */
    public byte[] getDataStreamAt(int index) {
        if (index < 0 || index >= receiverData.size()) {
            return null;
        }
        return receiverData.get(index);
    }

    /**
     * This function is used for get latest data receiver from stream function
     *
     * @return latest data receiver
     */
    public byte[] getDataStreamAtLatestIndex() {
        int size = receiverData.size();
        if (size == 0) {
            return null;
        }
        return receiverData.get(receiverData.size() - 1);
    }

    /**
     * This function is used for get latest data receiver from stream function
     *
     * @return latest data receiver
     */
    public int getLatestIndexOfDataStream() {
        return receiverData.size() - 1;
    }

    /**
     * This function is used for clear data buffer.
     */
    public void clearDataStream() {
        transferData.clear();
        receiverData.clear();
    }

    /**
     * This function is used for send data from addDataStream function and can get the result from
     * getDataStream or getDataStreamAt function.
     */
    public synchronized void stream() {
        byte[] recv = new byte[0];
        receiverData.clear();
        response = Flag.UNKNOWN;
        try {
            if (!nfcA.isConnected()) {
                nfcA.connect();
                nfcA.setTimeout(timeout);
            }
            for (byte[] data : transferData) {
                //SICLog.i(TAG, "SEND[" + data.length + "]: " + Utils.parseByteArrayToHexString(data));
                recv = nfcA.transceive(data);
                receiverData.add(recv);
                //SICLog.i(TAG, "RECV[" + recv.length + "]: " + Utils.parseByteArrayToHexString(recv));
            }
            response = recv[0];
            if (protection) {
                nfcA.close();
            }
        } catch (TagLostException e_lost) {
            response = Flag.DISAPPEAR;
            SICLog.e(TAG, e_lost.getMessage());

        } catch (IOException | NullPointerException e_io_null) {
            response = Flag.EXCEPTION;
            if (e_io_null.getMessage() != null) {
                SICLog.e(TAG, e_io_null.getMessage());
            } else {
                e_io_null.printStackTrace();
            }

        } catch (Exception e) {
            response = Flag.NFC_NAK;
            if (e.getMessage() != null) {
                SICLog.e(TAG, e.getMessage());
            }
        }
    }

    /**
     * NFC transceive function connects and closes NFC software service
     * automatically.
     *
     * @param send_data raw data sent from device.
     * @return data received from NFC.
     */
    public synchronized byte[] autoTransceive(byte[] send_data) {
        byte[] recv = null;
        response = Flag.UNKNOWN;
        try {
            if (!nfcA.isConnected()) {
                nfcA.connect();
                nfcA.setTimeout(timeout);
            }
            //SICLog.i(TAG, "SEND[" + send_data.length + "]: " + Utils.parseByteArrayToHexString(send_data));
            recv = nfcA.transceive(send_data);
            //SICLog.i(TAG, "RECV[" + recv.length + "]: " + Utils.parseByteArrayToHexString(recv));
            response = recv[0];
            if (protection) {
                nfcA.close();
            }
        } catch (Exception e) {
            if (e.getMessage() != null) {
                SICLog.e(TAG, e.getMessage());
                if (e.getMessage().contains("Transceive failed")) {
                    response = Flag.EXCEPTION;
                } else {
                    response = Flag.DISAPPEAR;
                }
            } else {
                response = Flag.DISAPPEAR;
            }
        }
        return recv;
    }

    /**
     * NFC transceive function connects and closes NFC software service
     * automatically.
     *
     * @return data received from NFC.
     */
    public boolean isSendCompleted() {
        return response != Flag.NFC_NAK && response != Flag.DISAPPEAR && response != Flag.EXCEPTION;
    }
}
