/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.interfaces;

import com.sic.module.nfc.tech.mandatories.NfcTypeA;
import com.sic.module.utils.Buffer;
import com.sic.module.utils.Utils;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.1
 * @since 20/Jan/2016
 */
public abstract class IRegister implements IDefine {

    private static final String TAG = IRegister.class.getSimpleName();

    private final NfcTypeA mNfcA;
    private final Buffer mRegBuffer;

    protected IRegister(NfcTypeA nfc) {
        mNfcA = nfc;
        mRegBuffer = Buffer.newInstance(getRegisterPage());
    }

    /**
     * Gets the Clear Flags protocol.
     *
     * @return byte array, data with Clear Flags command
     */
    public static byte[] getPackage_ClearFlags() {
        return new byte[]{(byte) 0xB4, (byte) 0xFF};
    }

    /**
     * Gets the Read Register protocol.
     *
     * @param address register address
     * @return byte array, Read Register command
     */
    public static byte[] getPackage_Read(int address) {
        return new byte[]{(byte) 0xB5, (byte) address};
    }

    /**
     * Gets the Write Register protocol.
     *
     * @param address register address
     * @param data register data
     * @return byte array, Write Register command
     */
    public static byte[] getPackage_Write(int address, int data) {
        return new byte[]{(byte) 0xB6, (byte) address, (byte) data};
    }

    /**
     * Gets the Write Register protocol.
     *
     * @param data register address and data (2 bytes)
     * @return byte array, Write Register command
     */
    public static byte[] getPackage_Write(byte[] data) {
        if (Utils.checkArraySize(data, 2, data.length)) {
            return new byte[]{(byte) 0xB6, data[0], data[1]};
        }
        return null;
    }

    protected abstract int getRegisterPage();

    /**
     * Reads a register buffer at a specific address.
     * If buffer is empty, it will read this register directly.
     *
     * @param address register address.
     * @return register data.
     */
    public byte readBuffer(int address) {
        if (mRegBuffer.isSyncAt(address)) {
            return mRegBuffer.get(address);
        } else {
            Byte recv = read(address);
            if (recv == null) {
                return 0x00;
            }
            return recv;
        }
    }

    /**
     * Reads a register of a SIC's chip at a specific address directly.
     *
     * @param address register address.
     * @return register data.
     */
    public Byte read(int address) {
        byte[] recv;
        recv = mNfcA.autoTransceive(getPackage_Read(address));
        if (mNfcA.isSendCompleted()) {
            mRegBuffer.set(address, recv[1]);
            return recv[1];
        }
        return null;
    }

    /**
     * Reads registers of a SIC's chip directly.
     *
     * @param addrBytes array of register address.
     * @return register data.
     */
    public byte[] read(byte[] addrBytes) {
        int length = addrBytes.length;
        byte[] recv = new byte[length];
        mNfcA.clearDataStream();
        for (int address : addrBytes) {
            mNfcA.addDataStream(getPackage_Read(address & 0xFF));
        }
        mNfcA.stream();
        if (mNfcA.isSendCompleted()) {
            for (int i = 0; i < length; ++i) {
                recv[i] = mNfcA.getDataStreamAt(i)[1];
            }
            mRegBuffer.set(addrBytes, recv);
            return recv;
        }
        return null;
    }

    /**
     * Configures a register of a SIC's chip.
     *
     * @param address register address.
     * @param data register data to be configured.
     */
    public void write(int address, int data) {
        mNfcA.autoTransceive(getPackage_Write(address, data));
        if (mNfcA.isSendCompleted()) {
            mRegBuffer.set(address, (byte) data);
        }
    }

    /**
     * Configures registers of a SIC's chip.
     *
     * @param addr_dataBytes array of register address and data.
     *                       each array must have 2 bytes for address(0) and data(1).
     */
    public void write(byte[]... addr_dataBytes) {
        mNfcA.clearDataStream();
        for (byte[] data : addr_dataBytes) {
            mNfcA.addDataStream(getPackage_Write(data));
        }
        mNfcA.stream();
        if (mNfcA.isSendCompleted()) {
            mRegBuffer.set(addr_dataBytes);
        }
    }

    /**
     * Clear the error flags presenting in the last B_NAK to make the communication process going on.
     */
    public void clearFlags() {
        mNfcA.autoTransceive(getPackage_ClearFlags());
    }

    /**
     * Configures a specific parameter of any register in chip.
     *
     * @param address register address.
     * @param pos bit position of register
     * @param valPos a specific value in a bit position
     *                (another bit out of bit position will be ignored)
     */
    public void writeParams(int address, int pos, int valPos) {
        int bConfig = readBuffer(address) & (~pos & 0xFF);
        valPos &= pos;
        write(address, bConfig | valPos);
    }

    public enum RegisterType {
        NULL, RFU, READ_ONLY_STATUS, READ_WRITE_CONFIG, READ_WRITE, READ_ONLY
    }
}
