/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.sic4310;

import com.sic.module.nfc.tech.chips.SIC4310;
import com.sic.module.nfc.tech.chips.registers.IGPIO;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.2
 * @since 25/Nov/2015
 */
public class GPIO extends IGPIO {

    public static final int Direction_Input = 0;
    public static final int Direction_Output = 1;
    public static final int NoPull = 0;
    public static final int PullUp = 1;
    public static final int Mode_GPIO = 0;
    public static final int Mode_Functional = 1;
    private static final String TAG = GPIO.class.getName();
    private static final byte ADDRESS_GPIO_DIRECTION = Register.GPIO_DIRECTION;
    private static final byte ADDRESS_GPIO_MODE = Register.GPIO_MODE;
    public static int Pin_UART_CTS;
    public static int Pin_UART_TXD;
    public static int Pin_UART_RXD;
    public static int Pin_UART_RTS;

    private static GPIO instance;

    private GPIO(SIC4310 sic4310) {
        super(sic4310);
    }

    public static GPIO getInstance(SIC4310 sic4310) {
        if (instance == null)
            instance = new GPIO(sic4310);
        return instance;
    }

    @Override
    protected void setRegisterAddress() {
        ADDRESS_GPIO_MODE_0 = Register.GPIO_DIRECTION;
        ADDRESS_GPIO_MODE_1 = Register.GPIO_MODE;
        ADDRESS_GPIO_OUT = Register.GPIO_OUT;
        ADDRESS_GPIO_IN = Register.GPIO_IN;
        ADDRESS_GPIO_MODE_2 = Register.GPIO_PU;
    }

    @Override
    protected void setDefaultBit() {
        Pin_UART_CTS = Pin_0;
        Pin_UART_TXD = Pin_1;
        Pin_UART_RXD = Pin_2;
        Pin_UART_RTS = Pin_3;
        Pin_Power_Ready = Pin_4;
        Pin_RF_Busy = Pin_5;
        Pin_RF_Detected = Pin_6;
    }

    @Deprecated
    public Byte getMode(int number) {
        return super.getMode(Address_Mode_1);
    }


    @Deprecated
    public void setMode(int number, int GPIO_Mode) {
        super.setMode(Address_Mode_1, GPIO_Mode);
    }

    @Deprecated
    public void setModeBit(int number, int GPIO_Pin, int GPIO_Mode) {
        super.setModeBit(Address_Mode_1, GPIO_Pin, GPIO_Mode);
    }

    /**
     * get the GPIO direction register directly.
     *
     * @return byte GPIO direction data port value.
     */
    public byte getDirection() {
        return super.getMode(Address_Mode_0);
    }

    /**
     * Sets the GPIO direction register directly.
     *
     * @param GPIO_Direction: Specifies the value to be set to the gpio direction register.
     *                        Each bit can be one of the following values:
     *                        0: {@link #Direction_Input}
     *                        1: {@link #Direction_Output}
     */
    public void setDirection(int GPIO_Direction) {
        super.setMode(Address_Mode_0, GPIO_Direction);
    }

    /**
     * Sets the GPIO direction according to the specified parameters.
     *
     * @param GPIO_Pin:       Specifies the port bit to be set.
     *                        This parameter can be any combination of the following values:
     *                        {@link #Pin_0}: Pin 0
     *                        {@link #Pin_1}: Pin 1
     *                        {@link #Pin_2}: Pin 2
     *                        {@link #Pin_3}: Pin 3
     *                        {@link #Pin_4}: Pin 4
     *                        {@link #Pin_5}: Pin 5
     *                        {@link #Pin_6}: Pin 6
     *                        {@link #Pin_7}: Pin 7
     *                        {@link #Pin_LNib}: Low nibble pins
     *                        {@link #Pin_HNib}: High nibble pins
     *                        {@link #Pin_All}: All pins
     * @param GPIO_Direction: Specifies the value to be set to the gpio direction register.
     *                        This parameter can be {@link #Direction_Input} or {@link #Direction_Output}
     */
    public void setDirectionBit(int GPIO_Pin, int GPIO_Direction) {
        super.setModeBit(Address_Mode_0, GPIO_Pin, GPIO_Direction);
    }

    /**
     * get the GPIO mode register directly.
     *
     * @return byte GPIO mode data port value.
     */
    public Byte getMode() {
        return super.getMode(Address_Mode_1);
    }

    /**
     * Sets the GPIO mode register directly.
     *
     * @param GPIO_Mode: Specifies the value to be set to the gpio mode register.
     *                   Each bit can be one of the following values:
     *                   0: {@link #Mode_GPIO}
     *                   1: {@link #Mode_Functional}
     */
    public void setMode(int GPIO_Mode) {
        super.setMode(Address_Mode_1, GPIO_Mode);
    }

    /**
     * Sets the GPIO mode according to the specified parameters.
     *
     * @param GPIO_Pin:  Specifies the port bit to be set.
     *                   This parameter can be any combination of the following values:
     *                   {@link #Pin_0}: Pin 0
     *                   {@link #Pin_1}: Pin 1
     *                   {@link #Pin_2}: Pin 2
     *                   {@link #Pin_3}: Pin 3
     *                   {@link #Pin_4}: Pin 4
     *                   {@link #Pin_5}: Pin 5
     *                   {@link #Pin_6}: Pin 6
     *                   {@link #Pin_7}: Pin 7
     *                   {@link #Pin_LNib}: Low nibble pins
     *                   {@link #Pin_HNib}: High nibble pins
     *                   {@link #Pin_All}: All pins
     * @param GPIO_Mode: Specifies the value to be set to the gpio mode register.
     *                   This parameter can be {@link #Mode_GPIO} or {@link #Mode_Functional}
     */
    public void setModeBit(int GPIO_Pin, int GPIO_Mode) {
        super.setModeBit(Address_Mode_1, GPIO_Pin, GPIO_Mode);
    }

    /**
     * get the GPIO pull-up register directly.
     *
     * @return byte GPIO pull-up data port value.
     */
    public Byte getPullUp() {
        return super.getMode(Address_Mode_2);
    }


    /**
     * Sets the GPIO pull-up register directly.
     *
     * @param GPIO_PullUp: Specifies the value to be set to the gpio pull-up register.
     *                     Each bit can be one of the following values:
     *                     0: No internal pull-up pin
     *                     1: Use the internal pull-up pin
     */
    public void setPullUp(int GPIO_PullUp) {
        super.setMode(Address_Mode_2, GPIO_PullUp);
    }

    /**
     * Set current GPIO pull-up resistor of particular pin when IO is set to GPIO
     * and input.
     *
     * @param GPIO_Pin:    Specifies the port bit to be set.
     *                     This parameter can be any combination of the following values:
     *                     {@link #Pin_0}: Pin 0
     *                     {@link #Pin_1}: Pin 1
     *                     {@link #Pin_2}: Pin 2
     *                     {@link #Pin_3}: Pin 3
     *                     {@link #Pin_4}: Pin 4
     *                     {@link #Pin_5}: Pin 5
     *                     {@link #Pin_6}: Pin 6
     *                     {@link #Pin_7}: Pin 7
     *                     {@link #Pin_LNib}: Low nibble pins
     *                     {@link #Pin_HNib}: High nibble pins
     *                     {@link #Pin_All}: All pins
     * @param GPIO_PullUp: Specifies the value to be set to the gpio pull-up register.
     *                     This parameter can be {@link #NoPull} or {@link #PullUp}
     */
    public void setBitPullup(int GPIO_Pin, int GPIO_PullUp) {
        super.setModeBit(Address_Mode_2, GPIO_Pin, GPIO_PullUp);
    }

    /**
     * Sets the GPIO according to the GPIO input mode.
     *
     * @param GPIO_Pin: Specifies the port bit to be set.
     *                  This parameter can be any combination of the following values:
     *                  {@link #Pin_0}: Pin 0
     *                  {@link #Pin_1}: Pin 1
     *                  {@link #Pin_2}: Pin 2
     *                  {@link #Pin_3}: Pin 3
     *                  {@link #Pin_4}: Pin 4
     *                  {@link #Pin_5}: Pin 5
     *                  {@link #Pin_6}: Pin 6
     *                  {@link #Pin_7}: Pin 7
     *                  {@link #Pin_LNib}: Low nibble pins
     *                  {@link #Pin_HNib}: High nibble pins
     *                  {@link #Pin_All}: All pins
     */
    public void setGPIOModeInputDirection(int GPIO_Pin) {
        byte direction = (byte) (mRegister.readBuffer(ADDRESS_GPIO_DIRECTION) & ~GPIO_Pin);
        byte mode = (byte) (mRegister.readBuffer(ADDRESS_GPIO_MODE) & ~GPIO_Pin);
        mRegister.write(
                new byte[]{ADDRESS_GPIO_DIRECTION, direction},
                new byte[]{ADDRESS_GPIO_MODE, mode});
    }

    /**
     * Sets the GPIO according to the GPIO output mode.
     *
     * @param GPIO_Pin: Specifies the port bit to be set.
     *                  This parameter can be any combination of the following values:
     *                  {@link #Pin_0}: Pin 0
     *                  {@link #Pin_1}: Pin 1
     *                  {@link #Pin_2}: Pin 2
     *                  {@link #Pin_3}: Pin 3
     *                  {@link #Pin_4}: Pin 4
     *                  {@link #Pin_5}: Pin 5
     *                  {@link #Pin_6}: Pin 6
     *                  {@link #Pin_7}: Pin 7
     *                  {@link #Pin_LNib}: Low nibble pins
     *                  {@link #Pin_HNib}: High nibble pins
     *                  {@link #Pin_All}: All pins
     */
    public void setGPIOModeOutputDirection(int GPIO_Pin) {
        byte direction = (byte) (mRegister.readBuffer(ADDRESS_GPIO_DIRECTION) | GPIO_Pin);
        byte mode = (byte) (mRegister.readBuffer(ADDRESS_GPIO_MODE) & ~GPIO_Pin);
        mRegister.write(
                new byte[]{ADDRESS_GPIO_DIRECTION, direction},
                new byte[]{ADDRESS_GPIO_MODE, mode});
    }

    /**
     * Sets the GPIO according to the Functional input mode.
     *
     * @param GPIO_Pin: Specifies the port bit to be set.
     *                  This parameter can be any combination of the following values:
     *                  {@link #Pin_0}: Pin 0
     *                  {@link #Pin_1}: Pin 1
     *                  {@link #Pin_2}: Pin 2
     *                  {@link #Pin_3}: Pin 3
     *                  {@link #Pin_4}: Pin 4
     *                  {@link #Pin_5}: Pin 5
     *                  {@link #Pin_6}: Pin 6
     *                  {@link #Pin_7}: Pin 7
     *                  {@link #Pin_LNib}: Low nibble pins
     *                  {@link #Pin_HNib}: High nibble pins
     *                  {@link #Pin_All}: All pins
     */
    public void setFunctionModeInputDirection(int GPIO_Pin) {
        byte direction = (byte) (mRegister.readBuffer(ADDRESS_GPIO_DIRECTION) & ~GPIO_Pin);
        byte mode = (byte) (mRegister.readBuffer(ADDRESS_GPIO_MODE) | GPIO_Pin);
        mRegister.write(
                new byte[]{ADDRESS_GPIO_DIRECTION, direction},
                new byte[]{ADDRESS_GPIO_MODE, mode});
    }

    /**
     * Sets the GPIO according to the Functional output mode.
     *
     * @param GPIO_Pin: Specifies the port bit to be set.
     *                  This parameter can be any combination of the following values:
     *                  {@link #Pin_0}: Pin 0
     *                  {@link #Pin_1}: Pin 1
     *                  {@link #Pin_2}: Pin 2
     *                  {@link #Pin_3}: Pin 3
     *                  {@link #Pin_4}: Pin 4
     *                  {@link #Pin_5}: Pin 5
     *                  {@link #Pin_6}: Pin 6
     *                  {@link #Pin_7}: Pin 7
     *                  {@link #Pin_LNib}: Low nibble pins
     *                  {@link #Pin_HNib}: High nibble pins
     *                  {@link #Pin_All}: All pins
     */
    public void setFunctionModeOutputDirection(int GPIO_Pin) {
        byte direction = (byte) (mRegister.readBuffer(ADDRESS_GPIO_DIRECTION) | GPIO_Pin);
        byte mode = (byte) (mRegister.readBuffer(ADDRESS_GPIO_MODE) | GPIO_Pin);
        mRegister.write(
                new byte[]{ADDRESS_GPIO_DIRECTION, direction},
                new byte[]{ADDRESS_GPIO_MODE, mode});
    }
}
