/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips;

import android.content.Context;

import com.sic.module.nfc.tech.chips.sic4310.UART;
import com.sic.module.nfc.tech.chips.sic431x.Flag;
import com.sic.module.nfc.tech.chips.sic431x.Status;
import com.sic.module.utils.Utils;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.2
 * @since 25/Nov/2015
 */
public abstract class SIC431x extends SIC43xx {

    private static final String TAG = SIC431x.class.getName();

    private Status mStatus;

    protected SIC431x(Context context) {
        super(context);
        setPeripheral();
        mStatus = Status.getInstance(this);
    }

    protected abstract void setPeripheral();

    @Override
    public int addDataStream(byte[]... buffer) {
        int index = 0;
        for (byte[] temp : buffer) {
            byte[][] chunk = Utils.chunkSplit(temp, UART.MAX_SIZE_FIFO + 1);
            index = super.addDataStream(chunk);
        }
        return index;
    }

    /**
     * Get status object for manage the status' chip
     *
     * @return status object
     */
    public Status getStatus() {
        return mStatus;
    }

    /**
     * Checks the latest response flag for a Downlink FIFO Overflow.
     *
     * @return true, if a Downlink FIFO overflow occurs.
     */
    public boolean isDownlinkFIFOOverflow() {
        byte flag = (Flag.B_NAK | Flag.B_NAK_DL_FIFO_OVF);
        return (response & flag) == flag;
    }

    /**
     * Checks the latest response flag for a Uplink FIFO Empty.
     *
     * @return true, if a Uplink FIFO is empty.
     */
    public boolean isUplinkFIFOEmpty() {
        byte flag = (Flag.B_NAK | Flag.B_NAK_UL_FIFO_EMP);
        return (response & flag) == flag;
    }

    /**
     * Checks the latest response flag for a Uplink FIFO Overflow.
     *
     * @return true, if a Uplink FIFO overflow occurs.
     */
    public boolean isUplinkFIFOOverflow() {
        byte flag = (Flag.B_NAK | Flag.B_NAK_UL_FIFO_OVF);
        return (response & flag) == flag;
    }

    /**
     * Checks the latest response flag for a insufficient power to source load.
     *
     * @return true, if a RSPW_RDY status becomes '0'.
     */
    public boolean isInsufficientPowerToSourceLoad() {
        byte flag = (Flag.B_NAK | Flag.B_NAK_RSPW_LOW);
        return (response & flag) == flag;
    }

    /**
     * Checks the latest response flag for a voltage on XVDD drop.
     *
     * @return true, if a XVDD_RDY status becomes '0'.
     */
    public boolean isVoltageOnXVddDrop() {
        byte flag = (Flag.B_NAK | Flag.B_NAK_XVDD_DROP);
        return (response & flag) == flag;
    }

    /**
     * Checks the latest response flag for a UART peripheral.
     *
     * @return true, if a UART_RDY status becomes '0'.
     */
    public boolean isUARTFail() {
        byte flag = (Flag.B_NAK | Flag.B_NAK_UART_FAIL);
        return (response & flag) == flag;
    }
}
