/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.utils;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Handler;
import android.os.Looper;
import android.provider.Settings;
import android.text.Spannable;
import android.text.SpannableString;
import android.text.style.ForegroundColorSpan;
import android.util.TypedValue;
import android.widget.TextView;

import java.math.BigInteger;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Locale;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.3
 * @since 14/Oct/2015
 */
public class Utils {

    public static final byte BIT_7 = (byte) 0x80;
    public static final byte BIT_6 = (byte) 0x40;
    public static final byte BIT_5 = (byte) 0x20;
    public static final byte BIT_4 = (byte) 0x10;
    public static final byte BIT_3 = (byte) 0x08;
    public static final byte BIT_2 = (byte) 0x04;
    public static final byte BIT_1 = (byte) 0x02;
    public static final byte BIT_0 = (byte) 0x01;
    private static final String TAG = Utils.class.getName();
    private static final String TAG_DIALOG_FRAGMENT = "dialog_fragment";
    private static final Handler mHandler = new Handler(Looper.getMainLooper());
    private static Utils instance;
    private final Context mContext;

    private Utils(Context context) {
        mContext = context;
    }

    private Utils() {
        this(AppContext.getInstance().getContext());
    }

    public static Utils getInstance() {
        if (instance == null)
            instance = new Utils();
        return instance;
    }

    public static Utils getInstance(Context context) {
        if (instance == null)
            instance = new Utils(context);
        return instance;
    }

    // String to ...

    /**
     * Parse decimal string to hex string.
     *
     * @param string decimal string.
     * @return hex string.
     */
    public static String parseDecimalStringToHexString(String string) {
        return String.format("%X ", new BigInteger(string.getBytes(/* YOUR_CHARSET? */)));
    }

    /**
     * Parse ASCII string to byte array.
     *
     * @param ascii ascii string.
     * @return byte array.
     */
    public static byte[] parseASCIIStringToByteArray(String ascii) {
        int length = ascii.length();
        byte[] b = new byte[length];
        for (int i = 0; i < length; ++i) {
            b[i] = (byte) ascii.charAt(i);
        }
        return b;
    }

    /**
     * Parse hex string to byte array.
     *
     * @param sarray Hex string.
     * @return byte array.
     */
    public static byte[] parseHexStringToByteArray(String sarray) {
        List<String> separated = new ArrayList<>();
        int index = 0;
        while (index < sarray.length()) {
            separated.add(sarray.substring(index,
                    Math.min(index + 2, sarray.length())));
            index += 2;
        }
        byte[] b = new byte[separated.size()];
        for (int i = 0; i < separated.size(); ++i) {
            try {
                b[i] = (byte) Integer.parseInt(separated.get(i), 16);
            } catch (NumberFormatException e) {
                b[i] = (byte) 0xFF;
            }
        }
        return b;
    }

    /**
     * Parse string to byte array, can chooses
     *
     * @param sarray Hex string.
     * @param radix  radix for number
     * @return byte array.
     */
    public static byte[] parseStringToByteArray(String sarray, int radix) {
        List<String> separated = new ArrayList<>();
        int index = 0;
        int width = (int) Math.ceil((Math.log10(0xFF) / Math.log10(radix)));
        while (index < sarray.length()) {
            String s = sarray.substring(index,
                    Math.min(index + width, sarray.length()));
            if (Integer.parseInt(s, radix) > 255) {
                sarray = sarray.substring(0, index) + "0"
                        + sarray.substring(index, sarray.length());
                s = sarray.substring(index,
                        Math.min(index + width, sarray.length()));
            }
            separated.add(s);
            index += width;
        }
        byte[] b = new byte[separated.size()];
        for (int i = 0; i < separated.size(); ++i) {
            try {
                b[i] = (byte) Integer.parseInt(separated.get(i), radix);
            } catch (NumberFormatException e) {
                b[i] = (byte) 0xFF;
            }
        }
        return b;
    }

    // Integer to ...

    /**
     * Parse integer to byte array. It is can limit array.
     *
     * @param intad integer.
     * @return byte array.
     */
    public static byte[] parseIntegerToByteArray(int intad) {
        String HexStr = Integer.toHexString(intad);
        return parseHexStringToByteArray(HexStr);
    }

    /**
     * Parse integer to byte array. It is can limit array.
     *
     * @param intad  integer.
     * @param length length of byte array.
     * @return byte array.
     */
    public static byte[] parseIntegerToByteArray(int intad, int length) {
        String HexStr = Integer.toHexString(intad);
        length <<= 1;
        while (HexStr.length() < length) {
            HexStr = "0" + HexStr;
        }
        if (HexStr.length() > length) {
            HexStr = HexStr.substring(HexStr.length() - length);
        }
        return parseHexStringToByteArray(HexStr);
    }

    // Byte to ...
    public static String parseByteToHexString(byte bytead) {
        return ("" + String.format("%02x", bytead & 0xff)).toUpperCase();
    }

    // Byte Array to ...

    /**
     * Parse byte array to ASCII string.
     *
     * @param barray byte array.
     * @return ASCII.
     */
    public static String parseByteArrayToASCIIString(byte[] barray) {
        StringBuilder strBuilder = new StringBuilder();
        for (byte aBarray : barray) {
            strBuilder.append((char) aBarray);
        }
        return strBuilder.toString();
    }

    /**
     * Parse byte array to hex string.
     *
     * @param barray byte array.
     * @return hex string.
     */
    public static String parseByteArrayToHexString(byte[] barray) {
        StringBuilder hexString = new StringBuilder("");
        if (barray == null) {
            return "";
        }
        for (byte b : barray) {
            hexString.append(String.format("%02x", b & 0xff));
        }
        return hexString.toString();
    }

    /**
     * Parse byte array to string by radix.
     *
     * @param barray byte array
     * @param radix  base number.
     * @return default string.
     */
    public static String parseByteArrayToString(byte[] barray, int radix) {
        StringBuilder string = new StringBuilder("");
        String temp;
        int width = (int) Math.ceil((Math.log10(0xFF) / Math.log10(radix)));
        for (byte b : barray) {
            temp = Integer.toString((int) b & 0xFF, radix);
            while (temp.length() % width != 0) {
                temp = "0" + temp;
            }
            string.append(temp);
        }

        return string.toString();
    }

    /**
     * Get current date and time
     *
     * @return date and time string format.
     */
    public static String getDateAndTimeString() {
        // date and time;
        Calendar calendar = Calendar.getInstance();
        Date date = calendar.getTime();
        SimpleDateFormat format = new SimpleDateFormat("   dd/MM/yyyy       HH:mm:ss    ", Locale.US);
        return format.format(date);
    }

    /**
     * Get current date and time
     *
     * @return date and time string format.
     */
    public static String getTimeString() {
        // date and time;
        Calendar calendar = Calendar.getInstance();
        Date date = calendar.getTime();
        SimpleDateFormat format = new SimpleDateFormat("HH:mm:ss", Locale.US);
        return format.format(date);
    }

    // dd
    public static String getCapitalizeFirstLetter(String original) {
        if (original.length() == 0)
            return original;
        return original.substring(0, 1).toUpperCase() + original.substring(1).toLowerCase();
    }

    public static byte[] concatenateByteArray(byte[]... arrays) {
        // Determine the length of the result array
        int totalLength = 0;
        for (byte[] array : arrays) {
            totalLength += array.length;
        }

        // create the result array
        byte[] result = new byte[totalLength];

        // copy the source arrays into the result array
        int currentIndex = 0;
        for (byte[] array : arrays) {
            System.arraycopy(array, 0, result, currentIndex, array.length);
            currentIndex += array.length;
        }

        return result;
    }

    public static void scrollAmountTextView(TextView view) {
        try {
            final int scrollAmount = view.getLayout().getLineTop(view.getLineCount()) - view.getHeight();
            // if there is no need to scroll, scrollAmount will be <=0
            if (scrollAmount > 0) {
                view.scrollTo(0, scrollAmount);
            } else {
                view.scrollTo(0, 0);
            }
        } catch (NullPointerException ignored) {

        }
    }

    public static String getOrdinal(int packetNumber) {
        String ordinal;
        if (packetNumber / 10 % 10 == 1) {
            ordinal = packetNumber + "th";
        } else {
            switch (packetNumber % 10) {
                case 1:
                    ordinal = packetNumber + "st";
                    break;
                case 2:
                    ordinal = packetNumber + "nd";
                    break;
                case 3:
                    ordinal = packetNumber + "rd";
                    break;
                default:
                    ordinal = packetNumber + "th";
                    break;
            }
        }
        return ordinal;
    }

    public static byte checkSum(byte[] input) {
        byte sum = 0x00;
        for (int i = 0; i < input.length - 1; i++) {
            sum ^= input[i];
        }
        return sum;
    }


    public static byte checkSum(byte[] data, int start, int length) {
        byte crc = 0x00;
        if (data == null) {
            return 0x00;
        }

        if (start < 0 || length < 1 || start + length > data.length + 1) {
            return 0x00;
        }
        for (int i = start; i < start + length - 1; ++i) {
            crc ^= data[i];
        }
        return crc;
    }

    public static Spannable toSpannable(String text) {
        return new SpannableString(text);
    }

    public static Spannable setSpannable(Spannable text, int color) {
        text.setSpan(new ForegroundColorSpan(color), 0, text.length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
        return text;
    }


    public static Spannable setSpannable(Spannable fulltext, String subtext, int color) {
        int index = fulltext.toString().indexOf(subtext);
        if (index < 0) {
            return fulltext;
        }
        fulltext.setSpan(new ForegroundColorSpan(color), index, index + subtext.length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);

        return fulltext;
    }

    public static Spannable setSpannable(Spannable text, int index, int length, int color) {
        if (index + length > text.length() || index < 0 || length < 0) {
            return text;
        }
        text.setSpan(new ForegroundColorSpan(color), index, index + length, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);

        return text;
    }

    public static boolean checkArraySize(byte[] data, int min_size, int max_size) {
        return data != null && !(min_size < 0 || data.length < min_size || data.length > max_size);
    }

    public static byte[][] chunkSplit(byte[] data, int size) {
        if (checkArraySize(data, 1, data.length)) {
            int dalen = data.length;
            int maxlen = (int) Math.ceil(dalen / (double) size);
            byte[][] buffer = new byte[maxlen--][];

            for (int i = 0; i < maxlen; ++i) {
                buffer[i] = new byte[size];
                System.arraycopy(data, i * size, buffer[i], 0, size);
            }
            if (dalen % size > 0) {
                size = dalen % size;
            }
            buffer[maxlen] = new byte[size];
            System.arraycopy(data, dalen - size, buffer[maxlen], 0, size);
            return buffer;
        }
        return null;
    }

    public static byte[] getPackageLimited(byte cmd, byte[] data, int size) {
        if (data == null) {
            return null;
        }
        int length = Math.min(size, data.length);
        byte[] buffer = new byte[length + 1];
        buffer[0] = cmd;
        System.arraycopy(data, 0, buffer, 1, length);
        return buffer;
    }

    public static byte[][] getPackageSplit(byte cmd, byte[][] data, int size) {
        return getPackageSplit(cmd, cmd, data, size);
    }

    public static byte[][] getPackageSplit(byte cmd, byte lastcmd, byte[][] data, int size) {
        List<byte[]> buffer = new ArrayList<>();
        for (byte[] each : data) {
            if (Utils.checkArraySize(each, 1, each.length)) {
                int dalen = each.length;
                int localSize = size;
                int maxlen = (int) Math.ceil(dalen / (double) localSize) - 1;
                byte[] temp;

                for (int i = 0; i < maxlen; ++i) {
                    temp = new byte[localSize + 1];
                    temp[0] = cmd;
                    System.arraycopy(each, i * localSize, temp, 1, localSize);
                    buffer.add(temp);
                }
                if (dalen % localSize > 0) {
                    localSize = dalen % localSize;
                }
                temp = new byte[localSize + 1];
                temp[0] = lastcmd;
                System.arraycopy(each, dalen - localSize, temp, 1, localSize);
                buffer.add(temp);
            }
        }
        byte[][] array = new byte[buffer.size()][];
        return buffer.toArray(array);
    }

    public static synchronized Handler postDelayed(Runnable runnable, long delay) {
        mHandler.postDelayed(runnable, delay);
        return mHandler;
    }

    public static synchronized void postCancelled() {
        if (mHandler != null) {
            mHandler.removeCallbacksAndMessages(null);
        }
    }

    public String getDeviceId() {
        return Settings.Secure.getString(mContext.getContentResolver(), Settings.Secure.ANDROID_ID);
    }

    public String getVersionName() {
        try {
            PackageInfo pInfo = mContext.getPackageManager().getPackageInfo(mContext.getPackageName(), 0);
            return pInfo.versionName;
        } catch (Exception e) {
            return "1.0";
        }
    }

    /**
     * Convert Dp to Pixel
     *
     * @param dp unit
     * @return Pixel unit
     */
    public int dpToPx(float dp) {
        float px = TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dp, mContext.getResources().getDisplayMetrics());
        return (int) px;
    }

    public List<ApplicationInfo> getInstalledApplication() {
        PackageManager packageManager = mContext.getPackageManager();
        List<ApplicationInfo> apps = packageManager.getInstalledApplications(0);
        Collections.sort(apps, new ApplicationInfo.DisplayNameComparator(packageManager));
        return apps;
    }
}
