/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.optionals;

import android.content.Context;
import android.content.Intent;
import android.nfc.FormatException;
import android.nfc.NdefMessage;
import android.nfc.NdefRecord;
import android.nfc.Tag;
import android.nfc.tech.Ndef;
import android.nfc.tech.NdefFormatable;

import com.sic.module.nfc.Nfc;
import com.sic.module.utils.AppContext;
import com.sic.module.utils.SICLog;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.2
 * @since 25/Nov/2015
 */
public class NfcTypeNdef extends Nfc {

    private static final String TAG = NfcTypeNdef.class.getName();

    private static NfcTypeNdef instance;
    private final Context mContext;
    private Ndef ndef;
    private NdefFormatable ndefFormatable;
    private List<NdefRecord> recordList = new ArrayList<>();

    private NfcTypeNdef(Context context) {
        super(context, null, new String[][]{new String[]{Ndef.class.getName(), NdefFormatable.class.getName()}});
        mContext = context;
    }

    private NfcTypeNdef() {
        this(AppContext.getInstance().getContext());
    }

    public static NfcTypeNdef getInstance() {
        if (instance == null)
            instance = new NfcTypeNdef();
        return instance;
    }

    public static NfcTypeNdef getInstance(Context context) {
        if (instance == null)
            instance = new NfcTypeNdef(context);
        return instance;
    }

    protected Context getContext() {
        return mContext;
    }

    @Override
    public boolean onTagFinding(Intent intent) {
        if (super.onTagFinding(intent)) {
            ndef = Ndef.get(getTag());
            ndefFormatable = NdefFormatable.get(getTag());
            return true;
        }
        return false;
    }

    /**
     * Call this method in your activity's onNewIntent(Intent) method body.
     * This function be used for optional tag.
     *
     * @param tag  NFC tag
     */
    public void onTagFinding(Tag tag) {
        ndef = Ndef.get(tag);
        ndefFormatable = NdefFormatable.get(tag);
    }

    /**
     * NDEF texture record format
     *
     * @param text message
     * @return NdefRecord
     * @throws UnsupportedEncodingException Unsupported encoding
     */
    public NdefRecord createTextRecord(String text)
            throws UnsupportedEncodingException {
        // create the message in according with the standard
        String lang = "en";
        byte[] textBytes = text.getBytes();
        byte[] langBytes = lang.getBytes("US-ASCII");
        int langLength = langBytes.length;
        int textLength = textBytes.length;

        byte[] payload = new byte[1 + langLength + textLength];
        payload[0] = (byte) langLength;

        // copy lang bytes and text bytes into payload
        System.arraycopy(langBytes, 0, payload, 1, langLength);
        System.arraycopy(textBytes, 0, payload, 1 + langLength, textLength);

        return new NdefRecord(NdefRecord.TNF_WELL_KNOWN, NdefRecord.RTD_TEXT, new byte[0], payload);
    }

    public NdefMessage createDataRecord(byte[] payload)
            throws FormatException {
        return new NdefMessage(payload);
    }
    /**
     * NDEF application record format
     *
     * @param appLocation locate of application.
     * @return NdefRecord
     */
    public NdefRecord createAppRecord(String appLocation) {
        return NdefRecord.createApplicationRecord(appLocation);
    }

    /**
     * NDEF URI record format
     *
     * @param uriLocation locate of application.
     * @return NdefRecord
     */
    public NdefRecord createURIRecord(String uriLocation) {
        return NdefRecord.createUri(uriLocation);
    }

    /**
     * add new record in NDEF format
     *
     * @param newRecord is a NdefRecord.
     */
    public void addRecords(NdefRecord newRecord) {
        recordList.add(newRecord);
    }

    /**
     * remove record in NDEF format form index.
     *
     * @param index the index of NdefRecord.
     */
    public void removeRecords(int index) {
        if (index < 0 || index > recordList.size()) {
            recordList.remove(index);
        }
    }

    /**
     * clear record in NDEF format.
     */
    public void clearRecords() {
        recordList.clear();
    }

    /**
     * NDEF transceiver function. it connect and close nfc software service
     * automatically.
     *
     * @return true if transmit is successful.
     * @throws IOException     is occurs if format tag is not support NDEF or tag is lost.
     * @throws FormatException is occurs if tag doesn't appear to support NDEF format.
     */
    public boolean writeNDEF() throws IOException, FormatException {

        try {
            int index = 0;
            NdefRecord[] record = new NdefRecord[recordList.size()];
            for (NdefRecord ndefRecord: recordList) {
                record[index++] = ndefRecord;
            }

            NdefMessage message = new NdefMessage(record);
            // see if tag is already NDEF formatted
            if (ndef != null) {
                ndef.close();
                if (!ndef.isConnected()) {
                    ndef.connect();
                }
                if (!ndef.isWritable()) {
                    SICLog.i(TAG, "Read-only tag.");
                    ndef.close();
                    return false;
                }

                // work out how much space we need for the data
                int size = message.toByteArray().length;
                if (ndef.getMaxSize() < size) {
                    SICLog.i(TAG, "Tag doesn't have enough free space.");
                    ndef.close();
                    return false;
                }

                ndef.writeNdefMessage(message);
                SICLog.i(TAG, "Tag written successfully.");
                ndef.close();
                return true;

            } else if (ndefFormatable != null) {
                try {
                    ndefFormatable.connect();
                    ndefFormatable.format(message);
                    SICLog.i(TAG, "Tag written successfully!");
                    ndefFormatable.close();
                    return true;
                } catch (IOException e) {
                    SICLog.e(TAG, "Unable to format tag to NDEF.");
                    throw new IOException();
                }
            } else {
                SICLog.i(TAG, "Tag doesn't appear to support NDEF format.");
                return false;
            }
        } catch (Exception e) {
            SICLog.e(TAG, "Failed to write tag" + e.toString());
            throw new IOException();
        }
    }

    public Ndef getRawNdef() {
        return ndef;
    }

    public NdefFormatable getRawNdefFormatable() {
        return ndefFormatable;
    }

}
