/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.sic431x;

import com.sic.module.nfc.tech.chips.SIC431x;
import com.sic.module.nfc.tech.chips.registers.IStatus;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.2
 * @since 25/Nov/2015
 */
public class Status extends IStatus {

    private static final byte ADDRESS_UART_STATUS = 0x00;
    private static final byte ADDRESS_OSC_TUNING = 0x07;

    public static final byte CTS = BIT5;
    public static final byte RTS = BIT4;
    public static final byte DL_FF_EMT = BIT3;
    public static final byte DL_FF_OVF = BIT2;
    public static final byte UL_FF_EMT = BIT1;
    public static final byte UL_FF_OVF = BIT0;

    public static final byte UART_RDY = BIT3;

    private static final String TAG = Status.class.getName();

    private static Status instance;

    private Status(SIC431x sic431x) {
        super(sic431x);
    }

    public static Status getInstance(SIC431x sic431x) {
        if (instance == null)
            instance = new Status(sic431x);
        return instance;
    }

    @Override
    protected void setRegisterAddress() {
        ADDRESS_POWER_STATUS = 0x01;
    }

    @Override
    protected void setDefaultBit() {
        SCAP_RDY = BIT4;
        XVDD_RDY = BIT2;
        RSPW_RDY = BIT1;
        LDO_ON = BIT0;
    }

    /**
     * Gets the UART status on register.
     *
     * @return  byte UART status data value.
     */
    public Byte getUARTStatus() {
        return getStatus(ADDRESS_POWER_STATUS);
    }

    /**
     * Checks the on-chip oscillator operate the UART communication.
     *
     * @return true, if this oscillator is stable and ready.
     */
    public boolean isPowerUARTReady() {
        Byte flag = getPowerStatus();
        return flag != null && ((flag & UART_RDY) == UART_RDY);
    }

    /**
     * Checks the external UART device.
     *
     * @return true, if this device is ready to receive data.
     */
    public boolean isExternalUARTReady() {
        Byte flag = getUARTStatus();
        return flag != null && ((flag & CTS) == CTS);
    }

    /**
     * Checks the Uplink FIFO in chip.
     *
     * @return true, if this FIFO is ready to receive data.
     */
    public boolean isUplinkFIFOReady() {
        Byte flag = getUARTStatus();
        return flag != null && ((flag & RTS) == RTS);
    }

    /**
     * Checks the Downlink FIFO Empty indicator
     *
     * @return true, if this FIFO is empty.
     */
    public boolean isDownlinkFIFOEmpty() {
        Byte flag = getUARTStatus();
        return flag != null && ((flag & DL_FF_EMT) == DL_FF_EMT);
    }

    /**
     * Checks the Downlink FIFO Overflow Indicator flag.
     * Can be clear by Clear_Flag command.
     *
     * @return true, if this FIFO is overflow.
     */
    public boolean isDownlinkFIFOOverflow() {
        Byte flag = getUARTStatus();
        return flag != null && ((flag & DL_FF_OVF) == DL_FF_OVF);
    }

    /**
     * Checks the Uplink FIFO Empty indicator
     *
     * @return true, if this FIFO is empty.
     */
    public boolean isUplinkFIFOEmpty() {
        Byte flag = getUARTStatus();
        return flag != null && ((flag & UL_FF_EMT) == UL_FF_EMT);
    }

    /**
     * Checks the Uplink FIFO Overflow Indicator flag.
     * Can be clear by Clear_Flag command.
     *
     * @return true, if this FIFO is overflow.
     */
    public boolean isUplinkFIFOOverflow() {
        Byte flag = getUARTStatus();
        return flag != null && ((flag & UL_FF_OVF) == UL_FF_OVF);
    }
}
