/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.sic4310;

import com.sic.module.nfc.tech.chips.SIC4310;
import com.sic.module.nfc.tech.interfaces.IDevice;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.2
 * @since 25/Nov/2015
 */
public class Peripheral extends IDevice {

    private static final String TAG = Peripheral.class.getName();
    private static final int ADDRESS_PERIPHERAL_CONFIG = Register.PERIPHERAL_CONFIG;
    private static final int ADDRESS_PERIPHERAL_ADJUSTMENT = Register.PERIPHERAL_ADJUSTMENT;

    public static final byte CONFIG_PW_LV = Bit.parseBit(5, 4);
    public static final byte CONFIG_PWCHK_EN = Bit.parseBit(2);
    public static final byte CONFIG_LDO_EN = Bit.parseBit(1);
    public static final byte CONFIG_OSC_EN = Bit.parseBit(0);

    public static final byte ADJUSTMENT_LDO_D_LV = Bit.parseBit(2);
    public static final byte ADJUSTMENT_RFLM_LV = Bit.parseBit(1);
    public static final byte ADJUSTMENT_RESP_2B_FLAG = Bit.parseBit(0);

    public static final byte PowerLv_500_UA = Bit.parseBit();
    public static final byte PowerLv_1250_UA = Bit.parseBit(4);
    public static final byte PowerLv_2500_UA = Bit.parseBit(5);
    public static final byte PowerLv_5000_UA = Bit.parseBit(5, 4);
    public static final byte PowerCheck_Disable = Bit.parseBit();
    public static final byte PowerCheck_Enable = Bit.parseBit(2);
    public static final byte LDO_Disable = Bit.parseBit();
    public static final byte LDO_Enable = Bit.parseBit(1);
    public static final byte OSC_Disable = Bit.parseBit();
    public static final byte OSC_Enable = Bit.parseBit(0);

    public static final byte LDODropLv_2_4 = Bit.parseBit();
    public static final byte LDODropLv_2_7 = Bit.parseBit(2);
    public static final byte RFLimitLv_5_2 = Bit.parseBit();
    public static final byte RFLimitLv_6_5 = Bit.parseBit(1);
    public static final byte RespBitFlag_1 = Bit.parseBit();
    public static final byte RespBitFlag_2 = Bit.parseBit(0);

    private static Peripheral instance;

    private Peripheral(SIC4310 sic4310) {
        super(sic4310.getRegister());
    }

    public static Peripheral getInstance(SIC4310 sic4310) {
        if (instance == null)
            instance = new Peripheral(sic4310);
        return instance;
    }

    @Override
    protected void setRegisterAddress() {}

    @Override
    protected void setDefaultBit() {}

    /**
     * Sets a threshold of reserved power level received from the coil to turn on LDO for
     * driving an external load on XVDD.
     *
     * @param  PW_Level: This parameter contains a threshold of reserved power level
     *          This parameter can be one of the following values:
     *            {@link #PowerLv_500_UA}: 500 µA
     *            {@link #PowerLv_1250_UA}: 1.25 mA
     *            {@link #PowerLv_2500_UA}: 2.50 mA
     *            {@link #PowerLv_5000_UA}: 5.00 mA
     */
    public void setPowerLevel(int PW_Level) {
        mRegister.writeParams(ADDRESS_PERIPHERAL_CONFIG, CONFIG_PW_LV, PW_Level);
    }

    /**
     * Sets a qualifying process before turning on the on-chip LDO.
     *
     * @param  enabled: This parameter contains a checking power
     *          This parameter can be one of the following values:
     *            false: Disabled ( on-chip LDO is turned on immediately when LDO_EN is set to '1'
     *                without qualifying process and RSPW_RDY is always be '1')
     *            true: Enabled
     */
    public void setPowerCheckEnabled(boolean enabled) {
        if (enabled) {
            mRegister.writeParams(ADDRESS_PERIPHERAL_CONFIG, CONFIG_PWCHK_EN, PowerCheck_Enable);
        } else {
            mRegister.writeParams(ADDRESS_PERIPHERAL_CONFIG, CONFIG_PWCHK_EN, PowerCheck_Disable);
        }
    }

    /**
     * Sets the on-chip LDO Regulator.
     *
     * @param  enabled: This parameter contains a checking power
     *          This parameter can be one of the following values:
     *            false: Disables LDO
     *            true: Enabled LDO
     */
    public void setLDORegulatorEnabled(boolean enabled) {
        if (enabled) {
            mRegister.writeParams(ADDRESS_PERIPHERAL_CONFIG, CONFIG_LDO_EN, LDO_Enable);
        } else {
            mRegister.writeParams(ADDRESS_PERIPHERAL_CONFIG, CONFIG_LDO_EN, LDO_Disable);
        }
    }

    /**
     * Sets the on-chip oscillator for UART communication.
     *
     * @param  enabled: This parameter contains the on-chip oscillator
     *          This parameter can be one of the following values:
     *            false: Disables on-chip oscillator
     *            true: Enabled on-chip oscillator
     */
    public void setUARTOscillatorEnabled(boolean enabled) {
        if (enabled) {
            mRegister.writeParams(ADDRESS_PERIPHERAL_CONFIG, CONFIG_OSC_EN, OSC_Enable);
        } else {
            mRegister.writeParams(ADDRESS_PERIPHERAL_CONFIG, CONFIG_OSC_EN, OSC_Disable);
        }
    }

    /**
     * Sets a threshold voltage drop level at the pin XVDD that set XVDD_RDY flag.
     *
     * @param  LDO_DropLevel: This parameter contains a threshold voltage drop level
     *          This parameter can be one of the following values:
     *            {@link #LDODropLv_2_4}: 2.4 volt
     *            {@link #LDODropLv_2_7}: 2.7 volt
     */
    public void setVoltageDropLevel(int LDO_DropLevel) {
        mRegister.writeParams(ADDRESS_PERIPHERAL_ADJUSTMENT, ADJUSTMENT_LDO_D_LV, LDO_DropLevel);
    }

    /**
     * Sets the RF limiter level to limit a charging voltage on the pin SCAP and pin HV
     *
     * @param  RFLM_Level: This parameter contains the RF limiter level
     *          This parameter can be one of the following values:
     *            {@link #RFLimitLv_5_2}: 5.2 volt (maximum of SCAP voltage = 4.8 volt)
     *            {@link #RFLimitLv_6_5}: 6.5 volt (maximum of SCAP voltage = 6.0 volt)
     */
    public void setRFLimiterLevel(int RFLM_Level) {
        mRegister.writeParams(ADDRESS_PERIPHERAL_CONFIG, ADJUSTMENT_RFLM_LV, RFLM_Level);
    }

    /**
     * Sets a number of response flags in an RF response packet to make the response compatible
     * with some NFC chips/phone in market to guarantee minimum two bytes in a response frame.
     *
     * @param  RESP_Flag: This parameter contains a number of response flags
     *          This parameter can be one of the following values:
     *            {@link #RespBitFlag_1}: Response Flag in a response package is set to one byte.
     *            {@link #RespBitFlag_2}: Response Flag in a response package is set to two bytes
     *                  for some downlink commands to guarantee that uplink response frame contains
     *                  at least 2 bytes to make operation compatible with some NFC phones in market
     *                  that don’t accept one byte response.
     */
    public void setResponseFlags(int RESP_Flag) {
        mRegister.writeParams(ADDRESS_PERIPHERAL_CONFIG, ADJUSTMENT_RESP_2B_FLAG, RESP_Flag);
    }
}
