/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package com.sic.module.nfc.tech.chips.registers;

import com.sic.module.nfc.tech.chips.SIC43xx;
import com.sic.module.nfc.tech.chips.sic4310.Register;
import com.sic.module.nfc.tech.chips.sic431x.Flag;
import com.sic.module.nfc.tech.interfaces.IDevice;
import com.sic.module.utils.SICLog;
import com.sic.module.utils.Utils;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.2
 * @since 25/Nov/2015
 */
public abstract class IUART extends IDevice {

    private static final String TAG = IUART.class.getName();

    protected static byte ADDRESS_TRXRU_RESPONSE_TIME;
    protected static byte ADDRESS_UART_BYTE_CONFIGURATION;
    protected static byte ADDRESS_UART_DIVISOR_M;
    protected static byte ADDRESS_UART_DIVISOR_N;

    protected static int SPEED_OF_UART;
    protected static int TRxRURespTimeShiftBit;

    public static int MAX_SIZE_FIFO;

    public static int STOP_BITS;
    public static int PARITY;
    public static int TRXRU_RESPONSE_TIME;
    public static byte DIVISOR_M;
    public static byte DIVISOR_N;

    public static final int TRxRURespTime_1 = 1;
    public static final int TRxRURespTime_2 = 2;
    public static final int TRxRURespTime_4 = 4;
    public static final int TRxRURespTime_8 = 8;
    public static final int TRxRURespTime_16 = 16;
    public static final int TRxRURespTime_32 = 32;
    public static final int TRxRURespTime_64 = 64;
    public static final int TRxRURespTime_128 = 128;
    public static final int TRxRURespTime_256 = 256;
    public static final int TRxRURespTime_512 = 512;
    public static final int TRxRURespTime_1024 = 1024;
    public static final int TRxRURespTime_2048 = 2048;
    public static final int TRxRURespTime_4096 = 4096;

    public static int StopBits_1;
    public static int StopBits_2;

    public static int Parity_None;
    public static int Parity_Space;
    public static int Parity_Mask;
    public static int Parity_Even;
    public static int Parity_Odd;

    public static int Mode_Rx;
    public static int Mode_Tx;
    public static int Mode_RTS;
    public static int Mode_CTS;

    protected SIC43xx mSic43xx;

    protected IUART(SIC43xx sic43xx) {
        super(sic43xx.getRegister());
        mSic43xx = sic43xx;
    }

    /**
     * Gets the TxRU protocol by limit data size follow form the FIFO size.
     *
     * @param data data
     * @return byte array, data with TxRU command (data equal or less the FIFO size)
     */
    public static byte[] getPackage_TxRU(byte[] data) {
        return Utils.getPackageLimited((byte) 0xB1, data, MAX_SIZE_FIFO);
    }

    /**
     * Gets the TxRU protocol by chunk data follow form the FIFO size.
     *
     * @param data data
     * @return byte 2D array, each chunk data (FIFO size limit) with TxRU command
     */
    public static byte[][] getPackage_TxRU_Unlimited(byte[]... data) {
        return Utils.getPackageSplit((byte) 0xB1, data, MAX_SIZE_FIFO);
    }

    /**
     * Gets the RxRU protocol.
     *
     * @return byte array, RxRU command
     */
    public static byte[] getPackage_RxUR() {
        return new byte[]{(byte) 0xB2, (byte) 0x00};
    }


    /**
     * Gets the TRxRU protocol by limit data size follow form the FIFO size.
     *
     * @param data data
     * @return byte array, data with TxRU command (data equal or less the FIFO size)
     */
    public static byte[] getPackage_TRxRU(byte[] data) {
        return Utils.getPackageLimited((byte) 0xB3, data, MAX_SIZE_FIFO);
    }

    /**
     * Gets the TxRU protocol by chunk data follow form the FIFO size.
     *
     * @param data data
     * @return byte 2D array, each chunk data (FIFO size limit) with TxRU command
     */
    public static byte[][] getPackage_TRxRU_Unlimited(byte[]... data) {
        return Utils.getPackageSplit((byte) 0xB1, (byte) 0xB3, data, MAX_SIZE_FIFO);
    }

    /**
     * Gets the UART Mode and hardware flow control
     *
     * @return byte, the specifies pin that use in UART
     */
    public abstract byte getMode();

    /**
     * Sets the UART Mode and hardware flow control
     *
     * @param UART_Mode: specifies the desired status to be set.
     *                   This parameter can be any combination of the following values:
     *                   {@link #Mode_Tx}: Tx pin is enabled
     *                   {@link #Mode_Rx}: Rx pin is enabled
     *                   {@link #Mode_CTS}: CTS pin is enabled
     *                   {@link #Mode_RTS}: RTS pin is enabled
     */
    public abstract void setMode(int UART_Mode);

    /**
     * Initializes the UART peripheral according to the specified parameters in the SIC4310 chip.
     * Auto-config:
     * Baudrate: 115200 bps
     * WordLength: 8 bit
     * StopBit: 1 bit
     * ParityBit: none
     * HWFlowControl: none (Tx and Rx Only)
     */
    public void initUART() {
        initUART(115200, StopBits_1, Parity_None, Mode_Rx | Mode_Tx);
    }

    /**
     * Initializes the UART peripheral according to the specified parameters in the SIC4310 chip.
     * Auto-config:
     * WordLength: 8 bit
     * StopBit: 1 bit
     * ParityBit: none
     * HWFlowControl: none (Tx and Rx Only)
     *
     * @param baudRate: specifies the desired status to be set UART baud rate.
     */
    public void initUART(int baudRate) {
        initUART(baudRate, StopBits_1, Parity_None, Mode_Rx | Mode_Tx);
    }

    /**
     * Initializes the UART peripheral according to the specified parameters in the SIC4310 chip.
     *
     * @param baudRate:     specifies the desired status to be set UART baud rate.
     * @param UART_StopLen: specifies the desired status to be set.
     *                      This parameter can be one of the following values:
     *                      {@link #StopBits_1}: 1 bit
     *                      {@link #StopBits_2}: 2 bit
     * @param UART_Parity:  specifies the desired status to be set.
     *                      This parameter can be one of the following values:
     *                      {@link #Parity_None}: No parity
     *                      {@link #Parity_Space}: '0' ( Space )
     *                      {@link #Parity_Mask}: '1' ( Mark )
     *                      {@link #Parity_Even}: Even
     *                      {@link #Parity_Odd}: Odd
     * @param UART_Mode:    specifies the desired status to be set.
     *                      This parameter can be any combination of the following values:
     *                      {@link #Mode_Tx}: Tx pin is enabled
     *                      {@link #Mode_Rx}: Rx pin is enabled
     *                      {@link #Mode_CTS}: CTS pin is enabled
     *                      {@link #Mode_RTS}: RTS pin is enabled
     */
    public void initUART(int baudRate, int UART_StopLen, int UART_Parity, int UART_Mode) {
        int UARTRateDiv, m, n;
        try {

            UARTRateDiv = SPEED_OF_UART / baudRate;
            m = 1;
            n = UARTRateDiv;

            while (n > 48) {
                m <<= 1;
                n >>= 1;
            }

            byte[][] buffer = {
                    {ADDRESS_UART_BYTE_CONFIGURATION, (byte) (UART_StopLen & STOP_BITS | UART_Parity & PARITY)},
                    {ADDRESS_UART_DIVISOR_M, (byte) m},
                    {ADDRESS_UART_DIVISOR_N, (byte) n}};
            mRegister.write(buffer);
            setMode(UART_Mode);

        } catch (Exception e) {
            SICLog.e(TAG, e.getMessage());
        }
    }

    /**
     * Gets the UART baud rate
     *
     * @return int, the UART baud rate in bps
     */
    public int getBaudRate() {
        int m = mRegister.readBuffer(ADDRESS_UART_DIVISOR_M) & DIVISOR_M;
        int n = mRegister.readBuffer(ADDRESS_UART_DIVISOR_N) & DIVISOR_N;
        if (n < 2) {
            return -1;
        } else {
            if (m == 0) {
                m = 64;
            }
            return SPEED_OF_UART / (m * n);
        }
    }

    /**
     * Sets the UART baud rate
     *
     * @param baudRate: specifies the desired status to be set UART baud rate.
     */
    public void setBaudRate(int baudRate) {
        int bitRate = SPEED_OF_UART / baudRate;
        int m = 1;
        int n = bitRate;

        while (n > 48) {
            m <<= 1;
            n >>= 1;
        }

        mRegister.write(
                new byte[]{ADDRESS_UART_DIVISOR_M, (byte) m},
                new byte[]{ADDRESS_UART_DIVISOR_N, (byte) n});
    }

    /**
     * Gets the UART Stop bit
     *
     * @return byte, the return value can be one of the following values:
     * {@link #StopBits_1}: 1 bit
     * {@link #StopBits_2}: 2 bit
     */
    public byte getStopLength() {
        return (byte) (mRegister.readBuffer(ADDRESS_UART_BYTE_CONFIGURATION) & STOP_BITS);
    }

    /**
     * Sets the UART Stop bit
     *
     * @param UART_StopLen: specifies the desired status to be set.
     *                      This parameter can be one of the following values:
     *                      {@link #StopBits_1}: 1 bit
     *                      {@link #StopBits_2}: 2 bit
     */
    public void setStopLength(int UART_StopLen) {
        mRegister.writeParams(ADDRESS_UART_BYTE_CONFIGURATION, STOP_BITS, UART_StopLen);
    }

    /**
     * Gets the UART Parity
     *
     * @return byte, the return value can be one of the following values:
     * {@link #Parity_None}: No parity
     * {@link #Parity_Space}: '0' ( Space )
     * {@link #Parity_Mask}: '1' ( Mark )
     * {@link #Parity_Even}: Even
     * {@link #Parity_Odd}: Odd
     */
    public byte getParity() {
        return (byte) (mRegister.readBuffer(ADDRESS_UART_BYTE_CONFIGURATION) & PARITY);
    }

    /**
     * Sets the UART Parity.
     *
     * @param UART_Parity: specifies the desired status to be set.
     *                     This parameter can be one of the following values:
     *                     {@link #Parity_None}: No parity
     *                     {@link #Parity_Space}: '0' ( Space )
     *                     {@link #Parity_Mask}: '1' ( Mark )
     *                     {@link #Parity_Even}: Even
     *                     {@link #Parity_Odd}: Odd
     */
    public void setParity(int UART_Parity) {
        mRegister.writeParams(ADDRESS_UART_BYTE_CONFIGURATION, PARITY, UART_Parity);
    }

    /**
     * Gets a response timeout for the TRxRU command.
     *
     * @return integer, a timeout for the TRxRU command in millisecond
     */
    public int getTRxRUResponseTime() {
        int time = (mRegister.readBuffer(ADDRESS_TRXRU_RESPONSE_TIME) & TRXRU_RESPONSE_TIME) >> TRxRURespTimeShiftBit;
        return (int) Math.pow(2, time);
    }

    /**
     * Sets a response timeout for the TRxRU command ( 1 - 4096 millisecond).
     *
     * @param time: specifies the desired for set timeout in millisecond.
     * @return integer, a response timeout that can be set in millisecond.
     */
    public int setTRxRUResponseTime(int time) {
        time = ((int) (Math.ceil(Math.log(time) / Math.log(2))) & TRXRU_RESPONSE_TIME) >> TRxRURespTimeShiftBit;
        mRegister.write(ADDRESS_TRXRU_RESPONSE_TIME, time);
        time = (int) Math.pow(2, time);
        mSic43xx.setTimeout(time + 128);
        return time;
    }

    /**
     * Transmits the data through the UART peripheral of SIC's chip.
     *
     * @param data the data to transmit.
     * @return byte array, response flag from NFC tag.
     */
    public byte[] send(byte[]... data) {
        mSic43xx.clearDataStream();
        mSic43xx.addDataStream(getPackage_TxRU_Unlimited(data));
        mSic43xx.stream();
        return mSic43xx.getDataStreamAtLatestIndex();
    }

    /**
     * Returns the received data by the UART peripheral of SIC's chip.
     *
     * @return byte array, response flag and data from NFC tag.
     */
    public byte[] receive() {
        return mSic43xx.autoTransceive(getPackage_RxUR());
    }


    /**
     * Clear flags and Uplink FIFO of SIC's chip.
     */
    public void clearFIFO() {
        mSic43xx.clearDataStream();
        mSic43xx.addDataStream(Register.getPackage_ClearFlags());
        mSic43xx.addDataStream(getPackage_RxUR());
        mSic43xx.stream();
    }

    /**
     * Transmits the data by TxRU and receive always by RxUR until count up.
     * in the Uplink FIFO of SIC's chip. It can be set the response time directly.
     *
     * @param ncommand specifies the desired for set number of RxUR command.
     * @param data     the data to transmit.
     * @return byte array, response flag and data from NFC tag.
     */
    public byte[] TxRUTranceiver(int ncommand, byte[]... data) {
        byte[] rxBuffer;
        mSic43xx.clearDataStream();
        mSic43xx.addDataStream(getPackage_TxRU_Unlimited(data));
        mSic43xx.stream();
        boolean protect = mSic43xx.isProtection();
        mSic43xx.setProtection(false);
        do {
            rxBuffer = receive();
            if (rxBuffer == null || rxBuffer.length < 1) {
                mSic43xx.setProtection(true);
                return null;
            }
            byte flag = (Flag.B_NAK | Flag.B_NAK_UL_FIFO_EMP);
            if ((rxBuffer[0] & flag) != flag) {
                break;
            }
        } while (ncommand-- > 0);
        mSic43xx.setProtection(protect);
        return rxBuffer;
    }

    /**
     * Transmits the data, waiting for response time (TRxRU Response Time) and receiving the data
     * in the Uplink FIFO of SIC's chip.
     *
     * @param data the data to transmit.
     * @return byte array, response flag and data from NFC tag.
     */
    public byte[] TRxRUTranceiver(byte[]... data) {
        mSic43xx.clearDataStream();
        mSic43xx.addDataStream(getPackage_TRxRU_Unlimited(data));
        mSic43xx.stream();
        return mSic43xx.getDataStreamAtLatestIndex();
    }

    /**
     * Transmits the data, waiting for response time (TRxRU Response Time) and receiving the data
     * in the Uplink FIFO of SIC's chip. It can be set the response time directly.
     *
     * @param time specifies the desired for set timeout in millisecond.
     * @param data the data to transmit.
     * @return byte array, response flag and data from NFC tag.
     */
    public byte[] TRxRUTranceiver(int time, byte[]... data) {
        time = ((int) (Math.ceil(Math.log(time) / Math.log(2))) & TRXRU_RESPONSE_TIME) >> TRxRURespTimeShiftBit;
        mSic43xx.clearDataStream();
        mSic43xx.addDataStream(Register.getPackage_Write(ADDRESS_TRXRU_RESPONSE_TIME, time));
        mSic43xx.addDataStream(getPackage_TRxRU_Unlimited(data));
        time = (int) Math.pow(2, time);
        mSic43xx.setTimeout(time + 128);
        mSic43xx.stream();
        return mSic43xx.getDataStreamAtLatestIndex();
    }
}
