/*
 * Copyright (c) 2016 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.managements;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Environment;

import com.sic.module.utils.AppContext;
import com.sic.module.utils.SICLog;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.1
 * @since 27/Mar/2015
 */
public class FileManager {

    public static final String DEFAULT_FOLDER_WORKSPACE = Environment.getExternalStorageDirectory().getAbsolutePath() + "/SIC/";
    public static final String DEFAULT_FOLDER_LOG = DEFAULT_FOLDER_WORKSPACE + "log/";
    public static final String DEFAULT_FOLDER_TEMP = DEFAULT_FOLDER_WORKSPACE + "tmp/";
    private static final String TAG = FileManager.class.getSimpleName();
    private static FileManager instance;
    private final Context mContext;

    private FileManager(Context context) {
        mContext = context;
    }

    public static FileManager getInstance() {
        if (instance == null)
            instance = new FileManager(AppContext.getInstance().getContext());
        return instance;
    }

    public static FileManager getInstance(Context context) {
        if (instance == null)
            instance = new FileManager(context);
        return instance;
    }

    /**
     * Check file is exist and not empty, is not?
     *
     * @param pathFile is a path of file
     * @return true, if it is exist and not empty.
     */
    public boolean isFileExistAndNotEmpty(String pathFile) {
        if (pathFile == null) {
            return false;
        }

        File f = new File(pathFile);
        long lengthInBytes = f.length();
        SICLog.i(TAG, pathFile + " length in bytes: " + lengthInBytes);
        return lengthInBytes > 100;
    }

    // check file and folder

    /**
     * Check folder is exist, is not?
     *
     * @param pathFolder is a path of folder
     * @return true, if it is exist.
     */
    public boolean isFolderExist(String pathFolder) {
        File f = new File(pathFolder);
        return f.exists() && f.isDirectory();
    }

    /**
     * Create folder.
     *
     * @param pathFolder is a path of folder
     * @return true, if can create.
     */
    public boolean createFolder(String pathFolder) {
        File f = new File(pathFolder);
        return f.mkdirs();
    }

    // create

    // write
    public boolean writeFromResourceId(String pathfile, int resid) {
        if (isFileExistAndNotEmpty(pathfile)) {
            return false;
        }

        int length;
        byte buffer[] = new byte[1024];
        File file = new File(pathfile);
        InputStream inputStream = mContext.getResources().openRawResource(resid);
        OutputStream outputStream;

        try {
            outputStream = new FileOutputStream(file);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            return false;
        }

        try {
            while ((length = inputStream.read(buffer)) > 0) {
                outputStream.write(buffer, 0, length);
            }
            outputStream.close();
            inputStream.close();
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        }

        return true;
    }

    /**
     * /**
     * Delete file.
     *
     * @param pathFile is a path of file
     * @return true, if can delete.
     */
    public boolean deleteFile(String pathFile) {
        File f = new File(pathFile);
        boolean isDeleted = f.delete();
        SICLog.i(TAG, "deleteing: " + pathFile + " isdeleted: " + isDeleted);
        return isDeleted;
    }

    /**
     * Get file name from file path.
     *
     * @param filePath is a path of file.
     * @return file name.
     */
    public String getFileNameFromFilePath(String filePath) {
        int index = filePath.lastIndexOf("/");
        return filePath.substring(index + 1, filePath.length());
    }

    // get

    public Bitmap getBitmapFromImageFile(File file, int width, int height) {
        // The new size we want to scale to

        try {
            // Decode image size
            BitmapFactory.Options o = new BitmapFactory.Options();
            o.inJustDecodeBounds = true;
            BitmapFactory.decodeStream(new FileInputStream(file), null, o);

            // Find the correct scale value. It should be the power of 2.
            int scale = 1;
            while (o.outWidth / scale / 2 >= width && o.outHeight / scale / 2 >= height) {
                scale *= 2;
            }

            // Decode with inSampleSize
            BitmapFactory.Options o2 = new BitmapFactory.Options();
            o2.inSampleSize = scale;
            return BitmapFactory.decodeStream(new FileInputStream(file), null, o2);
        } catch (FileNotFoundException ignored) {
        }
        return null;
    }

    /**
     * Get file type from file
     *
     * @param filePath is a path of file
     * @return file type.
     */
    public int getFileTypeFromFile(String filePath) {
        int type = filePath.indexOf('.');
        if (type < 0) {
            return FileType.no_ext.getCode();
        }
        String ext = filePath.substring(type + 1).toLowerCase();
        // Image file
        for (FileType fileType : FileType.values()) {
            if (ext.contains(fileType.getType())) {
                return fileType.getCode();
            }
        }

        return FileType.Unknown.getCode();

    }

    public enum FileType {
        // Image
        Image("IMAGE", 0x10), raw("raw", 0x10), jpg("jpg", 0x11), jpeg("jpeg", 0x11), png("png", 0x12), bmp("bmp", 0x13), gif("gif", 0x14), tiff("tiff", 0x15), ico("ico", 0x16),

        // Text
        Text("TEXT", 0x20), no_ext("", 0x20), txt("txt", 0x21), doc("doc", 0x22), docx("docx", 0x23),

        // unoffical
        hex("hex", 0x2F), s19("s19", 0x2E),

        // Audio
        Audio("AUDIO", 0x30), wav("wav", 0x30), mp3("mp3", 0x31), wma("wma", 0x32), ogg("ogg", 0x33), ac3("ac3", 0x34), aac("aac", 0x35),

        // MIME Type
        ndef("mndef", 0x44),

        // nothing
        Unknown("UNKNOWN", 0x00);

        private final String type;
        private final int code;

        FileType(String type, int code) {
            this.type = type;
            this.code = code;
        }

        /**
         * Get command ID code of file extension.
         *
         * @return command ID
         */
        public int getCode() {
            return code;
        }

        /**
         * Get extension of file.
         *
         * @return file type
         */
        public String getType() {
            return type;
        }
    }
}
