/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.sic4310;

import com.sic.module.nfc.tech.chips.SIC4310;
import com.sic.module.nfc.tech.chips.sic43xx.DefRegister;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.2
 * @since 16/Oct/2015
 */
public class Register extends DefRegister {

    public static final int REGISTER_PAGE = 0x10;

    public static final byte UART_STATUS = 0x00;
    public static final byte POWER_STATUS = 0x01;
    public static final byte TRXRU_RESPONSE_TIME = 0x03;
    public static final byte UART_BYTE_CONFIGURATION = 0x04;
    public static final byte UART_DIVISOR_M = 0x05;
    public static final byte UART_DIVISOR_N = 0x06;
    public static final byte OSC_TUNING = 0x07;
    public static final byte GPIO_DIRECTION = 0x08;
    public static final byte GPIO_MODE = 0x09;
    public static final byte GPIO_OUT = 0x0A;
    public static final byte GPIO_IN = 0x0B;
    public static final byte GPIO_PU = 0x0C;
    public static final byte PERIPHERAL_CONFIG = 0x0D;
    public static final byte PERIPHERAL_ADJUSTMENT = 0x0E;

    private static final String TAG = Register.class.getSimpleName();
    private static Register instance;

    private Register(SIC4310 sic4310) {
        super(sic4310);
    }

    public static Register getInstance(SIC4310 sic4310) {
        if (instance == null)
            instance = new Register(sic4310);
        return instance;
    }

    /**
     * Gets a register name at the specifics address.
     *
     * @param address register address
     * @return string, a register name
     */
    public static String getName(int address) {
        switch (address) {
            case UART_STATUS:
                return "UART Status";

            case POWER_STATUS:
                return "Power Status";

            case TRXRU_RESPONSE_TIME:
                return "TRxRU Response Time";

            case UART_BYTE_CONFIGURATION:
                return "UART Byte Configuration";

            case UART_DIVISOR_M:
                return "UART Divisor M";

            case UART_DIVISOR_N:
                return "UART Divisor N";

            case OSC_TUNING:
                return "OSC Tuning";

            case GPIO_DIRECTION:
                return "GPIO Direction";

            case GPIO_MODE:
                return "GPIO Mode";

            case GPIO_OUT:
                return "GPIO Output";

            case GPIO_IN:
                return "GPIO Input";

            case GPIO_PU:
                return "GPIO Pull-up";

            case PERIPHERAL_CONFIG:
                return "Peripheral Config";

            case PERIPHERAL_ADJUSTMENT:
                return "Peripheral Adjustment";

            default:
                return "RFU";
        }
    }

    /**
     * Gets a register's type at the specifics address.
     *
     * @param address register address
     * @return type of register, it can be one of the following values:
     *            {@link RegisterType#NULL}: Null register
     *            {@link RegisterType#RFU}: Reserved register
     *            {@link RegisterType#READ_ONLY_STATUS}: Readable only register (Status register)
     *            {@link RegisterType#READ_WRITE_CONFIG}: Readable and writable register (Configuration register)
     *            {@link RegisterType#READ_WRITE}: Readable and writable register
     *            {@link RegisterType#READ_ONLY}: Readable only register
     */
    public static RegisterType getRegisterType(int address) {
        switch (address) {
            case UART_STATUS:
            case POWER_STATUS:
            case GPIO_IN:
                return RegisterType.READ_ONLY_STATUS;

            case TRXRU_RESPONSE_TIME:
            case UART_BYTE_CONFIGURATION:
            case UART_DIVISOR_M:
            case UART_DIVISOR_N:
            case GPIO_DIRECTION:
            case GPIO_MODE:
            case GPIO_OUT:
            case GPIO_PU:
            case PERIPHERAL_CONFIG:
            case PERIPHERAL_ADJUSTMENT:
                return RegisterType.READ_WRITE_CONFIG;

            case OSC_TUNING:
                return RegisterType.READ_WRITE;


            default:
                return RegisterType.RFU;
        }
    }

    @Override
    protected int getRegisterPage() {
        return REGISTER_PAGE;
    }
}
