/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.registers;


import com.sic.module.nfc.tech.chips.SIC43xx;
import com.sic.module.nfc.tech.interfaces.IDevice;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.0
 * @since 25/Nov/2015
 */
public abstract class IStatus extends IDevice {

    private static final String TAG = IStatus.class.getName();
    protected static int ADDRESS_POWER_STATUS;

    public static int SCAP_RDY;
    public static int XVDD_RDY;
    public static int RSPW_RDY;
    public static int LDO_ON;

    private static final int TIMEOUT_OF_STATUS = 500;
    private long latestTime = 0;

    protected IStatus(SIC43xx sic43xx) {
        super(sic43xx.getRegister());
    }

    protected Byte getStatus(int address) {
        Byte status;
        long time = System.currentTimeMillis();
        if (time - latestTime < TIMEOUT_OF_STATUS) {
            status = mRegister.readBuffer(address);
        } else {
            status = mRegister.read(address);
        }
        latestTime = time;
        return status;
    }

    /**
     * Gets the Power status on register.
     *
     * @return  byte Power status data value.
     */
    public Byte getPowerStatus() {
        return getStatus(ADDRESS_POWER_STATUS);
    }

    /**
     * Checks the power of external super capacitor.
     *
     * @return true, if the voltage on pin SCAP is higher than 4.5V.
     */
    public boolean isPowerSuperCapReady() {
        Byte flag = getPowerStatus();
        return flag != null && ((flag & SCAP_RDY) == SCAP_RDY);
    }

    /**
     * Checks the external power.
     *
     * @return true, if the voltage on pin XVDD is higher than XVDD drop level.
     */
    public boolean isPowerXVDDReady() {
        Byte flag = getPowerStatus();
        return flag != null && ((flag & XVDD_RDY) == XVDD_RDY);
    }

    /**
     * Checks the power of NFC device.
     *
     * @return true, if the reserve power from RF is higher than a defined supplying level.
     */
    public boolean isReservePowerReady() {
        Byte flag = getPowerStatus();
        return flag != null && ((flag & RSPW_RDY) == RSPW_RDY);
    }

    /**
     * Checks the LDO regulator.
     *
     * @return true, if the LDO regulator is successfully turned on.
     */
    public boolean isLDORegulatorOn() {
        Byte flag = getPowerStatus();
        return flag != null && ((flag & LDO_ON) == LDO_ON);
    }
}
